{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- @GAction@ represents a single named action.
-- 
-- The main interface to an action is that it can be activated with
-- 'GI.Gio.Interfaces.Action.actionActivate'. This results in the \'activate\' signal being
-- emitted. An activation has a @GVariant@ parameter (which may be
-- @NULL@). The correct type for the parameter is determined by a static
-- parameter type (which is given at construction time).
-- 
-- An action may optionally have a state, in which case the state may be
-- set with 'GI.Gio.Interfaces.Action.actionChangeState'. This call takes a [type/@gLib@/.Variant]. The
-- correct type for the state is determined by a static state type
-- (which is given at construction time).
-- 
-- The state may have a hint associated with it, specifying its valid
-- range.
-- 
-- @GAction@ is merely the interface to the concept of an action, as
-- described above.  Various implementations of actions exist, including
-- t'GI.Gio.Objects.SimpleAction.SimpleAction'.
-- 
-- In all cases, the implementing class is responsible for storing the
-- name of the action, the parameter type, the enabled state, the optional
-- state type and the state and emitting the appropriate signals when these
-- change. The implementor is responsible for filtering calls to
-- 'GI.Gio.Interfaces.Action.actionActivate' and 'GI.Gio.Interfaces.Action.actionChangeState'
-- for type safety and for the state being enabled.
-- 
-- Probably the only useful thing to do with a @GAction@ is to put it
-- inside of a t'GI.Gio.Objects.SimpleActionGroup.SimpleActionGroup'.

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gio.Interfaces.Action
    ( 

-- * Exported types
    Action(..)                              ,
    IsAction                                ,
    toAction                                ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [activate]("GI.Gio.Interfaces.Action#g:method:activate"), [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [changeState]("GI.Gio.Interfaces.Action#g:method:changeState"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getData]("GI.GObject.Objects.Object#g:method:getData"), [getEnabled]("GI.Gio.Interfaces.Action#g:method:getEnabled"), [getName]("GI.Gio.Interfaces.Action#g:method:getName"), [getParameterType]("GI.Gio.Interfaces.Action#g:method:getParameterType"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata"), [getState]("GI.Gio.Interfaces.Action#g:method:getState"), [getStateHint]("GI.Gio.Interfaces.Action#g:method:getStateHint"), [getStateType]("GI.Gio.Interfaces.Action#g:method:getStateType").
-- 
-- ==== Setters
-- [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty").

#if defined(ENABLE_OVERLOADING)
    ResolveActionMethod                     ,
#endif

-- ** activate #method:activate#

#if defined(ENABLE_OVERLOADING)
    ActionActivateMethodInfo                ,
#endif
    actionActivate                          ,


-- ** changeState #method:changeState#

#if defined(ENABLE_OVERLOADING)
    ActionChangeStateMethodInfo             ,
#endif
    actionChangeState                       ,


-- ** getEnabled #method:getEnabled#

#if defined(ENABLE_OVERLOADING)
    ActionGetEnabledMethodInfo              ,
#endif
    actionGetEnabled                        ,


-- ** getName #method:getName#

#if defined(ENABLE_OVERLOADING)
    ActionGetNameMethodInfo                 ,
#endif
    actionGetName                           ,


-- ** getParameterType #method:getParameterType#

#if defined(ENABLE_OVERLOADING)
    ActionGetParameterTypeMethodInfo        ,
#endif
    actionGetParameterType                  ,


-- ** getState #method:getState#

#if defined(ENABLE_OVERLOADING)
    ActionGetStateMethodInfo                ,
#endif
    actionGetState                          ,


-- ** getStateHint #method:getStateHint#

#if defined(ENABLE_OVERLOADING)
    ActionGetStateHintMethodInfo            ,
#endif
    actionGetStateHint                      ,


-- ** getStateType #method:getStateType#

#if defined(ENABLE_OVERLOADING)
    ActionGetStateTypeMethodInfo            ,
#endif
    actionGetStateType                      ,


-- ** nameIsValid #method:nameIsValid#

    actionNameIsValid                       ,


-- ** parseDetailedName #method:parseDetailedName#

    actionParseDetailedName                 ,


-- ** printDetailedName #method:printDetailedName#

    actionPrintDetailedName                 ,




 -- * Properties


-- ** enabled #attr:enabled#
-- | If /@action@/ is currently enabled.
-- 
-- If the action is disabled then calls to 'GI.Gio.Interfaces.Action.actionActivate' and
-- 'GI.Gio.Interfaces.Action.actionChangeState' have no effect.
-- 
-- /Since: 2.28/

#if defined(ENABLE_OVERLOADING)
    ActionEnabledPropertyInfo               ,
#endif
#if defined(ENABLE_OVERLOADING)
    actionEnabled                           ,
#endif
    getActionEnabled                        ,


-- ** name #attr:name#
-- | The name of the action.  This is mostly meaningful for identifying
-- the action once it has been added to a [type/@gio@/.ActionGroup]. It is immutable.
-- 
-- /Since: 2.28/

#if defined(ENABLE_OVERLOADING)
    ActionNamePropertyInfo                  ,
#endif
#if defined(ENABLE_OVERLOADING)
    actionName                              ,
#endif
    getActionName                           ,


-- ** parameterType #attr:parameterType#
-- | The type of the parameter that must be given when activating the
-- action. This is immutable, and may be @NULL@ if no parameter is needed when
-- activating the action.
-- 
-- /Since: 2.28/

#if defined(ENABLE_OVERLOADING)
    ActionParameterTypePropertyInfo         ,
#endif
#if defined(ENABLE_OVERLOADING)
    actionParameterType                     ,
#endif
    getActionParameterType                  ,


-- ** state #attr:state#
-- | The state of the action, or @NULL@ if the action is stateless.
-- 
-- /Since: 2.28/

#if defined(ENABLE_OVERLOADING)
    ActionStatePropertyInfo                 ,
#endif
#if defined(ENABLE_OVERLOADING)
    actionState                             ,
#endif
    getActionState                          ,


-- ** stateType #attr:stateType#
-- | The [type/@gLib@/.VariantType] of the state that the action has, or @NULL@ if the
-- action is stateless. This is immutable.
-- 
-- /Since: 2.28/

#if defined(ENABLE_OVERLOADING)
    ActionStateTypePropertyInfo             ,
#endif
#if defined(ENABLE_OVERLOADING)
    actionStateType                         ,
#endif
    getActionStateType                      ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.GLib.Structs.VariantType as GLib.VariantType
import qualified GI.GObject.Objects.Object as GObject.Object

#else
import qualified GI.GLib.Structs.VariantType as GLib.VariantType
import qualified GI.GObject.Objects.Object as GObject.Object

#endif

-- interface Action 
-- | Memory-managed wrapper type.
newtype Action = Action (SP.ManagedPtr Action)
    deriving (Eq)

instance SP.ManagedPtrNewtype Action where
    toManagedPtr (Action p) = p

foreign import ccall "g_action_get_type"
    c_g_action_get_type :: IO B.Types.GType

instance B.Types.TypedObject Action where
    glibType = c_g_action_get_type

instance B.Types.GObject Action

-- | Type class for types which can be safely cast to t'Action', for instance with `toAction`.
class (SP.GObject o, O.IsDescendantOf Action o) => IsAction o
instance (SP.GObject o, O.IsDescendantOf Action o) => IsAction o

instance O.HasParentTypes Action
type instance O.ParentTypes Action = '[GObject.Object.Object]

-- | Cast to t'Action', for types for which this is known to be safe. For general casts, use 'Data.GI.Base.ManagedPtr.castTo'.
toAction :: (MIO.MonadIO m, IsAction o) => o -> m Action
toAction = MIO.liftIO . B.ManagedPtr.unsafeCastTo Action

-- | Convert t'Action' to and from t'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe Action) where
    gvalueGType_ = c_g_action_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr Action)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr Action)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject Action ptr
        else return P.Nothing
        
    

-- VVV Prop "enabled"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable]
   -- Nullable: (Just False,Nothing)

-- | Get the value of the “@enabled@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' action #enabled
-- @
getActionEnabled :: (MonadIO m, IsAction o) => o -> m Bool
getActionEnabled obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "enabled"

#if defined(ENABLE_OVERLOADING)
data ActionEnabledPropertyInfo
instance AttrInfo ActionEnabledPropertyInfo where
    type AttrAllowedOps ActionEnabledPropertyInfo = '[ 'AttrGet]
    type AttrBaseTypeConstraint ActionEnabledPropertyInfo = IsAction
    type AttrSetTypeConstraint ActionEnabledPropertyInfo = (~) ()
    type AttrTransferTypeConstraint ActionEnabledPropertyInfo = (~) ()
    type AttrTransferType ActionEnabledPropertyInfo = ()
    type AttrGetType ActionEnabledPropertyInfo = Bool
    type AttrLabel ActionEnabledPropertyInfo = "enabled"
    type AttrOrigin ActionEnabledPropertyInfo = Action
    attrGet = getActionEnabled
    attrSet = undefined
    attrPut = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Action.enabled"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Interfaces-Action.html#g:attr:enabled"
        })
#endif

-- VVV Prop "name"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable]
   -- Nullable: (Just False,Nothing)

-- | Get the value of the “@name@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' action #name
-- @
getActionName :: (MonadIO m, IsAction o) => o -> m T.Text
getActionName obj = MIO.liftIO $ checkUnexpectedNothing "getActionName" $ B.Properties.getObjectPropertyString obj "name"

#if defined(ENABLE_OVERLOADING)
data ActionNamePropertyInfo
instance AttrInfo ActionNamePropertyInfo where
    type AttrAllowedOps ActionNamePropertyInfo = '[ 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint ActionNamePropertyInfo = IsAction
    type AttrSetTypeConstraint ActionNamePropertyInfo = (~) ()
    type AttrTransferTypeConstraint ActionNamePropertyInfo = (~) ()
    type AttrTransferType ActionNamePropertyInfo = ()
    type AttrGetType ActionNamePropertyInfo = T.Text
    type AttrLabel ActionNamePropertyInfo = "name"
    type AttrOrigin ActionNamePropertyInfo = Action
    attrGet = getActionName
    attrSet = undefined
    attrPut = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Action.name"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Interfaces-Action.html#g:attr:name"
        })
#endif

-- VVV Prop "parameter-type"
   -- Type: TInterface (Name {namespace = "GLib", name = "VariantType"})
   -- Flags: [PropertyReadable]
   -- Nullable: (Just True,Nothing)

-- | Get the value of the “@parameter-type@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' action #parameterType
-- @
getActionParameterType :: (MonadIO m, IsAction o) => o -> m (Maybe GLib.VariantType.VariantType)
getActionParameterType obj = MIO.liftIO $ B.Properties.getObjectPropertyBoxed obj "parameter-type" GLib.VariantType.VariantType

#if defined(ENABLE_OVERLOADING)
data ActionParameterTypePropertyInfo
instance AttrInfo ActionParameterTypePropertyInfo where
    type AttrAllowedOps ActionParameterTypePropertyInfo = '[ 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint ActionParameterTypePropertyInfo = IsAction
    type AttrSetTypeConstraint ActionParameterTypePropertyInfo = (~) ()
    type AttrTransferTypeConstraint ActionParameterTypePropertyInfo = (~) ()
    type AttrTransferType ActionParameterTypePropertyInfo = ()
    type AttrGetType ActionParameterTypePropertyInfo = (Maybe GLib.VariantType.VariantType)
    type AttrLabel ActionParameterTypePropertyInfo = "parameter-type"
    type AttrOrigin ActionParameterTypePropertyInfo = Action
    attrGet = getActionParameterType
    attrSet = undefined
    attrPut = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Action.parameterType"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Interfaces-Action.html#g:attr:parameterType"
        })
#endif

-- VVV Prop "state"
   -- Type: TVariant
   -- Flags: [PropertyReadable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@state@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' action #state
-- @
getActionState :: (MonadIO m, IsAction o) => o -> m (Maybe GVariant)
getActionState obj = MIO.liftIO $ B.Properties.getObjectPropertyVariant obj "state"

#if defined(ENABLE_OVERLOADING)
data ActionStatePropertyInfo
instance AttrInfo ActionStatePropertyInfo where
    type AttrAllowedOps ActionStatePropertyInfo = '[ 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint ActionStatePropertyInfo = IsAction
    type AttrSetTypeConstraint ActionStatePropertyInfo = (~) ()
    type AttrTransferTypeConstraint ActionStatePropertyInfo = (~) ()
    type AttrTransferType ActionStatePropertyInfo = ()
    type AttrGetType ActionStatePropertyInfo = (Maybe GVariant)
    type AttrLabel ActionStatePropertyInfo = "state"
    type AttrOrigin ActionStatePropertyInfo = Action
    attrGet = getActionState
    attrSet = undefined
    attrPut = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Action.state"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Interfaces-Action.html#g:attr:state"
        })
#endif

-- VVV Prop "state-type"
   -- Type: TInterface (Name {namespace = "GLib", name = "VariantType"})
   -- Flags: [PropertyReadable]
   -- Nullable: (Just True,Nothing)

-- | Get the value of the “@state-type@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' action #stateType
-- @
getActionStateType :: (MonadIO m, IsAction o) => o -> m (Maybe GLib.VariantType.VariantType)
getActionStateType obj = MIO.liftIO $ B.Properties.getObjectPropertyBoxed obj "state-type" GLib.VariantType.VariantType

#if defined(ENABLE_OVERLOADING)
data ActionStateTypePropertyInfo
instance AttrInfo ActionStateTypePropertyInfo where
    type AttrAllowedOps ActionStateTypePropertyInfo = '[ 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint ActionStateTypePropertyInfo = IsAction
    type AttrSetTypeConstraint ActionStateTypePropertyInfo = (~) ()
    type AttrTransferTypeConstraint ActionStateTypePropertyInfo = (~) ()
    type AttrTransferType ActionStateTypePropertyInfo = ()
    type AttrGetType ActionStateTypePropertyInfo = (Maybe GLib.VariantType.VariantType)
    type AttrLabel ActionStateTypePropertyInfo = "state-type"
    type AttrOrigin ActionStateTypePropertyInfo = Action
    attrGet = getActionStateType
    attrSet = undefined
    attrPut = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Action.stateType"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Interfaces-Action.html#g:attr:stateType"
        })
#endif

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList Action
type instance O.AttributeList Action = ActionAttributeList
type ActionAttributeList = ('[ '("enabled", ActionEnabledPropertyInfo), '("name", ActionNamePropertyInfo), '("parameterType", ActionParameterTypePropertyInfo), '("state", ActionStatePropertyInfo), '("stateType", ActionStateTypePropertyInfo)] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
actionEnabled :: AttrLabelProxy "enabled"
actionEnabled = AttrLabelProxy

actionName :: AttrLabelProxy "name"
actionName = AttrLabelProxy

actionParameterType :: AttrLabelProxy "parameterType"
actionParameterType = AttrLabelProxy

actionState :: AttrLabelProxy "state"
actionState = AttrLabelProxy

actionStateType :: AttrLabelProxy "stateType"
actionStateType = AttrLabelProxy

#endif

#if defined(ENABLE_OVERLOADING)
type family ResolveActionMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveActionMethod "activate" o = ActionActivateMethodInfo
    ResolveActionMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolveActionMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolveActionMethod "changeState" o = ActionChangeStateMethodInfo
    ResolveActionMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolveActionMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolveActionMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolveActionMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolveActionMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolveActionMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolveActionMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolveActionMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolveActionMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolveActionMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolveActionMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolveActionMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolveActionMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolveActionMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolveActionMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolveActionMethod "getEnabled" o = ActionGetEnabledMethodInfo
    ResolveActionMethod "getName" o = ActionGetNameMethodInfo
    ResolveActionMethod "getParameterType" o = ActionGetParameterTypeMethodInfo
    ResolveActionMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolveActionMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolveActionMethod "getState" o = ActionGetStateMethodInfo
    ResolveActionMethod "getStateHint" o = ActionGetStateHintMethodInfo
    ResolveActionMethod "getStateType" o = ActionGetStateTypeMethodInfo
    ResolveActionMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolveActionMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolveActionMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolveActionMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveActionMethod t Action, O.OverloadedMethod info Action p) => OL.IsLabel t (Action -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveActionMethod t Action, O.OverloadedMethod info Action p, R.HasField t Action p) => R.HasField t Action p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveActionMethod t Action, O.OverloadedMethodInfo info Action) => OL.IsLabel t (O.MethodProxy info Action) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

-- method Action::activate
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "action"
--           , argType = TInterface Name { namespace = "Gio" , name = "Action" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a [type@Gio.Action]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "parameter"
--           , argType = TVariant
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the parameter to the activation"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_action_activate" g_action_activate :: 
    Ptr Action ->                           -- action : TInterface (Name {namespace = "Gio", name = "Action"})
    Ptr GVariant ->                         -- parameter : TVariant
    IO ()

-- | Activates the action.
-- 
-- /@parameter@/ must be the correct type of parameter for the action (ie:
-- the parameter type given at construction time).  If the parameter
-- type was @NULL@ then /@parameter@/ must also be @NULL@.
-- 
-- If the /@parameter@/ [type/@gLib@/.Variant] is floating, it is consumed.
-- 
-- /Since: 2.28/
actionActivate ::
    (B.CallStack.HasCallStack, MonadIO m, IsAction a) =>
    a
    -- ^ /@action@/: a [type/@gio@/.Action]
    -> Maybe (GVariant)
    -- ^ /@parameter@/: the parameter to the activation
    -> m ()
actionActivate action parameter = liftIO $ do
    action' <- unsafeManagedPtrCastPtr action
    maybeParameter <- case parameter of
        Nothing -> return FP.nullPtr
        Just jParameter -> do
            jParameter' <- unsafeManagedPtrGetPtr jParameter
            return jParameter'
    g_action_activate action' maybeParameter
    touchManagedPtr action
    whenJust parameter touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data ActionActivateMethodInfo
instance (signature ~ (Maybe (GVariant) -> m ()), MonadIO m, IsAction a) => O.OverloadedMethod ActionActivateMethodInfo a signature where
    overloadedMethod = actionActivate

instance O.OverloadedMethodInfo ActionActivateMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Action.actionActivate",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Interfaces-Action.html#v:actionActivate"
        })


#endif

-- method Action::change_state
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "action"
--           , argType = TInterface Name { namespace = "Gio" , name = "Action" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a [type@Gio.Action]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "value"
--           , argType = TVariant
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the new state" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_action_change_state" g_action_change_state :: 
    Ptr Action ->                           -- action : TInterface (Name {namespace = "Gio", name = "Action"})
    Ptr GVariant ->                         -- value : TVariant
    IO ()

-- | Request for the state of /@action@/ to be changed to /@value@/.
-- 
-- The action must be stateful and /@value@/ must be of the correct type.
-- See 'GI.Gio.Interfaces.Action.actionGetStateType'.
-- 
-- This call merely requests a change.  The action may refuse to change
-- its state or may change its state to something other than /@value@/.
-- See 'GI.Gio.Interfaces.Action.actionGetStateHint'.
-- 
-- If the /@value@/ [type/@gLib@/.Variant] is floating, it is consumed.
-- 
-- /Since: 2.30/
actionChangeState ::
    (B.CallStack.HasCallStack, MonadIO m, IsAction a) =>
    a
    -- ^ /@action@/: a [type/@gio@/.Action]
    -> GVariant
    -- ^ /@value@/: the new state
    -> m ()
actionChangeState action value = liftIO $ do
    action' <- unsafeManagedPtrCastPtr action
    value' <- unsafeManagedPtrGetPtr value
    g_action_change_state action' value'
    touchManagedPtr action
    touchManagedPtr value
    return ()

#if defined(ENABLE_OVERLOADING)
data ActionChangeStateMethodInfo
instance (signature ~ (GVariant -> m ()), MonadIO m, IsAction a) => O.OverloadedMethod ActionChangeStateMethodInfo a signature where
    overloadedMethod = actionChangeState

instance O.OverloadedMethodInfo ActionChangeStateMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Action.actionChangeState",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Interfaces-Action.html#v:actionChangeState"
        })


#endif

-- method Action::get_enabled
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "action"
--           , argType = TInterface Name { namespace = "Gio" , name = "Action" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a [type@Gio.Action]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_action_get_enabled" g_action_get_enabled :: 
    Ptr Action ->                           -- action : TInterface (Name {namespace = "Gio", name = "Action"})
    IO CInt

-- | Checks if /@action@/ is currently enabled.
-- 
-- An action must be enabled in order to be activated or in order to
-- have its state changed from outside callers.
-- 
-- /Since: 2.28/
actionGetEnabled ::
    (B.CallStack.HasCallStack, MonadIO m, IsAction a) =>
    a
    -- ^ /@action@/: a [type/@gio@/.Action]
    -> m Bool
    -- ^ __Returns:__ whether the action is enabled
actionGetEnabled action = liftIO $ do
    action' <- unsafeManagedPtrCastPtr action
    result <- g_action_get_enabled action'
    let result' = (/= 0) result
    touchManagedPtr action
    return result'

#if defined(ENABLE_OVERLOADING)
data ActionGetEnabledMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsAction a) => O.OverloadedMethod ActionGetEnabledMethodInfo a signature where
    overloadedMethod = actionGetEnabled

instance O.OverloadedMethodInfo ActionGetEnabledMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Action.actionGetEnabled",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Interfaces-Action.html#v:actionGetEnabled"
        })


#endif

-- method Action::get_name
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "action"
--           , argType = TInterface Name { namespace = "Gio" , name = "Action" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a [type@Gio.Action]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_action_get_name" g_action_get_name :: 
    Ptr Action ->                           -- action : TInterface (Name {namespace = "Gio", name = "Action"})
    IO CString

-- | Queries the name of /@action@/.
-- 
-- /Since: 2.28/
actionGetName ::
    (B.CallStack.HasCallStack, MonadIO m, IsAction a) =>
    a
    -- ^ /@action@/: a [type/@gio@/.Action]
    -> m T.Text
    -- ^ __Returns:__ the name of the action
actionGetName action = liftIO $ do
    action' <- unsafeManagedPtrCastPtr action
    result <- g_action_get_name action'
    checkUnexpectedReturnNULL "actionGetName" result
    result' <- cstringToText result
    touchManagedPtr action
    return result'

#if defined(ENABLE_OVERLOADING)
data ActionGetNameMethodInfo
instance (signature ~ (m T.Text), MonadIO m, IsAction a) => O.OverloadedMethod ActionGetNameMethodInfo a signature where
    overloadedMethod = actionGetName

instance O.OverloadedMethodInfo ActionGetNameMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Action.actionGetName",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Interfaces-Action.html#v:actionGetName"
        })


#endif

-- method Action::get_parameter_type
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "action"
--           , argType = TInterface Name { namespace = "Gio" , name = "Action" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a [type@Gio.Action]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "GLib" , name = "VariantType" })
-- throws : False
-- Skip return : False

foreign import ccall "g_action_get_parameter_type" g_action_get_parameter_type :: 
    Ptr Action ->                           -- action : TInterface (Name {namespace = "Gio", name = "Action"})
    IO (Ptr GLib.VariantType.VariantType)

-- | Queries the type of the parameter that must be given when activating
-- /@action@/.
-- 
-- When activating the action using 'GI.Gio.Interfaces.Action.actionActivate', the
-- [type/@gLib@/.Variant] given to that function must be of the type returned by
-- this function.
-- 
-- In the case that this function returns @NULL@, you must not give any
-- [type/@gLib@/.Variant], but @NULL@ instead.
-- 
-- /Since: 2.28/
actionGetParameterType ::
    (B.CallStack.HasCallStack, MonadIO m, IsAction a) =>
    a
    -- ^ /@action@/: a [type/@gio@/.Action]
    -> m (Maybe GLib.VariantType.VariantType)
    -- ^ __Returns:__ the parameter type
actionGetParameterType action = liftIO $ do
    action' <- unsafeManagedPtrCastPtr action
    result <- g_action_get_parameter_type action'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newBoxed GLib.VariantType.VariantType) result'
        return result''
    touchManagedPtr action
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data ActionGetParameterTypeMethodInfo
instance (signature ~ (m (Maybe GLib.VariantType.VariantType)), MonadIO m, IsAction a) => O.OverloadedMethod ActionGetParameterTypeMethodInfo a signature where
    overloadedMethod = actionGetParameterType

instance O.OverloadedMethodInfo ActionGetParameterTypeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Action.actionGetParameterType",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Interfaces-Action.html#v:actionGetParameterType"
        })


#endif

-- method Action::get_state
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "action"
--           , argType = TInterface Name { namespace = "Gio" , name = "Action" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a [type@Gio.Action]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just TVariant
-- throws : False
-- Skip return : False

foreign import ccall "g_action_get_state" g_action_get_state :: 
    Ptr Action ->                           -- action : TInterface (Name {namespace = "Gio", name = "Action"})
    IO (Ptr GVariant)

-- | Queries the current state of /@action@/.
-- 
-- If the action is not stateful then @NULL@ will be returned.  If the
-- action is stateful then the type of the return value is the type
-- given by 'GI.Gio.Interfaces.Action.actionGetStateType'.
-- 
-- The return value (if non-@NULL@) should be freed with
-- 'GI.GLib.Structs.Variant.variantUnref' when it is no longer required.
-- 
-- /Since: 2.28/
actionGetState ::
    (B.CallStack.HasCallStack, MonadIO m, IsAction a) =>
    a
    -- ^ /@action@/: a [type/@gio@/.Action]
    -> m (Maybe GVariant)
    -- ^ __Returns:__ the current state of the action
actionGetState action = liftIO $ do
    action' <- unsafeManagedPtrCastPtr action
    result <- g_action_get_state action'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- B.GVariant.wrapGVariantPtr result'
        return result''
    touchManagedPtr action
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data ActionGetStateMethodInfo
instance (signature ~ (m (Maybe GVariant)), MonadIO m, IsAction a) => O.OverloadedMethod ActionGetStateMethodInfo a signature where
    overloadedMethod = actionGetState

instance O.OverloadedMethodInfo ActionGetStateMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Action.actionGetState",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Interfaces-Action.html#v:actionGetState"
        })


#endif

-- method Action::get_state_hint
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "action"
--           , argType = TInterface Name { namespace = "Gio" , name = "Action" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a [type@Gio.Action]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just TVariant
-- throws : False
-- Skip return : False

foreign import ccall "g_action_get_state_hint" g_action_get_state_hint :: 
    Ptr Action ->                           -- action : TInterface (Name {namespace = "Gio", name = "Action"})
    IO (Ptr GVariant)

-- | Requests a hint about the valid range of values for the state of
-- /@action@/.
-- 
-- If @NULL@ is returned it either means that the action is not stateful
-- or that there is no hint about the valid range of values for the
-- state of the action.
-- 
-- If a [type/@gLib@/.Variant] array is returned then each item in the array is a
-- possible value for the state.  If a [type/@gLib@/.Variant] pair (ie: two-tuple) is
-- returned then the tuple specifies the inclusive lower and upper bound
-- of valid values for the state.
-- 
-- In any case, the information is merely a hint.  It may be possible to
-- have a state value outside of the hinted range and setting a value
-- within the range may fail.
-- 
-- The return value (if non-@NULL@) should be freed with
-- 'GI.GLib.Structs.Variant.variantUnref' when it is no longer required.
-- 
-- /Since: 2.28/
actionGetStateHint ::
    (B.CallStack.HasCallStack, MonadIO m, IsAction a) =>
    a
    -- ^ /@action@/: a [type/@gio@/.Action]
    -> m (Maybe GVariant)
    -- ^ __Returns:__ the state range hint
actionGetStateHint action = liftIO $ do
    action' <- unsafeManagedPtrCastPtr action
    result <- g_action_get_state_hint action'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- B.GVariant.wrapGVariantPtr result'
        return result''
    touchManagedPtr action
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data ActionGetStateHintMethodInfo
instance (signature ~ (m (Maybe GVariant)), MonadIO m, IsAction a) => O.OverloadedMethod ActionGetStateHintMethodInfo a signature where
    overloadedMethod = actionGetStateHint

instance O.OverloadedMethodInfo ActionGetStateHintMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Action.actionGetStateHint",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Interfaces-Action.html#v:actionGetStateHint"
        })


#endif

-- method Action::get_state_type
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "action"
--           , argType = TInterface Name { namespace = "Gio" , name = "Action" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a [type@Gio.Action]"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "GLib" , name = "VariantType" })
-- throws : False
-- Skip return : False

foreign import ccall "g_action_get_state_type" g_action_get_state_type :: 
    Ptr Action ->                           -- action : TInterface (Name {namespace = "Gio", name = "Action"})
    IO (Ptr GLib.VariantType.VariantType)

-- | Queries the type of the state of /@action@/.
-- 
-- If the action is stateful (e.g. created with
-- 'GI.Gio.Objects.SimpleAction.simpleActionNewStateful') then this function returns the
-- [type/@gLib@/.VariantType] of the state.  This is the type of the initial value
-- given as the state. All calls to 'GI.Gio.Interfaces.Action.actionChangeState' must give a
-- [type/@gLib@/.Variant] of this type and 'GI.Gio.Interfaces.Action.actionGetState' will return a
-- [type/@gLib@/.Variant] of the same type.
-- 
-- If the action is not stateful (e.g. created with 'GI.Gio.Objects.SimpleAction.simpleActionNew')
-- then this function will return @NULL@. In that case, 'GI.Gio.Interfaces.Action.actionGetState'
-- will return @NULL@ and you must not call 'GI.Gio.Interfaces.Action.actionChangeState'.
-- 
-- /Since: 2.28/
actionGetStateType ::
    (B.CallStack.HasCallStack, MonadIO m, IsAction a) =>
    a
    -- ^ /@action@/: a [type/@gio@/.Action]
    -> m (Maybe GLib.VariantType.VariantType)
    -- ^ __Returns:__ the state type, if the action is stateful
actionGetStateType action = liftIO $ do
    action' <- unsafeManagedPtrCastPtr action
    result <- g_action_get_state_type action'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newBoxed GLib.VariantType.VariantType) result'
        return result''
    touchManagedPtr action
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data ActionGetStateTypeMethodInfo
instance (signature ~ (m (Maybe GLib.VariantType.VariantType)), MonadIO m, IsAction a) => O.OverloadedMethod ActionGetStateTypeMethodInfo a signature where
    overloadedMethod = actionGetStateType

instance O.OverloadedMethodInfo ActionGetStateTypeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.Action.actionGetStateType",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Interfaces-Action.html#v:actionGetStateType"
        })


#endif

-- method Action::name_is_valid
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "action_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a potential action name"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_action_name_is_valid" g_action_name_is_valid :: 
    CString ->                              -- action_name : TBasicType TUTF8
    IO CInt

-- | Checks if /@actionName@/ is valid.
-- 
-- /@actionName@/ is valid if it consists only of alphanumeric characters,
-- plus @-@ and @.@.  The empty string is not a valid action name.
-- 
-- It is an error to call this function with a non-UTF-8 /@actionName@/.
-- /@actionName@/ must not be @NULL@.
-- 
-- /Since: 2.38/
actionNameIsValid ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@actionName@/: a potential action name
    -> m Bool
    -- ^ __Returns:__ @TRUE@ if /@actionName@/ is valid
actionNameIsValid actionName = liftIO $ do
    actionName' <- textToCString actionName
    result <- g_action_name_is_valid actionName'
    let result' = (/= 0) result
    freeMem actionName'
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method Action::parse_detailed_name
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "detailed_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a detailed action name"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "action_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the action name" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "target_value"
--           , argType = TVariant
--           , direction = DirectionOut
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the target value,\n  or `NULL` for no target"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_action_parse_detailed_name" g_action_parse_detailed_name :: 
    CString ->                              -- detailed_name : TBasicType TUTF8
    Ptr CString ->                          -- action_name : TBasicType TUTF8
    Ptr (Ptr GVariant) ->                   -- target_value : TVariant
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Parses a detailed action name into its separate name and target
-- components.
-- 
-- Detailed action names can have three formats.
-- 
-- The first format is used to represent an action name with no target
-- value and consists of just an action name containing no whitespace
-- nor the characters @:@, @(@ or @)@.  For example: @app.action@.
-- 
-- The second format is used to represent an action with a target value
-- that is a non-empty string consisting only of alphanumerics, plus @-@
-- and @.@.  In that case, the action name and target value are
-- separated by a double colon (@::@).  For example:
-- @app.action::target@.
-- 
-- The third format is used to represent an action with any type of
-- target value, including strings.  The target value follows the action
-- name, surrounded in parens.  For example: @app.action(42)@.  The
-- target value is parsed using 'GI.GLib.Structs.Variant.variantParse'.  If a tuple-typed
-- value is desired, it must be specified in the same way, resulting in
-- two sets of parens, for example: @app.action((1,2,3))@.  A string
-- target can be specified this way as well: @app.action(\'target\')@.
-- For strings, this third format must be used if target value is
-- empty or contains characters other than alphanumerics, @-@ and @.@.
-- 
-- If this function returns @TRUE@, a non-@NULL@ value is guaranteed to be returned
-- in /@actionName@/ (if a pointer is passed in). A @NULL@ value may still be
-- returned in /@targetValue@/, as the /@detailedName@/ may not contain a target.
-- 
-- If returned, the [type/@gLib@/.Variant] in /@targetValue@/ is guaranteed to not be floating.
-- 
-- /Since: 2.38/
actionParseDetailedName ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@detailedName@/: a detailed action name
    -> m ((T.Text, Maybe GVariant))
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
actionParseDetailedName detailedName = liftIO $ do
    detailedName' <- textToCString detailedName
    actionName <- callocMem :: IO (Ptr CString)
    targetValue <- callocMem :: IO (Ptr (Ptr GVariant))
    onException (do
        _ <- propagateGError $ g_action_parse_detailed_name detailedName' actionName targetValue
        actionName' <- peek actionName
        actionName'' <- cstringToText actionName'
        freeMem actionName'
        targetValue' <- peek targetValue
        maybeTargetValue' <- convertIfNonNull targetValue' $ \targetValue'' -> do
            targetValue''' <- B.GVariant.wrapGVariantPtr targetValue''
            return targetValue'''
        freeMem detailedName'
        freeMem actionName
        freeMem targetValue
        return (actionName'', maybeTargetValue')
     ) (do
        freeMem detailedName'
        freeMem actionName
        freeMem targetValue
     )

#if defined(ENABLE_OVERLOADING)
#endif

-- method Action::print_detailed_name
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "action_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid action name"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "target_value"
--           , argType = TVariant
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a [type@GLib.Variant] target value, or `NULL`"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_action_print_detailed_name" g_action_print_detailed_name :: 
    CString ->                              -- action_name : TBasicType TUTF8
    Ptr GVariant ->                         -- target_value : TVariant
    IO CString

-- | Formats a detailed action name from /@actionName@/ and /@targetValue@/.
-- 
-- It is an error to call this function with an invalid action name.
-- 
-- This function is the opposite of 'GI.Gio.Interfaces.Action.actionParseDetailedName'.
-- It will produce a string that can be parsed back to the /@actionName@/
-- and /@targetValue@/ by that function.
-- 
-- See that function for the types of strings that will be printed by
-- this function.
-- 
-- /Since: 2.38/
actionPrintDetailedName ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@actionName@/: a valid action name
    -> Maybe (GVariant)
    -- ^ /@targetValue@/: a [type/@gLib@/.Variant] target value, or @NULL@
    -> m T.Text
    -- ^ __Returns:__ a detailed format string
actionPrintDetailedName actionName targetValue = liftIO $ do
    actionName' <- textToCString actionName
    maybeTargetValue <- case targetValue of
        Nothing -> return FP.nullPtr
        Just jTargetValue -> do
            jTargetValue' <- unsafeManagedPtrGetPtr jTargetValue
            return jTargetValue'
    result <- g_action_print_detailed_name actionName' maybeTargetValue
    checkUnexpectedReturnNULL "actionPrintDetailedName" result
    result' <- cstringToText result
    freeMem result
    whenJust targetValue touchManagedPtr
    freeMem actionName'
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList Action = ActionSignalList
type ActionSignalList = ('[ '("notify", GObject.Object.ObjectNotifySignalInfo)] :: [(Symbol, DK.Type)])

#endif


