/*
 * // Copyright (c) Radzivon Bartoshyk 7/2025. All rights reserved.
 * //
 * // Redistribution and use in source and binary forms, with or without modification,
 * // are permitted provided that the following conditions are met:
 * //
 * // 1.  Redistributions of source code must retain the above copyright notice, this
 * // list of conditions and the following disclaimer.
 * //
 * // 2.  Redistributions in binary form must reproduce the above copyright notice,
 * // this list of conditions and the following disclaimer in the documentation
 * // and/or other materials provided with the distribution.
 * //
 * // 3.  Neither the name of the copyright holder nor the names of its
 * // contributors may be used to endorse or promote products derived from
 * // this software without specific prior written permission.
 * //
 * // THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * // AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * // IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * // DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * // FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * // DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * // SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * // CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * // OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * // OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/**
J1 zeros and extremums on [-76;76]

Generated in SageMath:
```python
from mpmath import mp, mpf, findroot, j1
from sage.all import *
import struct

DR = RealField(52)
DD = RealField(190)

def double_to_hex(f):
    packed = struct.pack('>d', float(f))
    return '0x' + packed.hex()

def split_double_double(x):
    x_hi = DR(x)  # convert to f64
    x_lo = x - DD(x_hi)
    return (x_lo,x_hi)

def print_double_double(mark, x):
    splat = split_double_double(x)
    print(f"{mark}({double_to_hex(splat[0])}, {double_to_hex(splat[1])}),")

zeros = []

# Step size to detect sign changes
mp.prec = 150

step = mpf("0.001")
epsilon = mpf("1e-35")
x = mpf("0.0")

previous_zero = R120(0)
j1_zeros = []

while x < mpf("76.0"):
    f1 = besselj(1, x)
    f2 = besselj(1, x + step)
    if f1 * f2 < 0:
        zero = findroot(lambda t: j1(t), (x, x + step), solver='bisect', tol=mp.mpf("1e-41"))
        previous_zero = zero
        j1_zeros.append(zero)
    if previous_zero is not None and abs(x - mpf(f'{round(x)}')) < epsilon:
        zeros.append(previous_zero)
    x += step

j1_extrema = []

x = mpf("0.0")
while x < mpf("76.0"):
    d1 = mp.diff(lambda t: j1(t), x)
    d2 = mp.diff(lambda t: j1(t), x + step)
    if d1 * d2 < 0:
        extremum = findroot(lambda t: mp.diff(lambda u: j1(u), t), (x, x + step), solver='bisect', tol=mp.mpf("1e-41"))
        j1_extrema.append(extremum)
    x += step

j1_zeros.extend(j1_extrema)
j1_zeros = sorted(j1_zeros)

print("static J1_ZEROS: [(u64, u64); 46] = [")
for z in j1_zeros:
    k = split_double_double(DD(z))
    hi = double_to_hex(k[1])
    lo = double_to_hex(k[0])
    print(f"({lo}, {hi}),")

print("];")
```
See notes/bessel_j1_taylor.ipynb
**/
pub(crate) static J1_ZEROS: [(u64, u64); 48] = [
    (0x0, 0x0),
    (0x3c5616d820cfdaeb, 0x3ffd757d1fec8a3a),
    (0xbca60155a9d1b257, 0x400ea75575af6f09),
    (0x3ca5c646a75d7539, 0x40155365bc032467),
    (0xbc9b226d9d243828, 0x401c0ff5f3b47250),
    (0xbca63e17ec20a31d, 0x402112980f0b88a1),
    (0x3cc02610a51562b6, 0x402458d0d0bdfc29),
    (0x3cc9a84d3a5fedc1, 0x40276979797ee5ac),
    (0x3cb2bce7fd18e693, 0x402aa5baf310e5a2),
    (0xbcc6932b987094f1, 0x402dba284a17ac59),
    (0xbcdd2a68e88ab318, 0x4030787b360508c5),
    (0xbca022f6b2b54db9, 0x403203f9a24e6527),
    (0xbcd21830197e9e86, 0x40339da8e7416ca4),
    (0x3cdeaafeaf8ec1af, 0x40352a1424a1a9fa),
    (0xbcc1bf33afef88f2, 0x4036c294e3d4d8ac),
    (0xbcb2d773b50cf8b9, 0x40384fb31dee1635),
    (0x3cc1a2686480d882, 0x4039e7570dcea106),
    (0x3cd0bdee27293d79, 0x403b75014427514d),
    (0xbcb42ce39ec976fb, 0x403d0bfcf471fccc),
    (0x3cbda49c2c143483, 0x403e9a179fba532a),
    (0xbcdbe3a1cd066b67, 0x404018476e6b2bf0),
    (0xbce6b00c1279ef0a, 0x4040df82ebd54e32),
    (0xbced5fbbff045068, 0x4041aa890dc5e97c),
    (0x3cd7d864bbf17a30, 0x404271eb1b80430e),
    (0x3cc9eafeca0ca4fc, 0x40433cc523d5cb69),
    (0xbce5cecac300a9a1, 0x40440447e50db184),
    (0x3cc489bd556e5109, 0x4044cefcf1734b62),
    (0x3cdd0fd96f29c211, 0x4045969bc7271083),
    (0x3ce4f716f3179d90, 0x404661315d6b133f),
    (0xbce158b763edd0e8, 0x404728e892a88fc9),
    (0xbcef3950a842db79, 0x4047f36312028ad6),
    (0x3ce97656bbc2396e, 0x4048bb2fa2037de3),
    (0x3ce85d7bdb30baf1, 0x404985928f96d51e),
    (0xbce71f8560ac9f18, 0x404a4d71fcb56f8c),
    (0x3ce3d41e041caa68, 0x404b17c038c2018c),
    (0xbcde6d04716d8d21, 0x404bdfb06eb790aa),
    (0x3cda139ce2cd08ac, 0x404ca9ec5a82324b),
    (0x3cc8b5cc7b4501c1, 0x404d71eb98682f07),
    (0xbcb12e6ef2e594e2, 0x404e3c1731d64f1e),
    (0x3cb399bfca430021, 0x404f0423f99b4b53),
    (0x3cdfd1ee8286358a, 0x404fce40efb1156e),
    (0x3c800660b51502f0, 0x40504b2cfcbb084d),
    (0x3ced3cacfc720418, 0x4050b034dde75b42),
    (0x3cc4b877d4f6d900, 0x40511446f60f1458),
    (0xbcee669304bfe748, 0x40517948db63675c),
    (0x3cfad20ca758a714, 0x4051dd600b743a9b),
    (0x3cf8eb4a94b63936, 0x4052425c7dcacdf6),
    (0xbcfa196892f68386, 0x4052a67859bc641e),
];

/**
Precomputed values in exact Bessel J1 zero.

Generated by MPFR:
```text
let mut arr = vec![];
for zeros in J1_ZEROS.iter() {
    let mpfr = Float::with_val(107, f64::from_bits(zeros.1)).j1();
    arr.push(mpfr.to_f64().to_bits());
}
println!(
    "arr: [{}]",
    arr.iter()
        .map(|x| format!("0x{:016X}", x))
        .collect::<Vec<_>>()
        .join(", ")
);
```
**/
pub(crate) static J1_ZEROS_VALUE: [u64; 48] = [
    0x0000000000000000,
    0x3FE29EA3D19F035F,
    0xBC91B9C1C3FB286F,
    0xBFD626EE83500BF2,
    0x3C8049770CE74C2E,
    0x3FD17DBF09D40D25,
    0x3CA0212F4E592523,
    0xBFCDDCEB4CE1BF4A,
    0xBC905DCC62D0D222,
    0x3FCA7F63FEA81F26,
    0xBCB6EB905BA2ABFA,
    0xBFC810F50225B04B,
    0x3CAA10B2F7B4E69D,
    0x3FC633E7F7F05301,
    0xBC97BC6D5A660382,
    0xBFC4B71D4CA2CC69,
    0xBC961C29FAC28FDF,
    0x3FC37DFA8F5A550A,
    0xBC87E3B01386785F,
    0xBFC2768D29C69936,
    0x3CAF5EFD41F756B6,
    0x3FC194EBA75B32F9,
    0xBCBF89DCEDB3EA9B,
    0xBFC0D0D36473E98C,
    0xBC9AAAF726A29E97,
    0x3FC02455675AB6D2,
    0x3C9451B6225ACBFB,
    0xBFBF161D0C28B48C,
    0xBCB40032091A4E00,
    0x3FBE0357C158B119,
    0xBCBCCB5A05A6E4AA,
    0xBFBD0B36E5737458,
    0xBCB5C457E4A6A2F1,
    0x3FBC29AE8400A320,
    0x3CB13169F65EFC7C,
    0xBFBB5B8273B75055,
    0xBCA5FB7DBD93E256,
    0x3FBA9E13A0DB6429,
    0xBC7C3482175F80D7,
    0xBFB9EF3BB2213B0B,
    0xBCA977092852774B,
    0x3FB94D3276914E51,
    0x3CB6D73591BFEB5D,
    0xBFB8B67A2481077D,
    0x3CB735BC851F7831,
    0x3FB829D06FEE9266,
    0x3CC29C7C75EEB12F,
    0xBFB7A62320798175,
];

/**
Following search for J1 (see [J1_ZEROS]) zeros and extremums:
at each zero and extremum we're doing Taylor series expansion
one that should be enough to cover whole interval between zero or peak
which is PI/4

Generated in SageMath and Sollya:
```python
def compute_intervals(zeros):
    intervals = []
    for i in range(0, len(zeros)):
        if i == 0:
            a = (zeros[i]) / 2 - 0.05 - zeros[i]
            b = (zeros[i] + zeros[i + 1]) / 2 + 0.05 - zeros[i]
            intervals.append((RealField(18)(a), RealField(18)(b), RealField(110)(zeros[i])))
        elif i + 1 > len(zeros) - 1:
            a = (zeros[i - 1] + zeros[i]) / 2 - 0.05 - zeros[i]
            b = (zeros[i]) + 0.83 + 0.05 - zeros[i]
            intervals.append((RealField(18)(a), RealField(18)(b), RealField(110)(zeros[i])))
        else:
            a = (zeros[i - 1] + zeros[i]) / 2 - zeros[i] - 0.05
            b = (zeros[i] + zeros[i + 1]) / 2 + 0.05  - zeros[i]
            intervals.append((RealField(18)(a), RealField(18)(b), RealField(110)(zeros[i])))
    return intervals

intervals = compute_intervals(j1_zeros)
# print(intervals)

def build_sollya_script(a, b, zero, deg):
    return f"""
prec = 500;
bessel_j1 = library("./notes/bessel_sollya/cmake-build-release/libbessel_sollya.dylib");
f = bessel_j1(x + {zero});
d = [{a}, {b}];
pf = remez(f, {deg}, d);
for i from 0 to degree(pf) do {{
    write(coeff(pf, i)) >> "coefficients.txt";
    write("\\n") >> "coefficients.txt";
}};
"""

def load_coefficients(filename):
    with open(filename, "r") as f:
        return [RealField(500)(line.strip()) for line in f if line.strip()]

def call_sollya_on_interval(a, b, zero, degree=12):
    sollya_script = build_sollya_script(a, b, zero, degree)
    with open("tmp_interval.sollya", "w") as f:
        f.write(sollya_script)
    import subprocess
    if os.path.exists("coefficients.txt"):
        os.remove("coefficients.txt")
    try:
        result = subprocess.run(
            ["sollya", "tmp_interval.sollya"],
            check=True,
            capture_output=True,
            text=True
        )
    except subprocess.CalledProcessError as e:
        return

def print_remez_coeffs(poly):
    print("J1TaylorExtendedSeries {")
    print_double_double("a0: ", poly[0])
    print_double_double("a1: ", poly[1])
    print_double_double("a2: ", poly[2])
    print_double_double("a3: ", poly[3])
    print_double_double("a4: ", poly[4])
    print_double_double("a5: ", poly[5])
    print_double_double("a6: ", poly[6])
    print_double_double("a7: ", poly[7])
    print_double_double("a8: ", poly[8])
    print_double_double("a9: ", poly[9])
    print_double_double("a10: ", poly[10])
    print_double_double("a11: ", poly[11])
    print_double_double("a12: ", poly[12])
    print_double_double("a13: ", poly[13])
    print("c: [")
    for i in range(14, len(poly)):
        coeff = poly[i]
        print(f"{double_to_hex(coeff)},")
    print("],")
    print("},")

degree = 23

print(f"pub(crate) static J1_COEFFS: [J1TaylorExtendedSeries; {len(intervals)}] = [")
for i in range(0, len(intervals)):
    interval = intervals[i]
    call_sollya_on_interval(interval[0], interval[1], interval[2], degree)
    coeffs = load_coefficients(f"coefficients.txt")
    print_remez_coeffs(coeffs)
print("];")
```
**/
pub(crate) static J1_COEFFS: [[(u64, u64); 24]; 47] = [
    [
        (0x3c61f1c324453b22, 0x3fe29ea3d19f035f),
        (0x35fa0d84854cf50d, 0xb9a948d98c79544d),
        (0xbc6e3631ae170265, 0xbfca41115c5df243),
        (0xbc18acc50ffb929f, 0x3f78d1448e6fed48),
        (0x3c0e85a7cffdb986, 0x3f8c441a2f9de22b),
        (0x3bd94ca1957aa906, 0xbf386671c18b088a),
        (0x3bd950ccbdd65736, 0xbf39e2504ddc7608),
        (0xbb61b3ae884e8323, 0x3ee34ccbca0c75d1),
        (0x3b7ab8f6e0e5328c, 0x3eda4973784d1087),
        (0x3b1f40e6cc67a3d5, 0xbe81045322aaab45),
        (0x3b06ea738f9ecf6a, 0xbe70fae0da6cdcef),
        (0xbabbaad3785f91ec, 0x3e13546cef5ed00e),
        (0x3a89c3da0f7d1467, 0x3dfe5ee82e6676f1),
        (0xba262fd170d5dbad, 0xbd9ec80cc8b644d5),
        (0xba23036bcf1b50d2, 0xbd83eb2e99627fad),
        (0xb9cc7616af2b9f56, 0x3d222bfcdb211968),
        (0xb9909fcb58b04d57, 0x3d03fb337676fc98),
        (0x3946bee5d8936909, 0xbca0901290ec992d),
        (0x391f5d570e3a4795, 0xbc7fa6a8aa2582e5),
        (0xb8b23101eae00b2a, 0x3c18078252545e8a),
        (0x386d13c8640abb7d, 0x3bf44b354a191eee),
        (0xb827a697c66ddb50, 0xbb8c75d1f25f35fc),
        (0x380d5a5801772dad, 0xbb6549fce4790a31),
        (0x378246c0eb5ba71c, 0x3affdbc62c23326e),
    ],
    [
        (0x35f9dfa8e2931f8b, 0xb95730bb59760ac5),
        (0x3c62de1143765a99, 0xbfd9c6cf582cbf7f),
        (0xbc46b8d73329a70b, 0x3faae8a39f51ad04),
        (0xbc47767d9698b3c4, 0x3fab589d1da13905),
        (0x3c0e65e41f715973, 0xbf7537544c331da7),
        (0x3bc117d1587f9833, 0xbf624b3409959064),
        (0x3bb8c81b4b053820, 0x3f26e4c2d5354224),
        (0x3b9859e946b98719, 0x3f083a06e30c4109),
        (0x3b51fd9c02e20b11, 0xbec9799d4c9f2549),
        (0xbb4ffc0b40a079e5, 0xbea33825cd2e2c16),
        (0x3aeb5032c6339f4e, 0x3e617069233e916c),
        (0x3a80c496c8db027f, 0x3e34569b22afc3c8),
        (0xba9a374b068a8d41, 0xbdf03b9e9651056f),
        (0x3a4c5798959dc28e, 0xbdbec62310af5fa2),
        (0xba11dc4cca029308, 0x3d75ec84e47b7d9e),
        (0x39d2cf2c8d7c612a, 0x3d417a40c942a964),
        (0x3971ff6b4d2284f1, 0xbcf67cb1d01a03c7),
        (0xb943da1ee0003f81, 0xbcbee7ff9372e125),
        (0xb8f9e098c8ea9826, 0x3c721fb8d23ce65d),
        (0x38d89765c4f16618, 0x3c35e053c1413792),
        (0xb8575b3ab1c49578, 0xbbe79265a43bea11),
        (0xb82c7bf56bef062f, 0xbba95d6cc94f810f),
        (0xb7f5bad532a4cc34, 0x3b59332b7e688450),
        (0x37ac610ae03b7f77, 0x3b1a2c1db24ae655),
    ],
    [
        (0xbc782d627160714a, 0xbfd626ee83500bf2),
        (0x35e215832cad36da, 0x39465d6b7a56edac),
        (0x3c6ae8952e6f1d78, 0x3fc55f6bec9ef962),
        (0x3c0d30f1a30544cb, 0xbf83d23336fd10e4),
        (0x3c2695c3589d6e1d, 0xbf88c77a983a0814),
        (0x3bea1200f42f4b09, 0x3f45cdc98db1cbe2),
        (0x3bd343248aee865b, 0x3f373576ff46ee3b),
        (0x3b8a8cd3c63aa60b, 0xbef2461447d7b423),
        (0x3b6a8a9edacbe6fe, 0xbed7b853456b6eaa),
        (0xbaff3625eaafdafc, 0x3e90abfc68274a98),
        (0xbaea64c013116245, 0x3e6ea7a1ee26124d),
        (0x3acef44f1141cf97, 0xbe235c0413e01419),
        (0xba7020655e19f397, 0xbdfb5c5d512fbafe),
        (0x3a2522345bdc9b62, 0x3daf4c5e26fd6f49),
        (0xba1a9c1d9fde6bb0, 0x3d81e4c43397bb8d),
        (0x39aead7c3865323c, 0xbd32addefc4e427f),
        (0x399ef826261673f0, 0xbd01e4fadc073884),
        (0xb9584f8fae96d4c1, 0x3cb12a0b89e91563),
        (0x3911cf3d91af7eef, 0x3c7c4246886f561c),
        (0xb8932b17cb520173, 0xbc290ee1bd0b60e5),
        (0x388f8ba7ff012875, 0xbbf21116fef61a90),
        (0x38093d7080102807, 0x3b9dcbd61b5fcfcc),
        (0xb7ddad88c94d63e2, 0x3b62f7833d3a6e0a),
        (0xb7a7e960c2811c48, 0xbb0f75a3322d6adb),
    ],
    [
        (0x35d923fac9298498, 0xb943ab00450c21a5),
        (0x3c7af22d033ee0a8, 0x3fd33518b3874e8a),
        (0xbc23b4d62984701e, 0xbf95e70dc60362bf),
        (0xbc476d8715df734e, 0xbfa80c83bdeee5b0),
        (0x3c03dac20aaf9d3e, 0x3f69a4b292e3de42),
        (0x3bd887eed7bf2f4a, 0x3f613fbc7d698217),
        (0xbbac52a914699396, 0xbf207358bbdbff91),
        (0xbb64764498326fe8, 0xbf0796a751f89051),
        (0xbb6df66e2a5b80ce, 0x3ec4255b015aded4),
        (0xbb3ae5bbd6b6b7b2, 0x3ea3026e0ce97ab9),
        (0xbaf53c3132b45416, 0xbe5d48dcdae92f2c),
        (0x3ac329ab58cdf5c8, 0xbe344639d7eeb0a6),
        (0xba7227361e36b492, 0x3dec62ccb4a32eb2),
        (0x3a436c156b981e3c, 0x3dbecae92e85448e),
        (0xba15900bb4077790, 0xbd73bb6898d7381b),
        (0x39ceeaf13516632f, 0xbd4183edbb8f3baa),
        (0xb946b19e901a3f2a, 0x3cf4ae3e7e418192),
        (0x395392c9d1b10508, 0x3cbefbb1e7e7f06a),
        (0xb91d78a8cbca8fc8, 0xbc70f28d91f2bb8c),
        (0xb8c1412abdb69981, 0xbc35ec327b001a36),
        (0x3866c77527da5dfe, 0x3be65536194079a9),
        (0x37d83c37616b601d, 0x3ba9644a2a48ce1f),
        (0xb7fc2f10d93b3bda, 0xbb58069a9c48e50d),
        (0x37b28c16423dadcd, 0xbb18f8666a0fbd52),
    ],
    [
        (0xbc7d48dbfa0ea1a4, 0x3fd17dbf09d40d25),
        (0xb5f12f0e1a0ea25f, 0xb95b2c846748c809),
        (0xbc61eb914d33c2b5, 0xbfc1404bf647c28f),
        (0x3c098a23a393f866, 0x3f74f4df2769f830),
        (0x3c24ae93dcb9bf99, 0x3f85c6285429b66d),
        (0x3bdddb259c934d19, 0xbf3d68ab722881bd),
        (0xbbd8cd731b94002b, 0xbf356acb6452d860),
        (0x3b60d2c0a7987aa7, 0x3eec10b47cf7ef69),
        (0xbb3cc8bfc9cf9d54, 0x3ed67eaae97bbc86),
        (0x3b021d7ef8f06ed1, 0xbe8bb6530c63f2df),
        (0xbaf7d003a6f8a38a, 0xbe6d87201e450edd),
        (0xba5df682e9056c25, 0x3e20f47c83ec550b),
        (0x3a784cc2fbcd5b3c, 0x3dfa98331f6ea797),
        (0x3a3af02acc5d445a, 0xbdac70414a236ade),
        (0xba1ac139214727ff, 0xbd817c057a5fc937),
        (0x39abeb4e52ce6eb7, 0x3d316fea14d29892),
        (0x39a6a5fe18553faf, 0x3d0189bd3eb9f3c6),
        (0x3946aa341fe1b254, 0xbcb05af44bb23c5f),
        (0x39010e85f5ece2bd, 0xbc7bbdb45d92e4fc),
        (0xb89b4dc8ed25771b, 0x3c284360a40ed89d),
        (0x38779d6e01f88769, 0x3bf1bfb967530933),
        (0xb82198845d1335e3, 0xbb9d37a860a66a8e),
        (0xb806974ac174343c, 0xbb62a0bbacab19c8),
        (0x37906170939ea2eb, 0x3b0e5619160fd338),
    ],
    [
        (0xb60ab4d43f876701, 0xb969264876fdf3dc),
        (0xbc5052a3a2541c57, 0xbfcff654544ebcd1),
        (0xbc01b402d42eae53, 0x3f89223ff2c0785b),
        (0xbc323a2755909c5f, 0x3fa4b0c5d5da6789),
        (0x3bb3ea54acd19ff8, 0xbf5f91a9ee0d2897),
        (0x3bfc41f7f5f8cde0, 0xbf5f51c2489b9e6f),
        (0x3bad7af9dcc67129, 0x3f16b4c9ca0f770d),
        (0xbbadd341169fa322, 0x3f063c5475439cb2),
        (0x3b59f6ed48ce32ee, 0xbebe3725daf69867),
        (0x3b21f00381a6cb34, 0xbea25c1238b32e59),
        (0xbaebcfa91cc88fb7, 0x3e57486f6b9aa951),
        (0x3adb600d25e5d0fc, 0x3e33e3bf248277ee),
        (0xba5d774620c03673, 0xbde78a38a73e7c07),
        (0x3a59e464c4822ecc, 0xbdbe844eb6b211d0),
        (0x3a19b0c1a7646942, 0x3d70e24abb406b2c),
        (0x39d0a87b33c9afed, 0x3d41797e5ead05d1),
        (0x397bece1bd73373b, 0xbcf21fc0f119f3a6),
        (0x39422a1731e7d087, 0xbcbf0c12e4c83411),
        (0x390bbdff1dae57a0, 0x3c6e44232fae84a9),
        (0x38d9844b6a7d039c, 0x3c360804e71ba342),
        (0x38824f2c3cf7a869, 0xbbe43e13e137bab9),
        (0xb8314bfff0c04984, 0xbba9903092d3c1c6),
        (0xb7e5d3b1cb26c861, 0x3b56091b8d2a60e0),
        (0x37988d9bfceddedd, 0x3b18e8b26041597f),
    ],
    [
        (0x3c50f8942d3f902b, 0xbfcddceb4ce1bf4a),
        (0xb62c35de62add74d, 0x3982dff6c1e15731),
        (0x3c433d5334b42b3d, 0x3fbda52116c0a640),
        (0xbbe72468a28b4443, 0xbf6a9da4603b67ea),
        (0xbc0c5e0831771de5, 0xbf8331e74ea59ab8),
        (0xbbdac4579b1b1f5c, 0x3f33e5cb6eba6eaa),
        (0x3bd0ad97399a9309, 0x3f33885fe9afa541),
        (0xbb74d732b8ca0df8, 0xbee494c0f4b0680b),
        (0x3b783e803ffeb597, 0xbed512b9d37762d7),
        (0x3b1382127dd60cd6, 0x3e85a861082bfb7f),
        (0xbb05de78d8bef340, 0x3e6c323ea0a042c3),
        (0xbaba616d9a2d0acd, 0xbe1bcc962f7b91eb),
        (0xba9907e5c13d4bf5, 0xbdf9bc94e2f29a4f),
        (0xba4dd160d698c980, 0x3da82bc6fcfa8ee1),
        (0xba0febc61d1c4bb8, 0x3d81141ce7b77ff8),
        (0xb9b71ce5b90d51b7, 0xbd2e79ccb186eec5),
        (0xb9729b55c285b0b8, 0xbd013e1fbe002f34),
        (0x394b7b4b328dbd37, 0x3cad36d12c86586f),
        (0x3913c4bb8b85fa48, 0x3c7b66070cb3ac2c),
        (0x38cf4dafb0ab1446, 0xbc260cbaacd645d2),
        (0xb892f6df683de3de, 0xbbf19605537cf4ab),
        (0xb806112791bea534, 0x3b9aed6b864dcd18),
        (0xb7dc24510173386c, 0x3b627eca9f946e65),
        (0xb79c60e9a1146a47, 0xbb0bfef86c0d63d6),
    ],
    [
        (0x3601c162477c9abb, 0x396159aca5db6572),
        (0x3c6c8c66d2e42063, 0x3fcbf3337873a7d8),
        (0x3c25e81c4baa875d, 0xbf80c83a2d7add33),
        (0x3c44192692d7c60a, 0xbfa251858011816b),
        (0x3bd0475c48fd4015, 0x3f559eb160bf72d8),
        (0x3b9e04d420af1ac6, 0x3f5c5bce33af2d77),
        (0x3b7eb4916c85fb39, 0xbf10413e306e0039),
        (0xbb9e3a42c4a3fc86, 0xbf04a6704d05ad0b),
        (0xbb5a4f57a7a75039, 0x3eb6c43eedfed6c9),
        (0x3b2d30c35715c7fa, 0x3ea16abd7815de74),
        (0xbae381d774667710, 0xbe5257f16f5d4346),
        (0xbad24f76ba3d2cac, 0xbe332db1b4b2ff8b),
        (0xba886b6bef46e0df, 0x3de33acccf7bfdcb),
        (0x3a40f79d1cb77e70, 0x3dbdc8f5682566ce),
        (0xb9f02059cbf2dc1b, 0xbd6c6513386da08e),
        (0xb9aee6c6f049704d, 0xbd413585a9b760a0),
        (0x396399b887813c07, 0x3cef322ea2ca7028),
        (0x3957137aaf1256dd, 0x3cbec749ba83ed87),
        (0xb90e8d0898b37a61, 0xbc6a8aba908ced81),
        (0xb8d53fcac77872af, 0xbc35f2abf33d3a06),
        (0xb881d148fd35588c, 0x3be206e2b69fc921),
        (0xb8490f14e83f785c, 0x3ba98eef781c5c36),
        (0xb7ea20a48a9f15a7, 0xbb53df43cc1537c9),
        (0x37b0c3f1e8cb94e8, 0xbb18d93abc409fbc),
    ],
    [
        (0xbc26397095487bbc, 0x3fca7f63fea81f26),
        (0x35c7769fcbedbba7, 0xb94670973020cb53),
        (0x3c4341d92ebaf230, 0xbfba60afb06640cf),
        (0x3c0aa0cf7ee02729, 0x3f62c1e930935d3c),
        (0x3c2253175b5c623d, 0x3f814506466d7f1f),
        (0x3bcf594f6bbbe6b1, 0xbf2cca8c0c0eaa3f),
        (0xbbc23306b101db0c, 0xbf31df821cc1377e),
        (0xbb6f426f813ba1cf, 0x3edee8814ed0ac45),
        (0x3b3530e090190f04, 0x3ed3a365a4199dd1),
        (0xbb253c7c31f63d20, 0xbe80ed2f9c3e458e),
        (0xbae6d6e3d35d7af7, 0xbe6ab3b37c5271b3),
        (0xbab5876ea3da2c96, 0x3e1684d6e62b5c66),
        (0xba61146a8712cc76, 0x3df8b105a5120ecf),
        (0x3a497e08482e4f36, 0xbda42dc5991b9d46),
        (0xba1c93b9ca69994f, 0xbd808d6405ffdf4d),
        (0xb9bb9c06669b183b, 0x3d2a152033665a46),
        (0x39a7ab21a3b60319, 0x3d00d7c118b17a82),
        (0x39456872094da304, 0xbca984ffd3ac0d7d),
        (0x3909c82687cc959e, 0xbc7aecb2b37b54c2),
        (0xb8cf42b62c996f29, 0x3c2397005ee8dd8b),
        (0xb886e35dc2991042, 0x3bf15c9c692b5bb4),
        (0xb832ecad9f92b6aa, 0xbb98431fbb78646d),
        (0x38062f0967c8bd95, 0xbb62529cf346132a),
        (0x37a9a4acee74819c, 0x3b0964837b2ece97),
    ],
    [
        (0x360120a78f21538d, 0xb97c8b54640b722a),
        (0x3c6e9557ccd17041, 0xbfc925c6fca08f55),
        (0x3c091bef692396e7, 0x3f786dd32e059b0e),
        (0x3c3dac1b118bbe70, 0x3fa09463bbd0367f),
        (0x3be231ff192e138d, 0xbf4fda0298c8768b),
        (0xbbf0185527f60d8b, 0xbf59f4be60758fb1),
        (0xbbac474076e2749b, 0x3f0877991af9d1bb),
        (0x3b85632e8431fcaa, 0x3f032cb00ee8c1f3),
        (0x3b46a07ad1a2663e, 0xbeb19d8ce8c35f58),
        (0x3b443944f6114d11, 0xbea06a042fbba455),
        (0xbaeb45186aece340, 0x3e4d3a689e677731),
        (0xbadf9a18bdd365cc, 0x3e325108c4ce2b64),
        (0xba6505bb4a19817a, 0xbddf7b8e9ab5314c),
        (0x3a5ad487a5bbd7ab, 0xbdbcc40d05652650),
        (0xb9dcf99ac25c15c5, 0x3d67cd76e2d7d943),
        (0xb9d024e4f22a308b, 0x3d40c58770232cfe),
        (0x3952f8f057a0b1c2, 0xbceaafec4cc364ff),
        (0x3933c2290c85768f, 0xbcbe36dd573e6a63),
        (0xb8dfd609484ae481, 0x3c67199e5ff69f74),
        (0x388f3830f4ff63d7, 0x3c35ab8bb9a71dac),
        (0x387ca89ca7c948bf, 0xbbdfd6f8b70e778b),
        (0xb844329e8362812f, 0xbba9598a6da0f2c3),
        (0x37fd2c2ab7de98e2, 0x3b51c436362641e5),
        (0x37b80a5354c74c14, 0x3b18a9fec4d5983d),
    ],
    [
        (0x3c2a5f1938003f60, 0xbfc810f50225b04b),
        (0x35f6ee82870d84bf, 0xb97d05564a508c39),
        (0x3c5462bc86c50e66, 0x3fb7fdf97ac36b1f),
        (0x3beaefb0d3cf3530, 0xbf5c3c256a8caa05),
        (0xbc191dbdbe49d9f5, 0xbf7f98feb7286b47),
        (0xbbcd0c6b4ea34036, 0x3f25f6559e5686e2),
        (0xbbd9de4a690cb448, 0x3f3080f57ac215af),
        (0xbb69ed0a7174b68e, 0xbed80c51397e5eba),
        (0x3ae06f2933360b4a, 0xbed256db543cd140),
        (0x3b1af76fcd5528a0, 0x3e7af7598a219824),
        (0x3b06515217062c3a, 0x3e69398226ca2305),
        (0xba795db0bb871535, 0xbe1260985d92587d),
        (0xba74a82a307b019b, 0xbdf792bb3eea6f67),
        (0x3a3817c8f7ec2b4c, 0x3da0d862695e1ab2),
        (0xba1b8749aa4dca22, 0x3d7fe52adc2ba19c),
        (0x39cda3a73a18530c, 0xbd263801377428c8),
        (0xb997a6607141db31, 0xbd005a1befd1620e),
        (0x394738fbaf9e72c3, 0x3ca620504263602e),
        (0x3919acd0b705176b, 0x3c7a4e14e3f85acf),
        (0x38bbce9a9cbe0c23, 0xbc213ef23a1a245a),
        (0x389387792bae4d33, 0xbbf10cae24ff432c),
        (0x383135b979266427, 0x3b95a41b5b0291e9),
        (0x380630df36241504, 0x3b62113e229c18af),
        (0x3779c024d092ed81, 0xbb06d663d6acddea),
    ],
    [
        (0xb5e79c1121733c06, 0xb97dd765a46686aa),
        (0x3c62da0057f84d3f, 0x3fc70c511227d5aa),
        (0xbbfb574e506cd46d, 0xbf72ccb0e97558da),
        (0x3c2e61277dedefa5, 0xbf9e7dc08e70e99a),
        (0x3ba77952d9976f3f, 0x3f48acdc5b058c0e),
        (0xbbf340f4df902276, 0x3f580503724ad30a),
        (0xbba115f68568f545, 0xbf032ee4ca1fcafb),
        (0x3ba6a66dcfa3c51f, 0xbf01e5d2836c8d99),
        (0xbb2b2738baeaafd8, 0x3eac129f077bb163),
        (0x3b33825b34f3c4e4, 0x3e9ef161591181a2),
        (0x3ae4e499d619d66a, 0xbe47b9bb07f19f82),
        (0x3ac1a055b5da7ef9, 0xbe316f3937595d96),
        (0xba7ad6f9f1cbf4a9, 0x3dda0bc8665b6876),
        (0xba4fc8c4f001bcc9, 0x3dbba135f99a9e23),
        (0xba00713e44285b86, 0xbd640d543d2cb545),
        (0xb9d8806fb4e057c8, 0xbd403d0592186aa2),
        (0x3976ebbceb30453a, 0x3ce6db5e22c00286),
        (0xb95cfcc8d365b2a3, 0x3cbd745a1f9b2cc0),
        (0xb8e84bda052a24a4, 0xbc64141c2da5de65),
        (0x38cd92252322f7fa, 0xbc353f4e8c94bba5),
        (0xb8715b904e388e03, 0x3bdc0988ad3d45d6),
        (0xb7f1437e75284c5c, 0x3ba8f98a7bde325e),
        (0xb7e5101387db5e8b, 0xbb4fa52f2ce39750),
        (0x37b59c36f27d26bd, 0xbb1859ec268d8e36),
    ],
    [
        (0xbc6b166d180d579d, 0x3fc633e7f7f05301),
        (0x35ffee1a88c4c7db, 0x39597a12d5b67dc0),
        (0xbc100659a075cf2f, 0xbfb6273784c1c06e),
        (0x3bfcb74bd087b3a0, 0x3f563ae94ade18d4),
        (0x3c02a45d712493c7, 0x3f7d4666536c88b9),
        (0x3b825e9ac0d7a01e, 0xbf216d528345ca11),
        (0x3bbb9405cb89e345, 0xbf2ec0dcdbb7c5fe),
        (0xbb7cb6decc20866e, 0x3ed34e966b0b09f8),
        (0xbb7a79f0d76f0cbb, 0x3ed135c64dc2d8d0),
        (0x3b01209dcbada35f, 0xbe75f7bc78b5fc2b),
        (0xbaf600f169603982, 0xbe67dc35b0764096),
        (0xba9e587e011af4be, 0x3e0e6d697361ea54),
        (0x3a9d43cb8e00cb1a, 0x3df679e3704987b1),
        (0xba3d2f784a54c5d4, 0xbd9c595f278a4dc9),
        (0x3a0e613d8bf8177d, 0xbd7ea36aef56e594),
        (0x39bad5b0fa90b31d, 0x3d22fd66b4e699f0),
        (0xb983e4c916a5d37c, 0x3cffa04c9f95e420),
        (0xb929ce3db1f0e449, 0xbca32f47f100cd0a),
        (0x38d1dc8a519c68bb, 0xbc7996582817ca81),
        (0x38bd0c28a3f651c5, 0x3c1e4c27a12867d7),
        (0x3893ddf51221f64b, 0x3bf0aafd1db63771),
        (0x383228e1d6fab232, 0xbb933b296a9fe2c2),
        (0xb8026ac12c9d03cc, 0xbb61bce573aebd8d),
        (0x37a6fcc6415460a7, 0x3b0476ec3eb1a5b8),
    ],
    [
        (0xb605d92a84ac636d, 0x396c0a10d4b8d049),
        (0x3c6a47ab4241a9f3, 0xbfc5664e13b70622),
        (0x3c04d78c254f378c, 0x3f6e16555e108dc6),
        (0x3c1fe75afd6ceb7e, 0x3f9c5e1ad9fb2f40),
        (0x3be099fe50ede362, 0xbf43d369f958e56a),
        (0xbbf3ed70de3ce58d, 0xbf566f4ec27a96e9),
        (0xbb9b30f0921d946f, 0x3eff0de0532652d5),
        (0xbb747f8e74699e57, 0x3f00cf264341409e),
        (0x3b4f302f945bfb6b, 0xbea6f46d51e5766f),
        (0xbb37090d5299f2bd, 0xbe9d407f7c248d45),
        (0xbaeee2a1b6297a03, 0x3e43a33cd9df6696),
        (0x3ad36edf4ce5ba78, 0x3e309901b0a816e5),
        (0xba69797c9a665948, 0xbdd5d856a58443f0),
        (0x3a581a0de548434c, 0xbdba7cbcd8fc075f),
        (0x39f4161aad5502dc, 0x3d610b62c2fd4020),
        (0x39b2429685721bb6, 0x3d3f56a09da1af17),
        (0xb986edb5e42e4df0, 0xbce3ae6849a19196),
        (0xb956860a8cce360d, 0xbcbc977524bd849e),
        (0x3904f25392c2b4b0, 0x3c617fc9c0a81a86),
        (0xb8b42258c6440740, 0x3c34bbe439c863fb),
        (0x385dba6e1d13451b, 0xbbd8b5326e15f6b7),
        (0xb84cacd442a85cbe, 0xbba87bd53addeb1f),
        (0xb7ed606f4edc8467, 0x3b4c2aa54d8d8127),
        (0xb788c3a332c0e10d, 0x3b17efec97ba8f61),
    ],
    [
        (0xbc4f6f339127993c, 0xbfc4b71d4ca2cc69),
        (0x35ca4f8d42ff3858, 0x3971bd20421445e5),
        (0x3c5422c1a1ae8e1c, 0x3fb4ae245697fba6),
        (0xbbf4ff572c18ea0c, 0xbf5215e4e1a5f1d6),
        (0xbbf259ec9aa6f76a, 0xbf7b633ed6d9cf61),
        (0x3bab33aa4933effd, 0x3f1c7f17b4b7dbbd),
        (0xbb79333bf7f0a288, 0x3f2ce01b8b6aa34c),
        (0x3b5f4e5959a7242e, 0xbecfced71b11e35b),
        (0x3b57f36535a55c97, 0xbed03c9d5823261d),
        (0xbb15ca3a5fd6e4c3, 0x3e724508091063b2),
        (0x3b0278797fb483cd, 0x3e66a2d20111e303),
        (0x3aa462d914cbe692, 0xbe0995a18f8e6888),
        (0xba92d2f68fe7bbf3, 0xbdf572d1a074f644),
        (0xba3d2febd6b25c97, 0x3d981df03c191241),
        (0xba1df51a16a51cab, 0x3d7d6895e48f3e4a),
        (0x3979fc3e189a5876, 0xbd205887f0b4463d),
        (0xb99ef66a10579222, 0xbcfe86703dca7086),
        (0xb93a92c2e3662797, 0x3ca0b3c70d522804),
        (0xb91294d0f373d163, 0x3c78d28b840bbf9e),
        (0xb8a3ac6e917d987d, 0xbc1aa928bdc7c69b),
        (0xb889057510b0376b, 0xbbf03e6bbda694dd),
        (0xb83da10a22fdbcd6, 0x3b9117421c5b8302),
        (0xb804a226000323af, 0x3b615b28f6d151e7),
        (0x378448f5a65efa22, 0xbb02540c2640944f),
    ],
    [
        (0x35f8435be9512d0c, 0x39596a809853afc4),
        (0x3c6316f8ffd294bc, 0x3fc40f90793605bb),
        (0xbbd411ad350e3915, 0xbf68c833077fbeae),
        (0x3c051eb6f09da299, 0xbf9aa0ce0421d1a8),
        (0xbbec0fe78ad65dee, 0x3f405fa598ef5d1d),
        (0x3bff2085596f93c4, 0x3f551d30d78ab526),
        (0x3b92b31b50a0ff60, 0xbef9c5807675c5f6),
        (0xbb7cb5b05cc545a3, 0xbeffc1bbf57e3ae2),
        (0x3b3cc24f7215eccc, 0x3ea32dfea2518ce6),
        (0x3b18c92945575b15, 0x3e9bc212085dcbc6),
        (0xbaecddf3334d6159, 0xbe408b946d64c5c2),
        (0x3ac0b1a7f29c826b, 0xbe2fa8f9d8da736a),
        (0xba756320d1b782d7, 0x3dd293fe14af1d0b),
        (0xba5e3bcddd76a2bf, 0x3db96544cb75a592),
        (0x39f0a916b1e33716, 0xbd5d4750748e1ec5),
        (0x3992aa2022ba01f9, 0xbd3e341812329adf),
        (0x396e5a994597409d, 0x3ce112aa494174e0),
        (0x395a9a5bb21cd5ce, 0x3cbbb1656dd4f875),
        (0x38fd92cb597fcae9, 0xbc5ea7b95ef8e08b),
        (0x38c2aa6384ba6bbe, 0xbc342cad8878d4a5),
        (0xb868a671070d345c, 0x3bd5d76ac6e68151),
        (0xb82311f5dacdef0c, 0x3ba7ec1ff36b8002),
        (0xb7db96a9ba77f2bd, 0xbb491cad22e089cd),
        (0xb7b0a5b54e7f6cac, 0xbb177445429e42ae),
    ],
    [
        (0x3c4f5ffd019535e1, 0x3fc37dfa8f5a550a),
        (0xb5aaaff7214acba6, 0xb93cb5c70d300ac5),
        (0xbc5c4cd2161ee66e, 0xbfb3775c1a04f09c),
        (0x3bd3d562913491bc, 0x3f4e2b4810a46c60),
        (0xbc1b976f331a69fb, 0x3f79d151a72b83a8),
        (0x3bbcef8b51459dfe, 0xbf17d8e5a090e4e6),
        (0xbbcb7ef1646f65dc, 0xbf2b49a6427386a0),
        (0x3b592486856820bb, 0x3ecac10957ddd2eb),
        (0x3b424317f186fd5b, 0x3ececa620745d3d3),
        (0xbaf34eef3c6c9553, 0xbe6eefc7e795dcdd),
        (0x3ae4b6734611daf2, 0xbe658c5d2a0da41d),
        (0x3aa97ce5a3d8c757, 0x3e05d4721f44a8f9),
        (0xba9b1942d77688ab, 0x3df481ce2314af57),
        (0x3a2ac654ac1c449a, 0xbd94c0d3279e9252),
        (0xba1f470c6e36c705, 0xbd7c3ea70752fc73),
        (0xb97829121cc317d9, 0x3d1c61d5b166a9e5),
        (0xb989ebd713c25b94, 0x3cfd72bf188712ae),
        (0xb9078152b026b94d, 0xbc9d427a9891c892),
        (0xb904e8f099d7cfab, 0xbc780c9634dc0029),
        (0x38b5c69bf0be9c48, 0x3c178e382c9b5ff2),
        (0x38536f9bd3face06, 0x3bef99f1b78427a0),
        (0xb828de3e89d5fd1e, 0xbb8e7352b4b18839),
        (0xb805ad010982fbed, 0xbb60f1a2b1d2a86e),
        (0x37abf1ea36b06bd9, 0x3b00708aca8b132c),
    ],
    [
        (0xb5c06f78c8d4a275, 0xb93d524b57168b9f),
        (0x3c689d1f48185c7e, 0xbfc2f2072e638cf4),
        (0x3c0f48257333a5e0, 0x3f64df208bbd44f1),
        (0xbc282c4cf012e4f5, 0x3f992bb5e1e159fc),
        (0x3bb5967313f39524, 0xbf3ba181c06897cd),
        (0xbbea6566cfb71c2e, 0xbf53fe9d5baa4a3d),
        (0xbb98b4ff32e89d05, 0x3ef5d17602b01cac),
        (0xbb84f1d77f1fffd7, 0x3efe26d3747fe829),
        (0xbb47896f2e82c323, 0xbea0509768ab6ecb),
        (0xbb30ede6356a7015, 0xbe9a70f232d9d06c),
        (0x3ade8fe4e5a1ed7e, 0x3e3c509252de33f9),
        (0x3ac6ae9c749edc67, 0x3e2e454fee07116e),
        (0x3a73b1f5bbe63372, 0xbdd0015b062ba122),
        (0xba44c8407e1ecc95, 0xbdb860e95adf8412),
        (0x39fddd4f18607329, 0x3d59691e90f7ccfc),
        (0x39d00120e60006cf, 0x3d3d1ce7997b42b4),
        (0x39682a77f2b9ac62, 0xbcdddcd1c52fea1b),
        (0xb95c20e8a1ea2f57, 0xbcbacd10a03d8921),
        (0xb8f1731302036655, 0x3c5b043a4e72efe2),
        (0xb8d6d22b4e2f1fae, 0x3c3399ba323db8d6),
        (0x385b209322c8d73d, 0xbbd364e661355e9a),
        (0x3838be7c1f6126ff, 0xbba75381cf058174),
        (0x37d1ce55ad6a8310, 0x3b46755328d7d302),
        (0xb7b75f604bab0893, 0x3b16ee50ed715cef),
    ],
    [
        (0xbc6b9fbd89653a0a, 0xbfc2768d29c69936),
        (0x35ca97e1f25db68b, 0xb93ffabfa86ef843),
        (0xbc592c5350d4d817, 0x3fb271811730b0ef),
        (0xbbcd42067c35395a, 0xbf49a8df96a1225e),
        (0xbbde8196594d5f87, 0xbf787c81cf1c6fc4),
        (0x3ba098686a079be4, 0x3f14549cdbb77978),
        (0xbbcc0c93813e02ce, 0x3f29ed2568116e19),
        (0xbb638a1fa72a2206, 0xbec6e4136f033ace),
        (0xbb692b62efc42c9a, 0xbecd53330316cde7),
        (0x3af835560f2b31cd, 0x3e6a983b5782dfca),
        (0x3b0d287b84d78d98, 0x3e64952ba7c5a1dc),
        (0xba72905636bd36a9, 0xbe02df3ad6f82e0d),
        (0x3a99c062ae844d27, 0xbdf3a70f9a89d2c0),
        (0x3a3c9ffd5cdbf957, 0x3d920e086c17f618),
        (0x3a1e25413f7cf79b, 0x3d7b29a554c10cda),
        (0x39b937b07d944fb1, 0xbd18dbe08f4b3b70),
        (0x399966ea8cf6d238, 0xbcfc6bd9fc31d25f),
        (0xb9310d7ab87b2019, 0x3c99ce5a745e4984),
        (0xb918ade94c5553fb, 0x3c774addf0c26059),
        (0x38be87d79b895c6e, 0xbc14eb89eaaa0820),
        (0x388e01622132e11d, 0xbbeeb605f9932592),
        (0x37f1d8f9c78e8f53, 0x3b8b3a602349f9f9),
        (0x37fda89d6c3f2718, 0x3b6084d788afe5ac),
        (0x379093629c8ee8ea, 0xbafd92b61b88c69e),
    ],
    [
        (0xb61e3a4d721e8324, 0xb970e12b7d27cf07),
        (0x3c51f9b16832f365, 0x3fc1ff5eec6a01cd),
        (0xbc0f89ce0d1cad55, 0xbf61e438b722c3b5),
        (0x3c39a4b7b3ed5aa3, 0xbf97ed5fffc1c774),
        (0xbbdc35d9a8eaece6, 0x3f37b7997babd9ca),
        (0xbbe39da40664c597, 0x3f53081def9612c5),
        (0x3b8551b9d43f8119, 0xbef2c5f5edafc4e9),
        (0xbb9dbae69d6d1983, 0xbefcc11a59e13739),
        (0x3b35152e3ec9bcce, 0x3e9c2c3a1b8014a3),
        (0xbb333a08049d2a47, 0x3e9946d1dab7bd01),
        (0x3ad9a43f93c7a3c8, 0xbe388db61946be64),
        (0x3ac51a4e1f57e228, 0xbe2d04d33be580e8),
        (0xba5f39a3631ac3ea, 0x3dcbe64386d2c5c9),
        (0xba50e72fbf713685, 0x3db77142e0e4497d),
        (0x39f05d87c3131a89, 0xbd56458476678b0a),
        (0x39dcadde2840dbd3, 0xbd3c15e96b25b19b),
        (0xb970bb70de53f974, 0x3cda545e6eacddb5),
        (0x39573a21f1dd7482, 0x3cb9f0a9b7519cb4),
        (0xb8f26c0a88ddf73c, 0xbc57f751c3434bc5),
        (0x38dba5906a1609ba, 0xbc33083bfa2b9853),
        (0xb876257782c7fb32, 0x3bd14f52f8c40156),
        (0xb8455e6398eb5536, 0x3ba6b866ba0cd5f9),
        (0xb7dd533054211520, 0xbb442a164ec53a73),
        (0xb7b5895e0b2acd11, 0xbb1663bd8ed78f9c),
    ],
    [
        (0x3c54fa3fb220c497, 0x3fc194eba75b32f9),
        (0x36084c365974fe27, 0xb97351f4ef13fe26),
        (0x3c59f5fdd12caab1, 0xbfb190f7dc27362b),
        (0xbbd96244746b5f38, 0x3f462bb47a5c5f7f),
        (0x3c106edbe0b8c444, 0x3f7756ef20f5d2e2),
        (0x3bb1231bc38d74fa, 0xbf1198b0ba97ecfb),
        (0xbbafcaf470009ca5, 0xbf28be8cf9358d55),
        (0x3b341ae8492fe909, 0x3ec3dd6f7c8cc3c0),
        (0xbb227b6134d4cb1b, 0x3ecc09c80ee7f9af),
        (0xbb0404bf1dcee0f4, 0xbe6728e46a451e32),
        (0xbb0524de73bc4a47, 0xbe63b91113508622),
        (0xba958725ca8bba0a, 0x3e0080fddad62bf8),
        (0xba979be80a70704c, 0x3df2e111e88dae1d),
        (0x3a16d9f11e8a571a, 0xbd8fbae88bdab281),
        (0x3a026df37d7f2f86, 0xbd7a2a4fbea86012),
        (0x39b683e401b5c962, 0x3d15f540e97728a3),
        (0xb95202b2fa6547b4, 0x3cfb74c0b254e833),
        (0xb938463548f90bb3, 0xbc96eb75a8c251ef),
        (0x391ab9d89901dd33, 0xbc76910c0073cf6f),
        (0x38b57304713e6ec7, 0x3c12af37349ec30d),
        (0x388e4a0ca7b6a403, 0x3bedd6cb83761ce6),
        (0xb82fbe9670a0583a, 0xbb88745a179736e3),
        (0x37f9ae92b9c8d8b8, 0xbb60180931dceaf3),
        (0xb78cba499e17a878, 0x3afab192c9241f99),
    ],
    [
        (0x35cbb600d4239a22, 0x39251b64cfa537e4),
        (0x3c6e71c482be67bd, 0xbfc12dd57bf18ada),
        (0xbbe9a8a827c4bbb7, 0x3f5f1e1e7f393e83),
        (0x3c3286f932bea35e, 0x3f96d9afe88301fa),
        (0xbbd360330de30bf2, 0xbf34a538a482979b),
        (0x3bef838ddd50c780, 0xbf52316250b4ae37),
        (0xbb94c13fc989fce7, 0x3ef05f11577b4627),
        (0x3b6771efd22cab1c, 0x3efb86bad42fc220),
        (0x3b1a6ca937a8bc13, 0xbe98a1b3a9e92749),
        (0x3b32d767937947d7, 0xbe983dcaf3f8fcc5),
        (0x3aba8d8d67ac9dda, 0x3e3589a7ca5fdcf1),
        (0x3ac9ad46d29f3be0, 0x3e2be3ee3298bb99),
        (0xba6e973d129d8f1f, 0xbdc8913f1d0ff123),
        (0x3a203a3c6881bd79, 0xbdb695c386660814),
        (0x39fb47463477c5c8, 0x3d53b25d3647586b),
        (0xb98bb89b4c085acf, 0x3d3b20c42e642e99),
        (0x397afedf65fe5bfe, 0xbcd76505329d2147),
        (0x3958669626edb388, 0xbcb91f514b70dc66),
        (0xb8ecf324e9269490, 0x3c55661a888ce787),
        (0xb8b1401ccd948cf2, 0x3c327b48d4454d52),
        (0xb8698f6cf1c5e01c, 0xbbcf10cd4224310e),
        (0x384b17d2f0b22cb7, 0xbba61f092d4fb132),
        (0xb7db3b4d370a111b, 0x3b422f16a2b51a21),
        (0x37ace9fba95b7f40, 0x3b15d89abfbbc58c),
    ],
    [
        (0xbc27736b1f56d6fe, 0xbfc0d0d36473e98c),
        (0xb5eeec74a121164b, 0xb94d417ee84a02e1),
        (0xbc517d3bbb8e77ff, 0x3fb0cda9974abe2b),
        (0xbbe2a43b589bb30c, 0xbf4367f38f201c25),
        (0xbc17d132300354c7, 0xbf7656b75e3c242e),
        (0x3b94b9f5aa61b16a, 0x3f0ed82abf7489f1),
        (0xbbcb0bedfa6ed1b8, 0x3f27b4e5b83eeb36),
        (0x3b0620c0f4833a16, 0xbec171fd0fb670e7),
        (0xbb638b5f7ce177c8, 0xbecae62b4ad017fb),
        (0x3b0b87c8b1ba0970, 0x3e64648495a7b49e),
        (0x3b05b3200be63707, 0x3e62f42a577135ad),
        (0xba91dadfc29e6554, 0xbdfd286e7fa32656),
        (0x3a9d8beeb2cb7298, 0xbdf22dbcbf76a1c5),
        (0x3a2a43a2ea6cef43, 0x3d8c222accc0d332),
        (0xba0327331125a6e3, 0x3d793fc7f6c8caef),
        (0xb9acf6791f2a444c, 0xbd138c7e1f6709fa),
        (0x399fac78a0046126, 0xbcfa8e4efc5ad38f),
        (0x3933d41876c7d322, 0x3c947e808f83bea6),
        (0xb91b783f58eeeb68, 0x3c75e0f0594a5008),
        (0xb86c4cbf1e35e546, 0xbc10c8678c595f1c),
        (0xb87ae60cd98137bf, 0xbbecffab1af4612b),
        (0x382a6f17570f961d, 0x3b86111d5963e0eb),
        (0x37f00fe3e9f49b7c, 0x3b5f5ac4d0ed3683),
        (0x377ece1c4a712e4e, 0xbaf8306e05270fd5),
    ],
    [
        (0xb60d733c7a7e52a7, 0x3970250757e24008),
        (0x3c61a13e2fee5687, 0x3fc076826cc2c191),
        (0xbbcb789ffb6667b2, 0xbf5b62885e0070c6),
        (0x3c35dbe9d7210bbe, 0xbf95e7f53001e4b1),
        (0xbbddb8eb1d2bf603, 0x3f322ebeb8dc2202),
        (0xbbf72618e8704270, 0x3f517444a7a04cd0),
        (0x3b7b01e86b4a3da0, 0xbeece06f1f1fcd7e),
        (0x3b8016817bdb5904, 0xbefa7006e6ad9cfe),
        (0x3b13ab2699f50e0a, 0x3e95c42f02cf15ca),
        (0x3b08b6ecfe623226, 0x3e9750ca5e1366b4),
        (0xbabee952793294d7, 0xbe3314982df7eaa2),
        (0x3abefa05a9bee7a7, 0xbe2aded75306b3b3),
        (0x3a67ece2ba1179c1, 0x3dc5d47847d8ebeb),
        (0xba5c6d6993547389, 0x3db5ccf44d287a22),
        (0x39e36e4501d8e34a, 0xbd518fce3e03f8ff),
        (0xb9a320cde8d9c336, 0xbd3a3d6bcad0c437),
        (0xb976652eddf4a16e, 0x3cd4efbd765ccd7d),
        (0x392c39469b09709e, 0x3cb85a4b163d5fb1),
        (0xb8fcb83bb2002a51, 0xbc5339ddcc1bcec2),
        (0xb8d45b1a219ef1df, 0xbc31f48b401dcae3),
        (0xb86e5c376fb9eb81, 0x3bcc067f698492f8),
        (0x384a2903db3f0c8f, 0x3ba58a055bf542b9),
        (0xb7e2ee806b9979fc, 0xbb4078cceb371d66),
        (0xb7bbbf36698a9adf, 0xbb154fa9d8bb2e05),
    ],
    [
        (0x3c61a6e02553980e, 0x3fc02455675ab6d2),
        (0xb5c5f31d846909a7, 0x3920fd28a7ab34ea),
        (0xbc5a58b3083e7da5, 0xbfb021c155a720df),
        (0xbbea19c1039cb49f, 0x3f412be56fc1449a),
        (0x3c01bad3e8f49c57, 0x3f75749d556ad61c),
        (0xbb75621379d06ca1, 0xbf0b51f1f9bea93e),
        (0x3bc81ccb322d1126, 0xbf26c96a07e236bd),
        (0xbb21804e8cca606d, 0x3ebef3a7abd5ac6b),
        (0xbb6f6eedeef435f9, 0x3ec9e207c257433a),
        (0xbaf0b5ea62a8de24, 0xbe6220b96eef8058),
        (0xbad2fa9552352869, 0xbe624317cb296737),
        (0xba7c0f987894ba0a, 0x3df9fc2f2cd3917f),
        (0xba7936059699e1c2, 0x3df18ae8347e8254),
        (0x3a1573486603ca10, 0xbd892540423f2e30),
        (0xba1e19de8fd09a84, 0xbd78687dcdc2db72),
        (0xb97684a35e3ea9dc, 0x3d1187909104433c),
        (0x39630ebfb454443d, 0x3cf9b8362860195d),
        (0x38c1923b5504cc9f, 0xbc92711ece62b404),
        (0xb903c12865011bca, 0xbc753b340e1ec1f7),
        (0xb8a22b4edd218aaf, 0x3c0e50ed1c32be11),
        (0x387ce89517a6eb64, 0x3bec3274fc0c19bf),
        (0x382cd7af1eec7b55, 0xbb8401a453695584),
        (0xb7f9f56cebafb7ea, 0xbb5e8c7520273be2),
        (0xb794336346140121, 0x3af602279247901e),
    ],
    [
        (0x35eae2ce139b755d, 0x396742f004f756a7),
        (0x3c5d7cc41717159f, 0xbfbfa8b41711c83a),
        (0x3bf6219a48a24bc6, 0x3f5857d3969997d1),
        (0xbc395ccf34fc8573, 0x3f9511c6dadaaa12),
        (0x3bd13cc55aed0677, 0xbf302c289dbdbd4f),
        (0xbbde8aacf938011b, 0xbf50cc2238d229f9),
        (0x3b831f1c6dd9cc15, 0x3ee9b64d5c63668f),
        (0xbb967f7ad298da49, 0x3ef976fb023f0f79),
        (0xbb26429ad31066a5, 0xbe93693ba0b5ba70),
        (0xbb30bc57284d2f2f, 0xbe967b952987350c),
        (0x3adfd4fe0a523682, 0x3e310cb79a2addab),
        (0x3abbd4987c3e4ca2, 0x3e29f2079f8e397f),
        (0xba6750593e1e01e8, 0xbdc38d957eaa53a8),
        (0x3a5878f2c117ca3c, 0xbdb51511e93ba74c),
        (0x39d6e8ba61764a78, 0x3d4f8bb4d9d2e233),
        (0x39d8bdf5fdad66d3, 0x3d396afe82155942),
        (0x39700aa17bd0529e, 0xbcd2dc3c5a2d6062),
        (0x39264d3a8a0c9e0a, 0xbcb7a1c8dec30b78),
        (0xb8fa02e0c4da574e, 0x3c51600c1fbbd4ea),
        (0xb8dd822a2730dcdd, 0x3c3174c6425ae3bb),
        (0x38597c34fcb10a17, 0xbbc969aad5156298),
        (0xb84606c1cf335ef0, 0xbba4fad8aa607bb2),
        (0xb78af3fa0ad5b88d, 0x3b3df9a69e01b8c9),
        (0x37a075924edc374d, 0x3b14caa46ecb07e4),
    ],
    [
        (0x3c50e4250a158a23, 0xbfbf161d0c28b48c),
        (0x3611a74b58e68914, 0x3973ee78b8c079ca),
        (0x3c421e360c4c6fb3, 0x3faf11d837aa6f64),
        (0x3bd18f48c3538dba, 0xbf3eab76da4d07a0),
        (0x3c1ebe3b989625cd, 0xbf74ab329f067aea),
        (0xbb5e80c38d8ec580, 0x3f086ada57bc1c51),
        (0x3bc03960e72796ae, 0x3f25f6e78f11ab9a),
        (0x3b5e771583ebe81e, 0xbebbb271f54c8965),
        (0x3b5cfd56b457272d, 0xbec8f85328c26cb7),
        (0x3b05a90e2d074d12, 0x3e603f82aebdeac1),
        (0x3b0054f935b9aa08, 0x3e61a3010279a195),
        (0x3a5b79af494102ab, 0xbdf75660809cdedf),
        (0x3a8b7a2894daf42e, 0xbdf0f6931774a05d),
        (0x3a13e36d17ff7a01, 0x3d86a2e612670759),
        (0x3a1478f3b027f12a, 0x3d77a2a8e0311432),
        (0x39a89f64fe6a7cac, 0xbd0fa4c8e3cec9ee),
        (0x399791d5ae2f24c9, 0xbcf8f1926f3bf5c9),
        (0xb90779dd9b4f4523, 0x3c90b165b307c656),
        (0x38c874530c0e61a2, 0x3c749fd30e4702fb),
        (0xb8a3d520095f9475, 0xbc0b864b8eb17f2d),
        (0xb887f6b35d01d945, 0xbbeb6fef448f9ec9),
        (0xb82c404eab8fd764, 0x3b8238c2939c0306),
        (0xb7cf45b246f68cc2, 0x3b5dc6a5afbef1d7),
        (0x37879b3e205ae80b, 0xbaf41b15e5934cd1),
    ],
    [
        (0x35e2402b29b452e6, 0xb95669e940f0e62d),
        (0x3c0020b4016594be, 0x3fbe8727daa3daed),
        (0xbbea4d873618607e, 0xbf55d353e2854a37),
        (0x3c3361836c5324f0, 0xbf94524d4813cc25),
        (0xbbb70735fac009ce, 0x3f2d037574e28370),
        (0x3bf7f1d7c0f3582e, 0x3f50356bb747a763),
        (0x3b73015c49ea72dd, 0xbee7156bfccef376),
        (0xbb99523f50a17202, 0xbef896d7dc819faf),
        (0x3b307ed5444cfe33, 0x3e9172c6dadf4149),
        (0xbb334d67f3854d6b, 0x3e95baae8efc2e31),
        (0x3abbd2db893ce2b0, 0xbe2eb347eb4d6941),
        (0x3acc3ffb82d08806, 0xbe291a60a72a20e0),
        (0x3a62e32cc76420f9, 0x3dc1a345a9a6a5f2),
        (0xba511d514fc93073, 0x3db46c56b01906be),
        (0x39eb5948642788cd, 0xbd4c84bb3767683a),
        (0xb9c6dbefdede6e96, 0xbd38a83e6e4c14a6),
        (0xb960551193c229eb, 0x3cd11796d0057ab1),
        (0xb952ad5a14be2d5e, 0x3cb6f5675ceddd1b),
        (0xb8d70078fb2fe673, 0xbc4f936ccfdc26d8),
        (0xb8bf16cd40d79f50, 0xbc30fc2f7536e2fb),
        (0xb82fad0a6a231690, 0x3bc727e83d15ee4b),
        (0x3807b3ed34c03c50, 0x3ba472426379c0dd),
        (0xb7d0fc2af29aaf50, 0xbb3b644d13049235),
        (0xb7a0b4b41cc19800, 0xbb144a95537eaf6c),
    ],
    [
        (0xbc5b4c98f0d3c4c3, 0x3fbe0357c158b119),
        (0xb60d0c75a77e2033, 0xb967816058366b9c),
        (0xbc410072ccb8850d, 0xbfadffc2fc1a91f5),
        (0x3bd769e0c0dcd2d3, 0x3f3b9b82ae07da44),
        (0xbc02d8dbfa1e178d, 0x3f73f64e05320ac6),
        (0x3bafe44f6d4ea208, 0xbf05fe4b66cf19d9),
        (0xbbc540e87c50de38, 0xbf2539518e1b00f5),
        (0xbb397c57bc8091bf, 0x3eb8f8d01c487905),
        (0xbb5e179d1aaec556, 0x3ec825045b97e2dc),
        (0xbad3613aacf5b7ad, 0xbe5d565f3bb61dea),
        (0x3af7612fa1765a21, 0xbe611186586f4f74),
        (0xba999c78eb7ba93a, 0x3df51a669158191c),
        (0x3a8fdeaddc76e697, 0x3df06ef52f6715a9),
        (0xb9f43ddc362f0509, 0xbd848215e95caabe),
        (0xb9e3ce8533da1d00, 0xbd76ec8422019334),
        (0xb977fdbacbb88faa, 0x3d0cbaf5fc1449cc),
        (0xb994adec13653643, 0x3cf8393ffa3d864a),
        (0xb920340fd1f6eaf4, 0xbc8e6232fcacc946),
        (0xb91a15a57d08156b, 0xbc740e69a1eec644),
        (0xb882efe890d1719f, 0x3c091cd31485cd1d),
        (0xb87204ec8daa0e68, 0x3beab83a7b21fd23),
        (0x37ee5c4a3769c7c2, 0xbb80ab43d4957dfa),
        (0x37d0433b7ef63b8f, 0xbb5d0a0f23f73309),
        (0xb76e179e9a438fab, 0x3af2707876868cba),
    ],
    [
        (0x3610ef51183cb002, 0x3972fe6576610a9d),
        (0xbc5cb1f28997ca3a, 0xbfbd8293aa55d18f),
        (0xbbd0e0b711c1383a, 0x3f53b6beb83f2596),
        (0x3c36c091c5e2bd45, 0x3f93a5ccbc12a67b),
        (0x3bc80bb5067d449d, 0xbf2a3765d26aa42b),
        (0x3bc464654b3effa9, 0xbf4f5ab33748c215),
        (0xbb86c3ad3a1d34af, 0x3ee4df6f1c257a5c),
        (0xbb9372510da194bb, 0x3ef7cbd49c315be0),
        (0x3b1dfd4ff2a51cd9, 0xbe8f96098cf07175),
        (0x3b0aec441511ad05, 0xbe950b37dd43531f),
        (0x3aa7073eee76aad1, 0x3e2bd2e6405c605d),
        (0xbaba14df19ca3ca5, 0x3e285530df0d4b70),
        (0xba5940f4b5dbe91a, 0xbdc0029e21930f20),
        (0x3a40c5ac889e279f, 0xbdb3d11aeba731a1),
        (0x39b3628671f6f9fe, 0x3d49ef077e064e5f),
        (0x39cb6ce5407798b1, 0x3d37f3d211d80a08),
        (0xb947488a9811c854, 0xbccf2617ceaee07b),
        (0x39261e70e580a2ad, 0xbcb654785f893e5c),
        (0xb8d177e15b92fb85, 0x3c4cd5d45e8ed84c),
        (0x38da56fe6bb5e9e1, 0x3c308aa980f020fb),
        (0x38655215f7c6e0cd, 0xbbc53213f4bfc10c),
        (0x3846a2e45c9c7024, 0xbba3f087bcfff455),
        (0xb7bf1d6ed9e53a8a, 0x3b3922460f936d70),
        (0x37bd5d3f90200af4, 0x3b13cfffa188113e),
    ],
    [
        (0xbc57ac02118ce034, 0xbfbd0b36e5737458),
        (0x35e5dbf97ada5b8e, 0x39455ca750ec6752),
        (0xbc494ec699987d83, 0x3fad082ce3c6b59b),
        (0x3bdbdb6b7d6ce172, 0xbf3905d00c5e6800),
        (0xbc14d8fddd6666b4, 0xbf7352b073fdac7b),
        (0x3ba8a390ec67540a, 0x3f03f1ccfec2fc88),
        (0xbb8f74934a18e861, 0x3f248d74583834bc),
        (0x3b536c48c738b669, 0xbeb6a9ef0d896bae),
        (0xbb686cf2b10eb75c, 0xbec764d9798d6a80),
        (0xbaf45b48c636943c, 0x3e5aa785d6736f5c),
        (0x3b032808a4ecc777, 0x3e608cae36118cdb),
        (0x3a954183d5d06233, 0xbdf332ddfb39cd01),
        (0xba85b57db0ba4e83, 0xbdefe502ff1a8f08),
        (0x3a0e117f9b9a2106, 0x3d82afc83348eef3),
        (0x3a1f9e55699c4976, 0x3d764468c0a30d64),
        (0xb9800b377972327f, 0xbd0a399e849ce64d),
        (0xb987e7e0201e1a08, 0xbcf78e09771972ad),
        (0xb92e0978fe137b3a, 0x3c8bc9dea78f9166),
        (0x3917220a53df58e4, 0x3c738663c7562da0),
        (0x3893066b32284ca1, 0xbc07042b3fa91e95),
        (0x3880dae944b873e0, 0xbbea0b14ac68d97e),
        (0x3816034e3ccdafa3, 0x3b7e9f946caccd4b),
        (0x37ee935573e4975e, 0x3b5c56e575651cdb),
        (0x378326c82431dace, 0xbaf0f996416a3127),
    ],
    [
        (0xb5cba20478c84ae1, 0xb941c202ce623e76),
        (0xbc49df1f0f8d2107, 0x3fbc96700bf039e2),
        (0x3bd298b7ed2d3ac6, 0xbf51ec0b5de4befe),
        (0xbc26397704521dc0, 0xbf93095734a24496),
        (0xbbbe43e7480a487e, 0x3f27d74e12285cb2),
        (0x3bbff97a4e7b98d7, 0x3f4e636fe259352c),
        (0x3b74153ba07b8f20, 0xbee2fe11972bc0c6),
        (0x3b833d8792db4991, 0xbef712e4d44c4a74),
        (0xbb0f1553b4c20b5d, 0x3e8cc3adabae0452),
        (0x3b2bb2d302a59ebb, 0x3e946ad2d9cbeb5c),
        (0x3abbcd480e0362ce, 0xbe295d81ae83f621),
        (0xbab3ff4398d1910c, 0xbe27a02aefea3d60),
        (0xba59673869d1d69f, 0x3dbd3a949a722395),
        (0xba4dc28fd0eb18e2, 0x3db341e0bb193b48),
        (0xb9c0a6ee415a5306, 0xbd47b550a4f76700),
        (0x3994e75cd4e25783, 0xbd374c654a26537e),
        (0x396bf6b9da90343a, 0x3ccc86173683ecb7),
        (0xb9343315e10774e3, 0x3cb5be2db45fdd99),
        (0xb8dd50a28dc1737f, 0xbc4a7432d3c802d5),
        (0xb89104cb13cbb909, 0xbc301fe8f4a0a59a),
        (0x3862fc2b2f292a7c, 0x3bc37bcc945caecc),
        (0x383f40a66926ee1f, 0x3ba375a2264aec44),
        (0x37876edaf3b8c254, 0xbb3726c4f48ef3c9),
        (0x37bfb858dba2f507, 0xbb135b197250609d),
    ],
    [
        (0x3c2c279ff462c3be, 0x3fbc29ae8400a320),
        (0x35cbc9db3681c2a9, 0x396a5c619f751287),
        (0xbc47ac9bcf3441f8, 0xbfac27138da31c2b),
        (0xbbbe5ab192ad8423, 0x3f36d141fcbea853),
        (0x3c12b724fd73605f, 0x3f72bdc71062acd6),
        (0x3baf5f2a5e20af51, 0xbf0231cf643ffc17),
        (0x3b6e53f73e07f7ec, 0xbf23f0bf3b3fe8be),
        (0xbb4cfe2786564eab, 0x3eb4b05e955de175),
        (0x3b6d19da721e7488, 0x3ec6b52b868fa5e2),
        (0x3afd657f133da2a0, 0xbe585a7aa3e84cc3),
        (0x3afd8305110d0415, 0xbe6012d3384c9164),
        (0x3a82a12fcbf74576, 0x3df18f8c4872544f),
        (0x3a83ddc6dcd20390, 0x3deeffc4029f2f01),
        (0x3a2a5f86153ccacb, 0xbd811d5a3daf7136),
        (0x3a1939ffd2f149a3, 0xbd75a8d84ce122b1),
        (0x39948d45aa4f471b, 0x3d080dfa27f2ad6a),
        (0x397ee5f9b8f0f308, 0x3cf6eec07f0d91c7),
        (0xb92b05e0bbac3b01, 0xbc8987d6f4b58a17),
        (0xb90c392890cae7e7, 0xbc7307194fcc3abc),
        (0xb894aeaf37740080, 0x3c052f0ff8dc5a44),
        (0x3880e377a138ed87, 0x3be96804c6d0c4c3),
        (0xb7fb165b04bc999e, 0xbb7c3d1f829825de),
        (0x37fc727aeddf2fd2, 0xbb5bad09acad456f),
        (0xb74fed0f6e3c7260, 0x3aef5d6f1dc95101),
    ],
    [
        (0xb5e9f9b32cc9c472, 0xb94f1c4dc0fd7ad0),
        (0x3c58fff4515190b5, 0xbfbbbf246914235f),
        (0xbbb59d638f72d376, 0x3f5062daee35411a),
        (0x3c1bef9e896a99ca, 0x3f927a96f174b6d1),
        (0x3bc79c688e87e02d, 0xbf25cdb5dea9c121),
        (0x3bd1f78082ed604a, 0xbf4d818348f98a0f),
        (0xbb8cf93ec0b60def, 0x3ee160aab829409d),
        (0xbb9dfe5ed21cbf06, 0x3ef6698d6ee99eb9),
        (0xbb0db19c59160d9d, 0xbe8a5633d8f0b3bf),
        (0x3b3148c171a390c6, 0xbe93d788d61154a7),
        (0xbabd69ba99ca8de9, 0x3e273ec2ae0084b9),
        (0x3a4e64bbbfd099bd, 0x3e26f958f6235deb),
        (0x3a5eece34fec9144, 0xbdbad0939c43a9f7),
        (0xba3e162f724391e6, 0xbdb2bd56309cf194),
        (0xb9d6bdc444caeccb, 0x3d45c709d717e64d),
        (0x39923c04d3fffb00, 0x3d36b0b8fe7370f7),
        (0x39668435dfe7dada, 0xbcca3cece6ce50aa),
        (0x3928e245f64dcddd, 0xbcb531b157c3eb03),
        (0x38d9c8cd5a159ad6, 0x3c485f346abae02a),
        (0xb8c42e265a4503fd, 0x3c2f771606e9b677),
        (0x3869cf65db30f118, 0xbbc1faf5bae17a4f),
        (0x3843ce2b3bffa99c, 0xbba3015dbdae4304),
        (0x37d01253f07a747a, 0x3b356721e926aabc),
        (0x37ab991b20a5ed5f, 0x3b12ebe23c0e019c),
    ],
    [
        (0xbc4b1bd5a08c4697, 0xbfbb5b8273b75055),
        (0xb5fc5d98089f691c, 0xb9617b443b19e784),
        (0xbc19066393ec8b4b, 0x3fab59418c36a684),
        (0x3bd4c6e4a4b5908a, 0xbf34eafeaa92aa79),
        (0xbc1e8c2463f5dff7, 0xbf7235801af9be44),
        (0x3b9bcf2d229387a5, 0x3f00af9747d0be92),
        (0xbbaf4068b21f5f13, 0x3f23611db0e1566f),
        (0xbb5fa296ae418fd9, 0xbeb2fbe414da1250),
        (0xbb6579a7a7d2b7a9, 0xbec613ccbb9cbe59),
        (0xbafb9e935c5d4727, 0x3e565cf274e84d31),
        (0xbae9f705bc9fc3a8, 0x3e5f452996e3dc2b),
        (0x3a88db40653fa6ad, 0xbdf023f5382da3ad),
        (0xba87d1ea787e96d5, 0xbdee2be24fbad63a),
        (0x39e1374c2f3e53a0, 0x3d7f7ed3740f8d64),
        (0x39f934955bbeefc4, 0x3d75187e998a123d),
        (0x39aeb3c518f70011, 0xbd06293c9e78a94a),
        (0xb980e7fd6e1aba60, 0xbcf65a49785b93de),
        (0xb925d4c892512e3e, 0x3c878dbf031e576d),
        (0xb912bf351c2e7bd2, 0x3c728fde13655c16),
        (0xb8aaed4885e1dc5d, 0xbc0392b9e7067906),
        (0x3814e81144aa2192, 0xbbe8ce75f24e6d2f),
        (0xb81d860830f44f3e, 0x3b7a224d38043629),
        (0xb7ef6d218b7f22e1, 0x3b5b0c2b72afb1a5),
        (0xb7213d15be1c6e9a, 0xbaed12d5bc6b57ab),
    ],
    [
        (0xb61ce6cbc1f04255, 0x397036fa5f6395cb),
        (0xbc5024304247ada4, 0x3fbaf9cb49c4f935),
        (0x3bd43675d81a335c, 0xbf4e1d930b513228),
        (0xbc26b1ae60058494, 0xbf91f7a8fec6eba8),
        (0xbbccaafb65fcb92e, 0x3f240a55310866fc),
        (0xbbe93eb318fd63c7, 0x3f4cb20c812fd3aa),
        (0x3b7bc11f9b0249b7, 0xbedff51953c6b6cc),
        (0x3b7ca6d618312011, 0xbef5cdc48f5d75eb),
        (0xbb1e5d2044572b21, 0x3e883b091952c721),
        (0x3b392f30805656b3, 0x3e934fb685e58ab7),
        (0x3abdef10942d29ba, 0xbe2566fc4369ab71),
        (0x3ac87311849ba940, 0xbe265f0f7de29720),
        (0x3a589755adec28c8, 0x3db8b61e5f9b79f9),
        (0xba46ab4cdf8eab9f, 0x3db24253069b78a8),
        (0xb9d3414dfa14e877, 0xbd441732d722a86f),
        (0xb9c20d9bc0b7d20e, 0xbd361fa985a1652f),
        (0xb962a44c7096efd1, 0x3cc83c1756587290),
        (0xb94098c5fbe9b36b, 0x3cb4ae329744cc53),
        (0xb8e5a6b655add69f, 0xbc468a7f4c14db50),
        (0xb8c263b696f21145, 0xbc2eba469216c470),
        (0x386c7b9219cb3b7a, 0x3bc0a74a6d599cb0),
        (0x383259264f03f94d, 0x3ba2936d1cb0c961),
        (0xb7de00e6c53ea913, 0xbb33da81fd8c724e),
        (0x37a15c8a04c7e7ef, 0xbb128235bc0c47ff),
    ],
    [
        (0x3c5ffacf3e2418f7, 0x3fba9e13a0db6429),
        (0x360c15fcefbc6cf7, 0xb96b692a38880539),
        (0xbc305f5aef32722d, 0xbfaa9c1ca2161b9b),
        (0xbbd5ab98ac975e2e, 0x3f3344a09efdc635),
        (0x3be8bf0e555b3029, 0x3f71b82c430a2381),
        (0xbb93a994567a3c7b, 0xbefebfb97bca01f2),
        (0xbb991369e36b1830, 0xbf22dcdb1bc1d038),
        (0x3b5ea31075526d31, 0x3eb180047f0b79ae),
        (0x3b46ada0a01da18a, 0x3ec57eeeee84d0d0),
        (0xbadb2a1ce7632789, 0xbe54a0c699c8318b),
        (0x3addd02fc0b7c908, 0xbe5e7594e8a2c760),
        (0xba80e5f270287d53, 0x3dedccbbb4c0ba7f),
        (0x3a7ca797a3565ecd, 0x3ded6766337b5c91),
        (0x3a1ab23ec0eb4599, 0xbd7d1a05cea18a69),
        (0x39e1f4b5b52d14cc, 0xbd74922fb50a22bf),
        (0x395024c9150c0f64, 0x3d047fa35bf6ab34),
        (0x3985f2bf77091074, 0x3cf5cfa08701ca2f),
        (0xb9266c2a76db875b, 0xbc85d0099455bd99),
        (0xb91f43baef384b49, 0xbc72200ae9780072),
        (0xb8a3a3083fb87713, 0x3c02265fdf769ade),
        (0x38836fad8505e902, 0x3be83dc86da8bc25),
        (0x37ea0aade21b140f, 0xbb78447b5e2dbf2f),
        (0xb7f62e5f79c1ca9c, 0xbb5a73de1878cbf9),
        (0x378acaea6a6d9658, 0x3aeb08df74d22d98),
    ],
    [
        (0xb5d780da95d79b82, 0x395404a4367c1acf),
        (0xbc55d35a88f1e0a3, 0xbfba4407e04298d1),
        (0x3bdf6dddc07aba4f, 0x3f4bcc9df0cf00b2),
        (0xbc3c3cb8ccc39d2a, 0x3f917f0266db2149),
        (0x3ba3d974ad41d7db, 0xbf2280a052234a05),
        (0x3bb902b66919d00f, 0xbf4bf2ada1f44071),
        (0xbb64ad3ffb27706b, 0x3edd83d58032b48d),
        (0x3b3d6efc3f5aa589, 0x3ef53dd972d8f232),
        (0xbb230bc40c1510e8, 0xbe8663c1fe202028),
        (0xbb26dba834dbae96, 0xbe92d1fbf2203ff6),
        (0x3ac7c87693a0820a, 0x3e23c9f0b759c5f9),
        (0x3ac4688bb194b145, 0x3e25cfe1b012696d),
        (0xba597597731528c4, 0xbdb6ddc0795781e2),
        (0x3a30fd07b7c74b96, 0xbdb1cfd495042669),
        (0xb9e56d22c0bed868, 0x3d429b7af52144d6),
        (0x39c7617697a68dba, 0x3d3598302ab3a84e),
        (0xb969ff73c4a2ed6f, 0xbcc677f6a5ebe777),
        (0x3900de58e64b147d, 0xbcb432ecc1192f3c),
        (0x38ee92d01dc42743, 0x3c44ec1126d1a78e),
        (0x38c2b1157c94ede8, 0x3c2e0887dbe88798),
        (0xb7d2d49d120bbb3a, 0xbbbef4037751e14b),
        (0x3837430b053ee391, 0xbba22b7609bcdbcb),
        (0x37d8cec91a9bdbe3, 0x3b327988001f32dc),
        (0x376c8d2e6d2182d4, 0x3b121ddd67c06d53),
    ],
    [
        (0x3c423404089aea02, 0xbfb9ef3bb2213b0b),
        (0xb5c87de2f8cebd53, 0xb962e8408ad4e1a5),
        (0x3c342c5cb51f294f, 0x3fa9ed82007a9a45),
        (0xbbad41b75b8ea526, 0xbf31d2fdeeb29f8a),
        (0x3c18994595321a31, 0xbf71446866ff1b83),
        (0x3b94bf0ff8b9b23d, 0x3efc73b684f93259),
        (0xbbc79edfbf5325fb, 0x3f22628de594b6c9),
        (0xbb2befa01f9526e8, 0xbeb03303c1427449),
        (0x3b691967f42c542c, 0xbec4f51007c51087),
        (0x3af5410acbffa9a8, 0x3e531adfa36f2213),
        (0xbafc56975a6a16e6, 0x3e5db4f306b19095),
        (0x3a79c0979a0204dd, 0xbdeb9e33598f899e),
        (0xba765a4f110b219b, 0xbdecb09ed6ecb892),
        (0xb9f410899a06c92c, 0x3d7afe1183148af8),
        (0x39f78f1dd4b75a7f, 0x3d7414e442aa864a),
        (0xb99a9b988c884db4, 0xbd0307c12941fdc5),
        (0xb990c58e388b6028, 0xbcf54ddabc97b3bb),
        (0x390b34a178291a11, 0x3c84455c63029495),
        (0x391979fc805b19bd, 0x3c71b7024cd44274),
        (0x389e56ba399caf6e, 0xbc00e2d078dd0efb),
        (0xb82f4378acefe136, 0xbbe7b55bb497c4cb),
        (0x381a7fc1f4e5a681, 0x3b769ad969d0625b),
        (0xb7931840ec89fbc9, 0x3b59e3a726e2743d),
        (0x378455d827026a82, 0xbae936884740c735),
    ],
    [
        (0x35fae2734bb315aa, 0x3968400e2d1b8167),
        (0x3c5728ab934a269f, 0x3fb99be744018c90),
        (0xbbdb8852614fe955, 0xbf49c3f52a2af724),
        (0xbc2f281d89ca125d, 0xbf910f5ca51f98b0),
        (0xbbc3c6681ad02312, 0x3f2126c8e8ca2766),
        (0x3babb471b2a0c061, 0x3f4b416f7d4fc313),
        (0xbb677b339729278d, 0xbedb5e2e5580e1ce),
        (0xbb988d465ffb2680, 0xbef4b862279de756),
        (0xbb2b440cbefcb86d, 0x3e84c5071b39dc13),
        (0xbb3de6305e87339f, 0x3e925d2fc3b19021),
        (0x3ac90d431ee8f3e6, 0xbe225df322279972),
        (0xbac3c7a2f5c16d4a, 0xbe254a971eb6fe3b),
        (0x3a515ee7ac48908a, 0x3db53cc6c9922f25),
        (0x3a5ec538d819665b, 0x3db164f95180d8bf),
        (0x39d2657db4413c0c, 0xbd414b9ef404e359),
        (0x39d5a070336f534a, 0xbd3519623b4e4c39),
        (0x396e222733381912, 0x3cc4e7271a5acddf),
        (0xb94d7cdd22f0028c, 0x3cb3bf2a1b7bb45e),
        (0xb8e550d76b23a690, 0xbc437bc0b629d672),
        (0x38b920c14a99cb5c, 0xbc2d61049bf3447b),
        (0x3855956d27abf79f, 0x3bbcdb091753dc1f),
        (0x384b894b04dc498b, 0x3ba1c91964954692),
        (0x37d5bcb82226b372, 0xbb313e10f59f02a2),
        (0x37bd81d4f7329d91, 0xbb11be93ca114b15),
    ],
    [
        (0xbc54096ec8637e04, 0x3fb94d3276914e51),
        (0x35b1f799a74fb76a, 0x395d690fadf22a83),
        (0x3c4baee1d6c0d48a, 0xbfa94bac1950e319),
        (0x3bdbfe3ad50b9727, 0x3f308d4ff8f2059e),
        (0xbc1cca1c2772d45e, 0x3f70d90d29bfeecd),
        (0xbb987fee6366d20f, 0xbefa6d56162f7fb4),
        (0xbbc7e7232a191e61, 0xbf21f107da23807d),
        (0x3b42cbe0cc45eb3d, 0x3eae1a626277437b),
        (0x3b665f948e1d7ee0, 0x3ec474eafd0cc642),
        (0x3acb746c7398995a, 0xbe51c27144f42d78),
        (0x3af77d2dd1ae1ba3, 0xbe5d01999b1a4fe9),
        (0x3a77ecdd37e45371, 0x3de9b014801da87a),
        (0x3a855ec04282c8cc, 0x3dec061740bc8a41),
        (0x3a1fc05e7d7c7991, 0xbd791f8c297254d7),
        (0x3a0ace8e68225aa3, 0xbd739fb558b7a363),
        (0x398b465ba48d7de4, 0x3d01b9f74d482556),
        (0x399084cac0e412c2, 0x3cf4d425556e6c52),
        (0xb91cfaa8a45d49ae, 0xbc82e617aee470c1),
        (0xb91bd25d6d983a31, 0xbc71543271ff906d),
        (0xb89586b26f4fbcba, 0x3bff8440e63c211d),
        (0x386772bb21a29d94, 0x3be734945f8177cb),
        (0xb80904837292f3d9, 0xbb751e13261a9dc1),
        (0x37f62a65371653ca, 0xbb595b06b5ab1c41),
        (0x378aee2000820cbf, 0x3ae79434d2d5aa8d),
    ],
    [
        (0x35fd3e35d9fc508f, 0x39692a03d61a6765),
        (0x3c5e213a1a4b3671, 0xbfb8ffc9bd24fe08),
        (0x3be5e8a5d70cddcc, 0x3f47f7d46ab33721),
        (0x3c2f532ddb23da23, 0x3f90a7a725d3fbc4),
        (0xbbb3f089aa77a72c, 0xbf1fea1728f216b4),
        (0xbbdf63ef331627c3, 0xbf4a9cac69f0ed64),
        (0x3b78f775a8392238, 0x3ed977f48ff1056b),
        (0x3b8a77447708dbe0, 0x3ef43c2d8e698c10),
        (0x3b293fa288b304e6, 0xbe8355d1a6765ea6),
        (0x3b17742d486c25ae, 0xbe91f0553501d121),
        (0xba8ae50dffeaa9a9, 0x3e211b47f6a44829),
        (0xbaa73bc70607579c, 0x3e24ce23303889a9),
        (0x3a5d5eb483ce8058, 0xbdb3ca98df62221a),
        (0x3a46a518fdfd8131, 0xbdb100fc746529d6),
        (0x39e722b8c5f7aad4, 0x3d4020f11e2dc07c),
        (0xb9d9956b0c21d835, 0x3d34a26ef221978f),
        (0x396850ab1f7d4d53, 0xbcc38203ff3d18df),
        (0x3945a4bf6ed3a626, 0xbcb35244c14991ae),
        (0xb8eeb070694cf5e7, 0x3c4232db9b60cb00),
        (0xb8a56af95593329b, 0x3c2cc2f2efd1e051),
        (0xb817a34eb41964e7, 0xbbbafa643c1fee7f),
        (0x38319c576c8452aa, 0xbba16bf81ee3b787),
        (0x37d076f627ec3422, 0x3b3022fbfe27424b),
        (0xb7a801683d487848, 0x3b116413b0cae3c2),
    ],
    [
        (0x3c4123b2f0e7c9dd, 0xbfb8b67a2481077d),
        (0xb5edd5946b3cb2a5, 0x3946a50da827f406),
        (0xbc4b4ca91be60c39, 0x3fa8b51f21068ea2),
        (0x3bb25eeb0c76260b, 0xbf2ed935c7aefa31),
        (0xbbf1e976299c9c65, 0xbf707522a5037f2d),
        (0x3b9aa02b2a63445a, 0x3ef8a196061f8bbc),
        (0x3bc503ddc4900b75, 0x3f21874a47e3c1e3),
        (0x3b255f883f2d82a2, 0xbeac10cf34c04f17),
        (0xbb5b1d29b71073a0, 0xbec3fd6c2d4fa2a4),
        (0xbaeb8c54608e9282, 0x3e50906d55522785),
        (0xbac0c29c0a96e3da, 0x3e5c5a1c124dfa08),
        (0xba7aa4d02da2e1f4, 0xbde7f883b31a5f59),
        (0xba7747971035692d, 0xbdeb668cf53028e0),
        (0x3a1b938795c32e6d, 0x3d7775372d05b2b0),
        (0xba0cfec68b9fa162, 0x3d7331d871d67118),
        (0xb9ae3aa9398e5a58, 0xbd0090102489f480),
        (0xb99e9cec5597a630, 0xbcf461c4066d0e4a),
        (0x3914deb7f36c5b05, 0x3c81abf9941a3423),
        (0xb8df6670bd950fcb, 0x3c70f715e31f4f53),
        (0xb88076dcbbb6bbc6, 0xbbfd7ed45d36754d),
        (0x3884befa4b49ff70, 0xbbe6badf59009f89),
        (0xb817484bd8d9cd12, 0x3b73c80adb5c7d1b),
        (0x37e6bcd3e344ffce, 0x3b58d97e797244fc),
        (0x377002e0099ec551, 0xbae61ba441542113),
    ],
    [
        (0xb5e15739920a6042, 0x3957411605b9cd8d),
        (0x3c5b1c9821974147, 0x3fb86e51be0a9153),
        (0xbbeaa494385da6b8, 0xbf465ed1b387e5da),
        (0x3c09cfc1363fac8e, 0xbf9046fc5a218a86),
        (0x3bbf17fc5592840d, 0x3f1dca617fefa913),
        (0xbbe411ddd3f3c7e1, 0x3f4a0300221528a7),
        (0xbb35922b096464c9, 0xbed7c7618906f1e2),
        (0x3b985c37a80f1f32, 0xbef3c838897d0a1e),
        (0xbb1cfd1e42294da1, 0x3e820ede9f9dd7dd),
        (0x3b3d337f0c931d76, 0x3e918a94165592bb),
        (0x3aba2064c2ef620a, 0xbe1ff76205118f09),
        (0x3ab62166ef13ed4e, 0xbe24599dfeef01a1),
        (0xba4104861aca6ff9, 0x3db2803e5998312f),
        (0x3a5eb79000c7671a, 0x3db0a33202feb041),
        (0x39ad6a6495675c7f, 0xbd3e2bff73866f0b),
        (0x39d02f8782470b05, 0xbd34329cf32bc6b1),
        (0xb94c1fab2f5d49d8, 0x3cc2424890e7103d),
        (0x395165f2cbeeb591, 0x3cb2eba6575a58fb),
        (0xb8ee488b45b19302, 0xbc410bda2b2fedfa),
        (0x38c169197f6744f6, 0xbc2c2d96bb6d554b),
        (0xb839df5ffc139231, 0x3bb94a6157935d73),
        (0x38438133f79f8aef, 0x3ba113b61477f5c4),
        (0xb7c400b9f75f31c1, 0xbb2e47f731863fb7),
        (0xb7bf6daddcc30386, 0xbb110e1251b8eda4),
    ],
    [
        (0x3c48caabfef07d2b, 0x3fb829d06fee9266),
        (0xb61c7c2ce51dc943, 0xb97019a7cd7017ec),
        (0xbc46f7f24522358e, 0xbfa8289a526d7785),
        (0xbbbd55f5936024f5, 0x3f2cd680355c9eb6),
        (0x3c19dafafbc6423c, 0x3f7017d70f512861),
        (0x3b94ef9247fbd18a, 0xbef707978e2a0db8),
        (0xbbb84f6dc795e259, 0xbf21247ce15e7385),
        (0x3b3c154e3f91cb1f, 0x3eaa3f6125485ec1),
        (0xbb6485e10f66ec36, 0x3ec38da848401be9),
        (0xbae822e833951880, 0xbe4efe39d8db4cd8),
        (0xbaf0a59f02f9da6b, 0xbe5bbd40f1db0e94),
        (0x3a88e5731b6d2f4f, 0x3de66f7d49436f83),
        (0xba3e79da115ba52a, 0x3dead0e8148229af),
        (0xb999a0303d870988, 0xbd75f78595133d59),
        (0x39d279bb13e85151, 0xbd72ca9bb1031b0b),
        (0x398dc73294489cd5, 0x3cff09e49c89467c),
        (0xb98c493ba49ccabf, 0x3cf3f60eeb57dab2),
        (0xb928107dd8db2eda, 0xbc8091d68a71f568),
        (0x391a43591422376a, 0xbc709f3347db6cf2),
        (0xb88bd7e4b3dfccd4, 0x3bfbad357dd86bf0),
        (0x386a0a018ebdb436, 0x3be647b364d0856e),
        (0x381f08d09391197b, 0xbb72939f68459b07),
        (0xb7ea807173f17da8, 0xbb585e93fb40bea8),
        (0xb78b806a2be014c6, 0x3ae4c7339928aa1f),
    ],
    [
        (0xb5dba30ce3053693, 0x3937ea2409abe46b),
        (0x3c21907f595a082a, 0xbfb7e656efb009ae),
        (0x3bea53822a52cdff, 0x3f44f15066f3d876),
        (0xbbef8f1c97361f96, 0x3f8fd932c26aad94),
        (0xbba4e6bbdef5d491, 0xbf1be460dd86a0a4),
        (0x3bdda15b3ef145a9, 0xbf49733b591879f8),
        (0x3b7b0e61e74c413c, 0x3ed64488c56022e0),
        (0xbb74588b74c5876f, 0x3ef35ba58bf2f993),
        (0xbb2d010d38a866aa, 0xbe80ea47bceb9a8f),
        (0xbb1839485029ea9f, 0xbe912b327055d0e9),
        (0xbab04b71fdf73b8a, 0x3e1df42fc4e2482c),
        (0xba732a2fe61d6156, 0x3e23ec3e76876dba),
        (0xba511e7db88fce7c, 0xbdb1580393f473be),
        (0xba5b5424c61c9156, 0xbdb04b0353c35933),
        (0xb9c9856b07421c8e, 0x3d3c4ca30ffe1933),
        (0x39c738eb74a7f61d, 0x3d33c947b936be26),
        (0xb9400509632244dc, 0xbcc122c73f63e006),
        (0x3916d7eae173a1b3, 0xbcb28ac72945c28e),
        (0xb8e73a0527a29464, 0x3c4002241bb36c01),
        (0x38cad2069d44143f, 0x3c2ba04276d6d12c),
        (0x385b64b543fd96d2, 0xbbb7c48f7753a6e6),
        (0xb848072abf17cd35, 0xbba0bffbbb75a502),
        (0xb7a49b7af751a348, 0x3b2c7ae054639d4d),
        (0x37b54394c52e23e8, 0x3b10bc481d251ae6),
    ],
    [
        (0xbc2f952341a4610c, 0xbfb7a62320798175),
        (0x3608d7aa0df33a13, 0x396cf7a4a97dd69b),
        (0x3c4f5aadff868840, 0x3fa7a50ca4504bb8),
        (0xbbc425ed1e39553c, 0xbf2b095ccb50a68c),
        (0x3c0088e41716f14b, 0xbf6f80ef11daa37a),
        (0xbb8b1b49b7c35e54, 0x3ef59822dc75b064),
        (0x3bc22455d3c1fa15, 0x3f20c7e6a7c66630),
        (0x3b4cb7a0033a7704, 0xbea89e00b5c358d0),
        (0x3b4c5ee49818e61f, 0xbec324d5238b26d0),
        (0x3ae1d17d1c1f9459, 0x3e4d13a888d5dd30),
        (0xbac771ff3649d30b, 0x3e5b29f941a95b07),
        (0xba65c3a8faacb7ca, 0xbde50e6ebb3d1df7),
        (0xba5e1fa2b117348a, 0xbdea4434c6b929d9),
        (0x3a1aa33d3910c73c, 0x3d74a03fe2c47ae3),
        (0x39fc6f3d0c6f0177, 0x3d7269628f2df267),
        (0x3998d07708e4c6bf, 0xbcfd28cc08f5cbae),
        (0xb95557f1f159116a, 0xbcf390706324d249),
        (0x3901d36b70d2e2f4, 0x3c7f26c3f4d4f4a6),
        (0x38f56e280758755e, 0x3c704c1cbbe93d3e),
        (0xb86ceb268d238b88, 0xbbfa089cde4d22c5),
        (0xb889788b18499f22, 0xbbe5da907334a8cc),
        (0x38130abe343bca1e, 0x3b7181ee2ddebf21),
        (0x37f04d8f909b3d76, 0x3b57e93ad9acbcc3),
        (0x378249417d24489f, 0xbae648d6e71bec37),
    ],
];
