// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_dq.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_dq__A4__F549

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr std::string_view PACKAGE_PATH { R"xyzw(amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_256_0_T_F_1___gfx120x)xyzw" };
constexpr std::string_view FUNC_NAME { R"xyzw(＊fp32@16_256_0_T_F_1)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx1200)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0x298021ddu, 0xb554184bu, 0, 126 }, // 298021ddb554184b = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_256_0_T_F_1___gfx120x__P__16_16__CO__wave1_warp2_stg1--Arch_gfx1200
    { 0x54422b5fu, 0x45301126u, 0, 6 }, // 54422b5f45301126 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_256_0_T_F_1___gfx120x__P__16_16__CO__wave2_warp2_stg1--Arch_gfx1200
    { 0xbadcd7a7u, 0xa865898eu, 0, 46 }, // badcd7a7a865898e = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_256_0_T_F_1___gfx120x__P__16_16__CO__wave3_warp2_stg1--Arch_gfx1200
    { 0x98bc78a2u, 0xff829979u, 0, 23 }, // 98bc78a2ff829979 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_256_0_T_F_1___gfx120x__P__16_16__CO__wave4_warp2_stg1--Arch_gfx1200
    { 0x030eb520u, 0x2c6fdf84u, 40, 155 }, // 030eb5202c6fdf84 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_256_0_T_F_1___gfx120x__P__32_16__CO__wave1_warp4_stg1--Arch_gfx1200
    { 0x18739f1fu, 0xdadba2ceu, 40, 86 }, // 18739f1fdadba2ce = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_256_0_T_F_1___gfx120x__P__32_16__CO__wave2_warp4_stg1--Arch_gfx1200
    { 0x63f54dd0u, 0x91afb70fu, 40, 103 }, // 63f54dd091afb70f = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_256_0_T_F_1___gfx120x__P__32_16__CO__wave3_warp4_stg1--Arch_gfx1200
    { 0x02dd3e4fu, 0x0ab4122eu, 40, 63 }, // 02dd3e4f0ab4122e = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_256_0_T_F_1___gfx120x__P__32_16__CO__wave4_warp4_stg1--Arch_gfx1200
    { 0x513036fau, 0x48cc458bu, 80, 155 }, // 513036fa48cc458b = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_256_0_T_F_1___gfx120x__P__32_32__CO__wave1_warp4_stg1--Arch_gfx1200
    { 0xcbe5d48cu, 0xf7643290u, 80, 103 }, // cbe5d48cf7643290 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_256_0_T_F_1___gfx120x__P__32_32__CO__wave3_warp4_stg1--Arch_gfx1200
    { 0x127b9683u, 0x5318ab3cu, 80, 63 }, // 127b96835318ab3c = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp32@16_256_0_T_F_1___gfx120x__P__32_32__CO__wave4_warp4_stg1--Arch_gfx1200
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_dq_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_dq_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx1200_mod0
{{ 3, 2, 3, 3, 3, 3, 3, 3, 3, 3},
 { 3, 2, 3, 4, 7, 4, 7, 6, 4, 7},
 { 0, 0, 0, 3, 3, 3, 3, 3, 3, 3},
 { 3, 0, 1, 3, 3, 3, 3, 3, 8, 9},
 { 3, 3, 2, 3, 3, 3, 3, 8,10,10},
 { 2, 0, 3, 2, 3, 2, 7, 5, 5, 4},
 { 2, 2, 3, 3, 2, 0, 0, 3, 2, 1},
 { 3, 3, 3, 2, 2, 2, 3, 3, 1, 2},
 { 2, 2, 1, 3, 2, 3, 2, 2, 3, 2},
 { 3, 2, 3, 2, 3, 2, 2, 3, 0, 0}}
// End of GPU gfx1200_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelDqContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelDqContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_dq__A4__F549 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 3;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_dq__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 3;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*fp32:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = "fp32"
// philox_seed_ptr = "*u64"
// philox_offset1 = "*u64"
// philox_offset2 = "u64"
// Window_left = 0
// BLOCK_DMODEL = 256
// CAUSAL_TYPE = 0
// ENABLE_DROPOUT = True
// PADDED_HEAD = False
// BIAS_TYPE = 1

// vim: set fileencoding=utf-8

