/* src/bazaar.p/bz-hook.c
 *
 * Copyright 2026 AUTOGEN
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 * SPDX-License-Identifier: GPL-3.0-or-later
 */

#include "src/bazaar.p/bz-hook.h"

G_DEFINE_ENUM_TYPE (
    BzHookSignal,
    bz_hook_signal,
    G_DEFINE_ENUM_VALUE (BZ_HOOK_SIGNAL_BEFORE_TRANSACTION, "before-transaction"),
    G_DEFINE_ENUM_VALUE (BZ_HOOK_SIGNAL_AFTER_TRANSACTION, "after-transaction"));

G_DEFINE_ENUM_TYPE (
    BzHookReturnStatus,
    bz_hook_return_status,
    G_DEFINE_ENUM_VALUE (BZ_HOOK_RETURN_STATUS_CONTINUE, "continue"),
    G_DEFINE_ENUM_VALUE (BZ_HOOK_RETURN_STATUS_STOP, "stop"),
    G_DEFINE_ENUM_VALUE (BZ_HOOK_RETURN_STATUS_CONFIRM, "confirm"),
    G_DEFINE_ENUM_VALUE (BZ_HOOK_RETURN_STATUS_DENY, "deny"));

G_DEFINE_ENUM_TYPE (
    BzHookTransactionType,
    bz_hook_transaction_type,
    G_DEFINE_ENUM_VALUE (BZ_HOOK_TRANSACTION_TYPE_INSTALL, "install"),
    G_DEFINE_ENUM_VALUE (BZ_HOOK_TRANSACTION_TYPE_UPDATE, "update"),
    G_DEFINE_ENUM_VALUE (BZ_HOOK_TRANSACTION_TYPE_REMOVAL, "removal"));

struct _BzHook
{
  GObject parent_instance;

  char *id;
  BzHookSignal when;
  GListModel *dialogs;
  char *shell;
};

G_DEFINE_FINAL_TYPE (BzHook, bz_hook, G_TYPE_OBJECT);

enum
{
  PROP_0,

  PROP_ID,
  PROP_WHEN,
  PROP_DIALOGS,
  PROP_SHELL,

  LAST_PROP
};
static GParamSpec *props[LAST_PROP] = { 0 };

static void
bz_hook_dispose (GObject *object)
{
  BzHook *self = BZ_HOOK (object);

  g_clear_pointer (&self->id, g_free);
  g_clear_pointer (&self->dialogs, g_object_unref);
  g_clear_pointer (&self->shell, g_free);

  G_OBJECT_CLASS (bz_hook_parent_class)->dispose (object);
}

static void
bz_hook_get_property (GObject    *object,
                       guint       prop_id,
                       GValue     *value,
                       GParamSpec *pspec)
{
  BzHook *self = BZ_HOOK (object);

  switch (prop_id)
    {
    case PROP_ID:
      g_value_set_string (value, bz_hook_get_id (self));
      break;
    case PROP_WHEN:
      g_value_set_enum (value, bz_hook_get_when (self));
      break;
    case PROP_DIALOGS:
      g_value_set_object (value, bz_hook_get_dialogs (self));
      break;
    case PROP_SHELL:
      g_value_set_string (value, bz_hook_get_shell (self));
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
    }
}

static void
bz_hook_set_property (GObject      *object,
                       guint         prop_id,
                       const GValue *value,
                       GParamSpec   *pspec)
{
  BzHook *self = BZ_HOOK (object);

  switch (prop_id)
    {
    case PROP_ID:
      bz_hook_set_id (self, g_value_get_string (value));
      break;
    case PROP_WHEN:
      bz_hook_set_when (self, g_value_get_enum (value));
      break;
    case PROP_DIALOGS:
      bz_hook_set_dialogs (self, g_value_get_object (value));
      break;
    case PROP_SHELL:
      bz_hook_set_shell (self, g_value_get_string (value));
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
    }
}

static void
bz_hook_class_init (BzHookClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  g_type_ensure (BZ_TYPE_HOOK_DIALOG);
  g_type_ensure (BZ_TYPE_HOOK_DIALOG_OPTION);

  object_class->set_property = bz_hook_set_property;
  object_class->get_property = bz_hook_get_property;
  object_class->dispose      = bz_hook_dispose;

  props[PROP_ID] =
      g_param_spec_string (
          "id",
          NULL, NULL, NULL,
          G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS | G_PARAM_EXPLICIT_NOTIFY);

  props[PROP_WHEN] =
      g_param_spec_enum (
          "when",
          NULL, NULL,
          BZ_TYPE_HOOK_SIGNAL, 0,
          G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS | G_PARAM_EXPLICIT_NOTIFY);

  props[PROP_DIALOGS] =
      g_param_spec_object (
          "dialogs",
          NULL, NULL,
          G_TYPE_LIST_MODEL,
          G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS | G_PARAM_EXPLICIT_NOTIFY);

  props[PROP_SHELL] =
      g_param_spec_string (
          "shell",
          NULL, NULL, NULL,
          G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS | G_PARAM_EXPLICIT_NOTIFY);

  g_object_class_install_properties (object_class, LAST_PROP, props);
}

static void
bz_hook_init (BzHook *self)
{
}

BzHook *
bz_hook_new (void){
  return g_object_new (BZ_TYPE_HOOK, NULL);
}

const char *
bz_hook_get_id (BzHook *self){
  g_return_val_if_fail (BZ_IS_HOOK (self), NULL);
  return self->id;
}

BzHookSignal
bz_hook_get_when (BzHook *self){
  g_return_val_if_fail (BZ_IS_HOOK (self), 0);
  return self->when;
}

GListModel *
bz_hook_get_dialogs (BzHook *self){
  g_return_val_if_fail (BZ_IS_HOOK (self), NULL);
  return self->dialogs;
}

const char *
bz_hook_get_shell (BzHook *self){
  g_return_val_if_fail (BZ_IS_HOOK (self), NULL);
  return self->shell;
}

void
bz_hook_set_id (BzHook *self,
    const char *id){
  g_return_if_fail (BZ_IS_HOOK (self));

  if (id == self->id || (id != NULL && self->id != NULL && g_strcmp0(id, self->id) == 0))
    return;

  g_clear_pointer (&self->id, g_free);
  if (id != NULL)
    self->id = g_strdup (id);

  g_object_notify_by_pspec (G_OBJECT (self), props[PROP_ID]);
}

void
bz_hook_set_when (BzHook *self,
    BzHookSignal when){
  g_return_if_fail (BZ_IS_HOOK (self));

  if (when == self->when)
    return;

  self->when = when;

  g_object_notify_by_pspec (G_OBJECT (self), props[PROP_WHEN]);
}

void
bz_hook_set_dialogs (BzHook *self,
    GListModel *dialogs){
  g_return_if_fail (BZ_IS_HOOK (self));

  if (dialogs == self->dialogs)
    return;

  g_clear_pointer (&self->dialogs, g_object_unref);
  if (dialogs != NULL)
    self->dialogs = g_object_ref (dialogs);

  g_object_notify_by_pspec (G_OBJECT (self), props[PROP_DIALOGS]);
}

void
bz_hook_set_shell (BzHook *self,
    const char *shell){
  g_return_if_fail (BZ_IS_HOOK (self));

  if (shell == self->shell || (shell != NULL && self->shell != NULL && g_strcmp0(shell, self->shell) == 0))
    return;

  g_clear_pointer (&self->shell, g_free);
  if (shell != NULL)
    self->shell = g_strdup (shell);

  g_object_notify_by_pspec (G_OBJECT (self), props[PROP_SHELL]);
}

/* End of src/bazaar.p/bz-hook.c */
