/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * vim: set ts=8 sts=2 et sw=2 tw=80:
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

#ifndef builtin_intl_Collator_h
#define builtin_intl_Collator_h

#include <stdint.h>

#include "builtin/SelfHostingDefines.h"
#include "js/Class.h"
#include "vm/NativeObject.h"

namespace mozilla::intl {
class Collator;
}

namespace js {

/******************** Collator ********************/

class CollatorObject : public NativeObject {
 public:
  static const JSClass class_;
  static const JSClass& protoClass_;

  static constexpr uint32_t INTERNALS_SLOT = 0;
  static constexpr uint32_t INTL_COLLATOR_SLOT = 1;
  static constexpr uint32_t SLOT_COUNT = 2;

  static_assert(INTERNALS_SLOT == INTL_INTERNALS_OBJECT_SLOT,
                "INTERNALS_SLOT must match self-hosting define for internals "
                "object slot");

  // Estimated memory use for UCollator (see IcuMemoryUsage).
  static constexpr size_t EstimatedMemoryUse = 1128;

  mozilla::intl::Collator* getCollator() const {
    const auto& slot = getFixedSlot(INTL_COLLATOR_SLOT);
    if (slot.isUndefined()) {
      return nullptr;
    }
    return static_cast<mozilla::intl::Collator*>(slot.toPrivate());
  }

  void setCollator(mozilla::intl::Collator* collator) {
    setFixedSlot(INTL_COLLATOR_SLOT, PrivateValue(collator));
  }

 private:
  static const JSClassOps classOps_;
  static const ClassSpec classSpec_;

  static void finalize(JS::GCContext* gcx, JSObject* obj);
};

/**
 * Returns an array with the collation type identifiers per Unicode
 * Technical Standard 35, Unicode Locale Data Markup Language, for the
 * collations supported for the given locale. "standard" and "search" are
 * excluded.
 *
 * Usage: collations = intl_availableCollations(locale)
 */
[[nodiscard]] extern bool intl_availableCollations(JSContext* cx, unsigned argc,
                                                   JS::Value* vp);

/**
 * Compares x and y (which must be String values), and returns a number less
 * than 0 if x < y, 0 if x = y, or a number greater than 0 if x > y according
 * to the sort order for the locale and collation options of the given
 * Collator.
 *
 * Spec: ECMAScript Internationalization API Specification, 10.3.2.
 *
 * Usage: result = intl_CompareStrings(collator, x, y)
 */
[[nodiscard]] extern bool intl_CompareStrings(JSContext* cx, unsigned argc,
                                              JS::Value* vp);

/**
 * Returns true if the given locale sorts upper-case before lower-case
 * characters.
 *
 * Usage: result = intl_isUpperCaseFirst(locale)
 */
[[nodiscard]] extern bool intl_isUpperCaseFirst(JSContext* cx, unsigned argc,
                                                JS::Value* vp);

/**
 * Returns true if the given locale ignores punctuation by default.
 *
 * Usage: result = intl_isIgnorePunctuation(locale)
 */
[[nodiscard]] extern bool intl_isIgnorePunctuation(JSContext* cx, unsigned argc,
                                                   JS::Value* vp);

namespace intl {

/**
 * Returns a new instance of the standard built-in Collator constructor.
 */
[[nodiscard]] extern CollatorObject* CreateCollator(
    JSContext* cx, JS::Handle<JS::Value> locales,
    JS::Handle<JS::Value> options);

/**
 * Returns a possibly cached instance of the standard built-in Collator
 * constructor.
 */
[[nodiscard]] extern CollatorObject* GetOrCreateCollator(
    JSContext* cx, JS::Handle<JS::Value> locales,
    JS::Handle<JS::Value> options);

[[nodiscard]] extern bool CompareStrings(JSContext* cx,
                                         JS::Handle<CollatorObject*> collator,
                                         JS::Handle<JSString*> str1,
                                         JS::Handle<JSString*> str2,
                                         JS::MutableHandle<JS::Value> result);

}  // namespace intl

}  // namespace js

#endif /* builtin_intl_Collator_h */
