// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
import { needs, } from '@aws-crypto/material-management';
import { getRegionFromIdentifier } from './arn_parsing';
import { awsKmsMrkAreUnique } from './aws_kms_mrk_are_unique';
export function getAwsKmsMrkAwareStrictMultiKeyringBuilder(MrkAwareKeyring, MultiKeyring, defaultClientProvider) {
    //= compliance/framework/aws-kms/aws-kms-mrk-aware-multi-keyrings.txt#2.6
    //# The caller MUST provide:
    return function buildAwsKmsMrkAwareStrictMultiKeyring({ generatorKeyId, keyIds = [], 
    //= compliance/framework/aws-kms/aws-kms-mrk-aware-multi-keyrings.txt#2.6
    //# If
    //# a regional client supplier is not passed, then a default MUST be
    //# created that takes a region string and generates a default AWS SDK
    //# client for the given region.
    clientProvider = defaultClientProvider, grantTokens, } = {}) {
        const identifier2Client = identifier2ClientBuilder(clientProvider);
        const allIdentifiers = generatorKeyId ? [generatorKeyId, ...keyIds] : keyIds;
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-multi-keyrings.txt#2.6
        //# At least one non-null or non-empty string AWS
        //# KMS key identifiers exists in the input this function MUST fail.
        //
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-multi-keyrings.txt#2.6
        //# If any of the AWS KMS key identifiers is null or an empty string this
        //# function MUST fail.
        needs(allIdentifiers.length &&
            allIdentifiers.every((key) => typeof key === 'string' && key !== ''), 'Noop keyring is not allowed: Set a generatorKeyId or at least one keyId.');
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-multi-keyrings.txt#2.6
        //# All
        //# AWS KMS identifiers are passed to Assert AWS KMS MRK are unique (aws-
        //# kms-mrk-are-unique.md#Implementation) and the function MUST return
        //# success otherwise this MUST fail.
        awsKmsMrkAreUnique(allIdentifiers);
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-multi-keyrings.txt#2.6
        //# If there is a generator input then the generator keyring MUST be a
        //# AWS KMS MRK Aware Symmetric Keyring (aws-kms-mrk-aware-symmetric-
        //# keyring.md) initialized with
        const generator = generatorKeyId
            ? new MrkAwareKeyring({
                //= compliance/framework/aws-kms/aws-kms-mrk-aware-multi-keyrings.txt#2.6
                //# *  The AWS KMS client that MUST be created by the regional client
                //# supplier when called with the region part of the generator ARN or
                //# a signal for the AWS SDK to select the default region.
                client: identifier2Client(generatorKeyId),
                keyId: generatorKeyId,
                grantTokens,
            })
            : undefined;
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-multi-keyrings.txt#2.6
        //# If there is a set of child identifiers then a set of AWS KMS MRK
        //# Aware Symmetric Keyring (aws-kms-mrk-aware-symmetric-keyring.md) MUST
        //# be created for each AWS KMS key identifier by initialized each
        //# keyring with
        const children = keyIds.map((keyId) => {
            return new MrkAwareKeyring({
                //= compliance/framework/aws-kms/aws-kms-mrk-aware-multi-keyrings.txt#2.6
                //# *  The AWS KMS client that MUST be created by the regional client
                //# supplier when called with the region part of the AWS KMS key
                //# identifier or a signal for the AWS SDK to select the default
                //# region.
                client: identifier2Client(keyId),
                keyId,
                grantTokens,
            });
        });
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-multi-keyrings.txt#2.6
        //# Then a Multi-Keyring (../multi-keyring.md#inputs) MUST be initialize
        //# by using this generator keyring as the generator keyring (../multi-
        //# keyring.md#generator-keyring) and this set of child keyrings as the
        //# child keyrings (../multi-keyring.md#child-keyrings).
        //
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-multi-keyrings.txt#2.6
        //# This Multi-
        //# Keyring MUST be this functions output.
        return new MultiKeyring({
            generator,
            children,
        });
    };
}
export function identifier2ClientBuilder(clientProvider) {
    return function identifier2Client(identifier) {
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-multi-keyrings.txt#2.6
        //# NOTE: The AWS Encryption SDK SHOULD NOT attempt to evaluate its own
        //# default region.
        const region = getRegionFromIdentifier(identifier);
        const client = clientProvider(region);
        /* Postcondition: If the configured clientProvider is not able to create a client for a defined generator key, throw an error. */
        needs(client, `Configured clientProvider is unable to create a client for configured ${identifier}.`);
        return client;
    };
}
//# sourceMappingURL=data:application/json;base64,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