﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/sagemaker/SageMakerRequest.h>
#include <aws/sagemaker/SageMaker_EXPORTS.h>
#include <aws/sagemaker/model/ArtifactSource.h>
#include <aws/sagemaker/model/MetadataProperties.h>
#include <aws/sagemaker/model/Tag.h>

#include <utility>

namespace Aws {
namespace SageMaker {
namespace Model {

/**
 */
class CreateArtifactRequest : public SageMakerRequest {
 public:
  AWS_SAGEMAKER_API CreateArtifactRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateArtifact"; }

  AWS_SAGEMAKER_API Aws::String SerializePayload() const override;

  AWS_SAGEMAKER_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the artifact. Must be unique to your account in an Amazon Web
   * Services Region.</p>
   */
  inline const Aws::String& GetArtifactName() const { return m_artifactName; }
  inline bool ArtifactNameHasBeenSet() const { return m_artifactNameHasBeenSet; }
  template <typename ArtifactNameT = Aws::String>
  void SetArtifactName(ArtifactNameT&& value) {
    m_artifactNameHasBeenSet = true;
    m_artifactName = std::forward<ArtifactNameT>(value);
  }
  template <typename ArtifactNameT = Aws::String>
  CreateArtifactRequest& WithArtifactName(ArtifactNameT&& value) {
    SetArtifactName(std::forward<ArtifactNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID, ID type, and URI of the source.</p>
   */
  inline const ArtifactSource& GetSource() const { return m_source; }
  inline bool SourceHasBeenSet() const { return m_sourceHasBeenSet; }
  template <typename SourceT = ArtifactSource>
  void SetSource(SourceT&& value) {
    m_sourceHasBeenSet = true;
    m_source = std::forward<SourceT>(value);
  }
  template <typename SourceT = ArtifactSource>
  CreateArtifactRequest& WithSource(SourceT&& value) {
    SetSource(std::forward<SourceT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The artifact type.</p>
   */
  inline const Aws::String& GetArtifactType() const { return m_artifactType; }
  inline bool ArtifactTypeHasBeenSet() const { return m_artifactTypeHasBeenSet; }
  template <typename ArtifactTypeT = Aws::String>
  void SetArtifactType(ArtifactTypeT&& value) {
    m_artifactTypeHasBeenSet = true;
    m_artifactType = std::forward<ArtifactTypeT>(value);
  }
  template <typename ArtifactTypeT = Aws::String>
  CreateArtifactRequest& WithArtifactType(ArtifactTypeT&& value) {
    SetArtifactType(std::forward<ArtifactTypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of properties to add to the artifact.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetProperties() const { return m_properties; }
  inline bool PropertiesHasBeenSet() const { return m_propertiesHasBeenSet; }
  template <typename PropertiesT = Aws::Map<Aws::String, Aws::String>>
  void SetProperties(PropertiesT&& value) {
    m_propertiesHasBeenSet = true;
    m_properties = std::forward<PropertiesT>(value);
  }
  template <typename PropertiesT = Aws::Map<Aws::String, Aws::String>>
  CreateArtifactRequest& WithProperties(PropertiesT&& value) {
    SetProperties(std::forward<PropertiesT>(value));
    return *this;
  }
  template <typename PropertiesKeyT = Aws::String, typename PropertiesValueT = Aws::String>
  CreateArtifactRequest& AddProperties(PropertiesKeyT&& key, PropertiesValueT&& value) {
    m_propertiesHasBeenSet = true;
    m_properties.emplace(std::forward<PropertiesKeyT>(key), std::forward<PropertiesValueT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const MetadataProperties& GetMetadataProperties() const { return m_metadataProperties; }
  inline bool MetadataPropertiesHasBeenSet() const { return m_metadataPropertiesHasBeenSet; }
  template <typename MetadataPropertiesT = MetadataProperties>
  void SetMetadataProperties(MetadataPropertiesT&& value) {
    m_metadataPropertiesHasBeenSet = true;
    m_metadataProperties = std::forward<MetadataPropertiesT>(value);
  }
  template <typename MetadataPropertiesT = MetadataProperties>
  CreateArtifactRequest& WithMetadataProperties(MetadataPropertiesT&& value) {
    SetMetadataProperties(std::forward<MetadataPropertiesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of tags to apply to the artifact.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateArtifactRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateArtifactRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_artifactName;
  bool m_artifactNameHasBeenSet = false;

  ArtifactSource m_source;
  bool m_sourceHasBeenSet = false;

  Aws::String m_artifactType;
  bool m_artifactTypeHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_properties;
  bool m_propertiesHasBeenSet = false;

  MetadataProperties m_metadataProperties;
  bool m_metadataPropertiesHasBeenSet = false;

  Aws::Vector<Tag> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace SageMaker
}  // namespace Aws
