﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/chime-sdk-messaging/ChimeSDKMessagingRequest.h>
#include <aws/chime-sdk-messaging/ChimeSDKMessaging_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace ChimeSDKMessaging {
namespace Model {

/**
 */
class ListSubChannelsRequest : public ChimeSDKMessagingRequest {
 public:
  AWS_CHIMESDKMESSAGING_API ListSubChannelsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListSubChannels"; }

  AWS_CHIMESDKMESSAGING_API Aws::String SerializePayload() const override;

  AWS_CHIMESDKMESSAGING_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  AWS_CHIMESDKMESSAGING_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The ARN of elastic channel.</p>
   */
  inline const Aws::String& GetChannelArn() const { return m_channelArn; }
  inline bool ChannelArnHasBeenSet() const { return m_channelArnHasBeenSet; }
  template <typename ChannelArnT = Aws::String>
  void SetChannelArn(ChannelArnT&& value) {
    m_channelArnHasBeenSet = true;
    m_channelArn = std::forward<ChannelArnT>(value);
  }
  template <typename ChannelArnT = Aws::String>
  ListSubChannelsRequest& WithChannelArn(ChannelArnT&& value) {
    SetChannelArn(std::forward<ChannelArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The <code>AppInstanceUserArn</code> of the user making the API call.</p>
   */
  inline const Aws::String& GetChimeBearer() const { return m_chimeBearer; }
  inline bool ChimeBearerHasBeenSet() const { return m_chimeBearerHasBeenSet; }
  template <typename ChimeBearerT = Aws::String>
  void SetChimeBearer(ChimeBearerT&& value) {
    m_chimeBearerHasBeenSet = true;
    m_chimeBearer = std::forward<ChimeBearerT>(value);
  }
  template <typename ChimeBearerT = Aws::String>
  ListSubChannelsRequest& WithChimeBearer(ChimeBearerT&& value) {
    SetChimeBearer(std::forward<ChimeBearerT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of sub-channels that you want to return.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListSubChannelsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The token passed by previous API calls until all requested sub-channels are
   * returned.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListSubChannelsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_channelArn;

  Aws::String m_chimeBearer;

  int m_maxResults{0};

  Aws::String m_nextToken;
  bool m_channelArnHasBeenSet = false;
  bool m_chimeBearerHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace ChimeSDKMessaging
}  // namespace Aws
