﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/codecatalyst/CodeCatalystRequest.h>
#include <aws/codecatalyst/CodeCatalyst_EXPORTS.h>
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace CodeCatalyst {
namespace Model {

/**
 */
class StartWorkflowRunRequest : public CodeCatalystRequest {
 public:
  AWS_CODECATALYST_API StartWorkflowRunRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "StartWorkflowRun"; }

  AWS_CODECATALYST_API Aws::String SerializePayload() const override;

  AWS_CODECATALYST_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The name of the space.</p>
   */
  inline const Aws::String& GetSpaceName() const { return m_spaceName; }
  inline bool SpaceNameHasBeenSet() const { return m_spaceNameHasBeenSet; }
  template <typename SpaceNameT = Aws::String>
  void SetSpaceName(SpaceNameT&& value) {
    m_spaceNameHasBeenSet = true;
    m_spaceName = std::forward<SpaceNameT>(value);
  }
  template <typename SpaceNameT = Aws::String>
  StartWorkflowRunRequest& WithSpaceName(SpaceNameT&& value) {
    SetSpaceName(std::forward<SpaceNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the project in the space.</p>
   */
  inline const Aws::String& GetProjectName() const { return m_projectName; }
  inline bool ProjectNameHasBeenSet() const { return m_projectNameHasBeenSet; }
  template <typename ProjectNameT = Aws::String>
  void SetProjectName(ProjectNameT&& value) {
    m_projectNameHasBeenSet = true;
    m_projectName = std::forward<ProjectNameT>(value);
  }
  template <typename ProjectNameT = Aws::String>
  StartWorkflowRunRequest& WithProjectName(ProjectNameT&& value) {
    SetProjectName(std::forward<ProjectNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The system-generated unique ID of the workflow. To retrieve a list of
   * workflow IDs, use <a>ListWorkflows</a>.</p>
   */
  inline const Aws::String& GetWorkflowId() const { return m_workflowId; }
  inline bool WorkflowIdHasBeenSet() const { return m_workflowIdHasBeenSet; }
  template <typename WorkflowIdT = Aws::String>
  void SetWorkflowId(WorkflowIdT&& value) {
    m_workflowIdHasBeenSet = true;
    m_workflowId = std::forward<WorkflowIdT>(value);
  }
  template <typename WorkflowIdT = Aws::String>
  StartWorkflowRunRequest& WithWorkflowId(WorkflowIdT&& value) {
    SetWorkflowId(std::forward<WorkflowIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A user-specified idempotency token. Idempotency ensures that an API request
   * completes only once. With an idempotent request, if the original request
   * completes successfully, the subsequent retries return the result from the
   * original successful request and have no additional effect.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  StartWorkflowRunRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_spaceName;

  Aws::String m_projectName;

  Aws::String m_workflowId;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_spaceNameHasBeenSet = false;
  bool m_projectNameHasBeenSet = false;
  bool m_workflowIdHasBeenSet = false;
  bool m_clientTokenHasBeenSet = true;
};

}  // namespace Model
}  // namespace CodeCatalyst
}  // namespace Aws
