﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/codestar-connections/CodeStarconnectionsRequest.h>
#include <aws/codestar-connections/CodeStarconnections_EXPORTS.h>
#include <aws/codestar-connections/model/ProviderType.h>
#include <aws/codestar-connections/model/Tag.h>
#include <aws/codestar-connections/model/VpcConfiguration.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace CodeStarconnections {
namespace Model {

/**
 */
class CreateHostRequest : public CodeStarconnectionsRequest {
 public:
  AWS_CODESTARCONNECTIONS_API CreateHostRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateHost"; }

  AWS_CODESTARCONNECTIONS_API Aws::String SerializePayload() const override;

  AWS_CODESTARCONNECTIONS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the host to be created.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateHostRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the installed provider to be associated with your connection. The
   * host resource represents the infrastructure where your provider type is
   * installed. The valid provider type is GitHub Enterprise Server.</p>
   */
  inline ProviderType GetProviderType() const { return m_providerType; }
  inline bool ProviderTypeHasBeenSet() const { return m_providerTypeHasBeenSet; }
  inline void SetProviderType(ProviderType value) {
    m_providerTypeHasBeenSet = true;
    m_providerType = value;
  }
  inline CreateHostRequest& WithProviderType(ProviderType value) {
    SetProviderType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The endpoint of the infrastructure to be represented by the host after it is
   * created.</p>
   */
  inline const Aws::String& GetProviderEndpoint() const { return m_providerEndpoint; }
  inline bool ProviderEndpointHasBeenSet() const { return m_providerEndpointHasBeenSet; }
  template <typename ProviderEndpointT = Aws::String>
  void SetProviderEndpoint(ProviderEndpointT&& value) {
    m_providerEndpointHasBeenSet = true;
    m_providerEndpoint = std::forward<ProviderEndpointT>(value);
  }
  template <typename ProviderEndpointT = Aws::String>
  CreateHostRequest& WithProviderEndpoint(ProviderEndpointT&& value) {
    SetProviderEndpoint(std::forward<ProviderEndpointT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The VPC configuration to be provisioned for the host. A VPC must be
   * configured and the infrastructure to be represented by the host must already be
   * connected to the VPC.</p>
   */
  inline const VpcConfiguration& GetVpcConfiguration() const { return m_vpcConfiguration; }
  inline bool VpcConfigurationHasBeenSet() const { return m_vpcConfigurationHasBeenSet; }
  template <typename VpcConfigurationT = VpcConfiguration>
  void SetVpcConfiguration(VpcConfigurationT&& value) {
    m_vpcConfigurationHasBeenSet = true;
    m_vpcConfiguration = std::forward<VpcConfigurationT>(value);
  }
  template <typename VpcConfigurationT = VpcConfiguration>
  CreateHostRequest& WithVpcConfiguration(VpcConfigurationT&& value) {
    SetVpcConfiguration(std::forward<VpcConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Tags for the host to be created.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateHostRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateHostRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;

  ProviderType m_providerType{ProviderType::NOT_SET};

  Aws::String m_providerEndpoint;

  VpcConfiguration m_vpcConfiguration;

  Aws::Vector<Tag> m_tags;
  bool m_nameHasBeenSet = false;
  bool m_providerTypeHasBeenSet = false;
  bool m_providerEndpointHasBeenSet = false;
  bool m_vpcConfigurationHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace CodeStarconnections
}  // namespace Aws
