﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/config/ConfigServiceRequest.h>
#include <aws/config/ConfigService_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace ConfigService {
namespace Model {

/**
 */
class SelectAggregateResourceConfigRequest : public ConfigServiceRequest {
 public:
  AWS_CONFIGSERVICE_API SelectAggregateResourceConfigRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "SelectAggregateResourceConfig"; }

  AWS_CONFIGSERVICE_API Aws::String SerializePayload() const override;

  AWS_CONFIGSERVICE_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The SQL query SELECT command. </p>
   */
  inline const Aws::String& GetExpression() const { return m_expression; }
  inline bool ExpressionHasBeenSet() const { return m_expressionHasBeenSet; }
  template <typename ExpressionT = Aws::String>
  void SetExpression(ExpressionT&& value) {
    m_expressionHasBeenSet = true;
    m_expression = std::forward<ExpressionT>(value);
  }
  template <typename ExpressionT = Aws::String>
  SelectAggregateResourceConfigRequest& WithExpression(ExpressionT&& value) {
    SetExpression(std::forward<ExpressionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the configuration aggregator.</p>
   */
  inline const Aws::String& GetConfigurationAggregatorName() const { return m_configurationAggregatorName; }
  inline bool ConfigurationAggregatorNameHasBeenSet() const { return m_configurationAggregatorNameHasBeenSet; }
  template <typename ConfigurationAggregatorNameT = Aws::String>
  void SetConfigurationAggregatorName(ConfigurationAggregatorNameT&& value) {
    m_configurationAggregatorNameHasBeenSet = true;
    m_configurationAggregatorName = std::forward<ConfigurationAggregatorNameT>(value);
  }
  template <typename ConfigurationAggregatorNameT = Aws::String>
  SelectAggregateResourceConfigRequest& WithConfigurationAggregatorName(ConfigurationAggregatorNameT&& value) {
    SetConfigurationAggregatorName(std::forward<ConfigurationAggregatorNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of query results returned on each page. </p>
   */
  inline int GetLimit() const { return m_limit; }
  inline bool LimitHasBeenSet() const { return m_limitHasBeenSet; }
  inline void SetLimit(int value) {
    m_limitHasBeenSet = true;
    m_limit = value;
  }
  inline SelectAggregateResourceConfigRequest& WithLimit(int value) {
    SetLimit(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of query results returned on each page. Config also allows
   * the Limit request parameter.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline SelectAggregateResourceConfigRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The nextToken string returned in a previous request that you use to request
   * the next page of results in a paginated response. </p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  SelectAggregateResourceConfigRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_expression;

  Aws::String m_configurationAggregatorName;

  int m_limit{0};

  int m_maxResults{0};

  Aws::String m_nextToken;
  bool m_expressionHasBeenSet = false;
  bool m_configurationAggregatorNameHasBeenSet = false;
  bool m_limitHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace ConfigService
}  // namespace Aws
