﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/detective/DetectiveRequest.h>
#include <aws/detective/Detective_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Detective {
namespace Model {

/**
 */
class ListMembersRequest : public DetectiveRequest {
 public:
  AWS_DETECTIVE_API ListMembersRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListMembers"; }

  AWS_DETECTIVE_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The ARN of the behavior graph for which to retrieve the list of member
   * accounts.</p>
   */
  inline const Aws::String& GetGraphArn() const { return m_graphArn; }
  inline bool GraphArnHasBeenSet() const { return m_graphArnHasBeenSet; }
  template <typename GraphArnT = Aws::String>
  void SetGraphArn(GraphArnT&& value) {
    m_graphArnHasBeenSet = true;
    m_graphArn = std::forward<GraphArnT>(value);
  }
  template <typename GraphArnT = Aws::String>
  ListMembersRequest& WithGraphArn(GraphArnT&& value) {
    SetGraphArn(std::forward<GraphArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>For requests to retrieve the next page of member account results, the
   * pagination token that was returned with the previous page of results. The
   * initial request does not include a pagination token.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListMembersRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of member accounts to include in the response. The total
   * must be less than the overall limit on the number of results to return, which is
   * currently 200.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListMembersRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_graphArn;

  Aws::String m_nextToken;

  int m_maxResults{0};
  bool m_graphArnHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Detective
}  // namespace Aws
