﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/networkmanager/NetworkManager_EXPORTS.h>
#include <aws/networkmanager/model/Tag.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace NetworkManager {
namespace Model {

/**
 * <p>Describes a network resource.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/networkmanager-2019-07-05/NetworkResource">AWS
 * API Reference</a></p>
 */
class NetworkResource {
 public:
  AWS_NETWORKMANAGER_API NetworkResource() = default;
  AWS_NETWORKMANAGER_API NetworkResource(Aws::Utils::Json::JsonView jsonValue);
  AWS_NETWORKMANAGER_API NetworkResource& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_NETWORKMANAGER_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The ARN of the gateway.</p>
   */
  inline const Aws::String& GetRegisteredGatewayArn() const { return m_registeredGatewayArn; }
  inline bool RegisteredGatewayArnHasBeenSet() const { return m_registeredGatewayArnHasBeenSet; }
  template <typename RegisteredGatewayArnT = Aws::String>
  void SetRegisteredGatewayArn(RegisteredGatewayArnT&& value) {
    m_registeredGatewayArnHasBeenSet = true;
    m_registeredGatewayArn = std::forward<RegisteredGatewayArnT>(value);
  }
  template <typename RegisteredGatewayArnT = Aws::String>
  NetworkResource& WithRegisteredGatewayArn(RegisteredGatewayArnT&& value) {
    SetRegisteredGatewayArn(std::forward<RegisteredGatewayArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of a core network.</p>
   */
  inline const Aws::String& GetCoreNetworkId() const { return m_coreNetworkId; }
  inline bool CoreNetworkIdHasBeenSet() const { return m_coreNetworkIdHasBeenSet; }
  template <typename CoreNetworkIdT = Aws::String>
  void SetCoreNetworkId(CoreNetworkIdT&& value) {
    m_coreNetworkIdHasBeenSet = true;
    m_coreNetworkId = std::forward<CoreNetworkIdT>(value);
  }
  template <typename CoreNetworkIdT = Aws::String>
  NetworkResource& WithCoreNetworkId(CoreNetworkIdT&& value) {
    SetCoreNetworkId(std::forward<CoreNetworkIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Web Services Region.</p>
   */
  inline const Aws::String& GetAwsRegion() const { return m_awsRegion; }
  inline bool AwsRegionHasBeenSet() const { return m_awsRegionHasBeenSet; }
  template <typename AwsRegionT = Aws::String>
  void SetAwsRegion(AwsRegionT&& value) {
    m_awsRegionHasBeenSet = true;
    m_awsRegion = std::forward<AwsRegionT>(value);
  }
  template <typename AwsRegionT = Aws::String>
  NetworkResource& WithAwsRegion(AwsRegionT&& value) {
    SetAwsRegion(std::forward<AwsRegionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Web Services account ID.</p>
   */
  inline const Aws::String& GetAccountId() const { return m_accountId; }
  inline bool AccountIdHasBeenSet() const { return m_accountIdHasBeenSet; }
  template <typename AccountIdT = Aws::String>
  void SetAccountId(AccountIdT&& value) {
    m_accountIdHasBeenSet = true;
    m_accountId = std::forward<AccountIdT>(value);
  }
  template <typename AccountIdT = Aws::String>
  NetworkResource& WithAccountId(AccountIdT&& value) {
    SetAccountId(std::forward<AccountIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The resource type.</p> <p>The following are the supported resource types for
   * Direct Connect:</p> <ul> <li> <p> <code>dxcon</code> </p> </li> <li> <p>
   * <code>dx-gateway</code> </p> </li> <li> <p> <code>dx-vif</code> </p> </li> </ul>
   * <p>The following are the supported resource types for Network Manager:</p> <ul>
   * <li> <p> <code>attachment</code> </p> </li> <li> <p> <code>connect-peer</code>
   * </p> </li> <li> <p> <code>connection</code> </p> </li> <li> <p>
   * <code>core-network</code> </p> </li> <li> <p> <code>device</code> </p> </li>
   * <li> <p> <code>link</code> </p> </li> <li> <p> <code>peering</code> </p> </li>
   * <li> <p> <code>site</code> </p> </li> </ul> <p>The following are the supported
   * resource types for Amazon VPC:</p> <ul> <li> <p> <code>customer-gateway</code>
   * </p> </li> <li> <p> <code>transit-gateway</code> </p> </li> <li> <p>
   * <code>transit-gateway-attachment</code> </p> </li> <li> <p>
   * <code>transit-gateway-connect-peer</code> </p> </li> <li> <p>
   * <code>transit-gateway-route-table</code> </p> </li> <li> <p>
   * <code>vpn-connection</code> </p> </li> </ul>
   */
  inline const Aws::String& GetResourceType() const { return m_resourceType; }
  inline bool ResourceTypeHasBeenSet() const { return m_resourceTypeHasBeenSet; }
  template <typename ResourceTypeT = Aws::String>
  void SetResourceType(ResourceTypeT&& value) {
    m_resourceTypeHasBeenSet = true;
    m_resourceType = std::forward<ResourceTypeT>(value);
  }
  template <typename ResourceTypeT = Aws::String>
  NetworkResource& WithResourceType(ResourceTypeT&& value) {
    SetResourceType(std::forward<ResourceTypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the resource.</p>
   */
  inline const Aws::String& GetResourceId() const { return m_resourceId; }
  inline bool ResourceIdHasBeenSet() const { return m_resourceIdHasBeenSet; }
  template <typename ResourceIdT = Aws::String>
  void SetResourceId(ResourceIdT&& value) {
    m_resourceIdHasBeenSet = true;
    m_resourceId = std::forward<ResourceIdT>(value);
  }
  template <typename ResourceIdT = Aws::String>
  NetworkResource& WithResourceId(ResourceIdT&& value) {
    SetResourceId(std::forward<ResourceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the resource.</p>
   */
  inline const Aws::String& GetResourceArn() const { return m_resourceArn; }
  inline bool ResourceArnHasBeenSet() const { return m_resourceArnHasBeenSet; }
  template <typename ResourceArnT = Aws::String>
  void SetResourceArn(ResourceArnT&& value) {
    m_resourceArnHasBeenSet = true;
    m_resourceArn = std::forward<ResourceArnT>(value);
  }
  template <typename ResourceArnT = Aws::String>
  NetworkResource& WithResourceArn(ResourceArnT&& value) {
    SetResourceArn(std::forward<ResourceArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Information about the resource, in JSON format. Network Manager gets this
   * information by describing the resource using its Describe API call.</p>
   */
  inline const Aws::String& GetDefinition() const { return m_definition; }
  inline bool DefinitionHasBeenSet() const { return m_definitionHasBeenSet; }
  template <typename DefinitionT = Aws::String>
  void SetDefinition(DefinitionT&& value) {
    m_definitionHasBeenSet = true;
    m_definition = std::forward<DefinitionT>(value);
  }
  template <typename DefinitionT = Aws::String>
  NetworkResource& WithDefinition(DefinitionT&& value) {
    SetDefinition(std::forward<DefinitionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The time that the resource definition was retrieved.</p>
   */
  inline const Aws::Utils::DateTime& GetDefinitionTimestamp() const { return m_definitionTimestamp; }
  inline bool DefinitionTimestampHasBeenSet() const { return m_definitionTimestampHasBeenSet; }
  template <typename DefinitionTimestampT = Aws::Utils::DateTime>
  void SetDefinitionTimestamp(DefinitionTimestampT&& value) {
    m_definitionTimestampHasBeenSet = true;
    m_definitionTimestamp = std::forward<DefinitionTimestampT>(value);
  }
  template <typename DefinitionTimestampT = Aws::Utils::DateTime>
  NetworkResource& WithDefinitionTimestamp(DefinitionTimestampT&& value) {
    SetDefinitionTimestamp(std::forward<DefinitionTimestampT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  NetworkResource& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  NetworkResource& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The resource metadata.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetMetadata() const { return m_metadata; }
  inline bool MetadataHasBeenSet() const { return m_metadataHasBeenSet; }
  template <typename MetadataT = Aws::Map<Aws::String, Aws::String>>
  void SetMetadata(MetadataT&& value) {
    m_metadataHasBeenSet = true;
    m_metadata = std::forward<MetadataT>(value);
  }
  template <typename MetadataT = Aws::Map<Aws::String, Aws::String>>
  NetworkResource& WithMetadata(MetadataT&& value) {
    SetMetadata(std::forward<MetadataT>(value));
    return *this;
  }
  template <typename MetadataKeyT = Aws::String, typename MetadataValueT = Aws::String>
  NetworkResource& AddMetadata(MetadataKeyT&& key, MetadataValueT&& value) {
    m_metadataHasBeenSet = true;
    m_metadata.emplace(std::forward<MetadataKeyT>(key), std::forward<MetadataValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_registeredGatewayArn;

  Aws::String m_coreNetworkId;

  Aws::String m_awsRegion;

  Aws::String m_accountId;

  Aws::String m_resourceType;

  Aws::String m_resourceId;

  Aws::String m_resourceArn;

  Aws::String m_definition;

  Aws::Utils::DateTime m_definitionTimestamp{};

  Aws::Vector<Tag> m_tags;

  Aws::Map<Aws::String, Aws::String> m_metadata;
  bool m_registeredGatewayArnHasBeenSet = false;
  bool m_coreNetworkIdHasBeenSet = false;
  bool m_awsRegionHasBeenSet = false;
  bool m_accountIdHasBeenSet = false;
  bool m_resourceTypeHasBeenSet = false;
  bool m_resourceIdHasBeenSet = false;
  bool m_resourceArnHasBeenSet = false;
  bool m_definitionHasBeenSet = false;
  bool m_definitionTimestampHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
  bool m_metadataHasBeenSet = false;
};

}  // namespace Model
}  // namespace NetworkManager
}  // namespace Aws
