﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/ssm-incidents/SSMIncidentsRequest.h>
#include <aws/ssm-incidents/SSMIncidents_EXPORTS.h>

#include <utility>

namespace Aws {
namespace SSMIncidents {
namespace Model {

/**
 */
class ListRelatedItemsRequest : public SSMIncidentsRequest {
 public:
  AWS_SSMINCIDENTS_API ListRelatedItemsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListRelatedItems"; }

  AWS_SSMINCIDENTS_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the incident record containing the listed
   * related items.</p>
   */
  inline const Aws::String& GetIncidentRecordArn() const { return m_incidentRecordArn; }
  inline bool IncidentRecordArnHasBeenSet() const { return m_incidentRecordArnHasBeenSet; }
  template <typename IncidentRecordArnT = Aws::String>
  void SetIncidentRecordArn(IncidentRecordArnT&& value) {
    m_incidentRecordArnHasBeenSet = true;
    m_incidentRecordArn = std::forward<IncidentRecordArnT>(value);
  }
  template <typename IncidentRecordArnT = Aws::String>
  ListRelatedItemsRequest& WithIncidentRecordArn(IncidentRecordArnT&& value) {
    SetIncidentRecordArn(std::forward<IncidentRecordArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of related items per page.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListRelatedItemsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The pagination token for the next set of items to return. (You received this
   * token from a previous call.)</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListRelatedItemsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_incidentRecordArn;

  int m_maxResults{0};

  Aws::String m_nextToken;
  bool m_incidentRecordArnHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace SSMIncidents
}  // namespace Aws
