﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/iam/IAMRequest.h>
#include <aws/iam/IAM_EXPORTS.h>
#include <aws/iam/model/SortKeyType.h>

#include <utility>

namespace Aws {
namespace IAM {
namespace Model {

/**
 */
class GetOrganizationsAccessReportRequest : public IAMRequest {
 public:
  AWS_IAM_API GetOrganizationsAccessReportRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetOrganizationsAccessReport"; }

  AWS_IAM_API Aws::String SerializePayload() const override;

 protected:
  AWS_IAM_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The identifier of the request generated by the <a
   * href="https://docs.aws.amazon.com/IAM/latest/APIReference/API_GenerateOrganizationsAccessReport.html">GenerateOrganizationsAccessReport</a>
   * operation.</p>
   */
  inline const Aws::String& GetJobId() const { return m_jobId; }
  inline bool JobIdHasBeenSet() const { return m_jobIdHasBeenSet; }
  template <typename JobIdT = Aws::String>
  void SetJobId(JobIdT&& value) {
    m_jobIdHasBeenSet = true;
    m_jobId = std::forward<JobIdT>(value);
  }
  template <typename JobIdT = Aws::String>
  GetOrganizationsAccessReportRequest& WithJobId(JobIdT&& value) {
    SetJobId(std::forward<JobIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Use this only when paginating results to indicate the maximum number of items
   * you want in the response. If additional items exist beyond the maximum you
   * specify, the <code>IsTruncated</code> response element is <code>true</code>.</p>
   * <p>If you do not include this parameter, the number of items defaults to 100.
   * Note that IAM might return fewer results, even when there are more results
   * available. In that case, the <code>IsTruncated</code> response element returns
   * <code>true</code>, and <code>Marker</code> contains a value to include in the
   * subsequent call that tells the service where to continue from.</p>
   */
  inline int GetMaxItems() const { return m_maxItems; }
  inline bool MaxItemsHasBeenSet() const { return m_maxItemsHasBeenSet; }
  inline void SetMaxItems(int value) {
    m_maxItemsHasBeenSet = true;
    m_maxItems = value;
  }
  inline GetOrganizationsAccessReportRequest& WithMaxItems(int value) {
    SetMaxItems(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Use this parameter only when paginating results and only after you receive a
   * response indicating that the results are truncated. Set it to the value of the
   * <code>Marker</code> element in the response that you received to indicate where
   * the next call should start.</p>
   */
  inline const Aws::String& GetMarker() const { return m_marker; }
  inline bool MarkerHasBeenSet() const { return m_markerHasBeenSet; }
  template <typename MarkerT = Aws::String>
  void SetMarker(MarkerT&& value) {
    m_markerHasBeenSet = true;
    m_marker = std::forward<MarkerT>(value);
  }
  template <typename MarkerT = Aws::String>
  GetOrganizationsAccessReportRequest& WithMarker(MarkerT&& value) {
    SetMarker(std::forward<MarkerT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The key that is used to sort the results. If you choose the namespace key,
   * the results are returned in alphabetical order. If you choose the time key, the
   * results are sorted numerically by the date and time.</p>
   */
  inline SortKeyType GetSortKey() const { return m_sortKey; }
  inline bool SortKeyHasBeenSet() const { return m_sortKeyHasBeenSet; }
  inline void SetSortKey(SortKeyType value) {
    m_sortKeyHasBeenSet = true;
    m_sortKey = value;
  }
  inline GetOrganizationsAccessReportRequest& WithSortKey(SortKeyType value) {
    SetSortKey(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_jobId;

  int m_maxItems{0};

  Aws::String m_marker;

  SortKeyType m_sortKey{SortKeyType::NOT_SET};
  bool m_jobIdHasBeenSet = false;
  bool m_maxItemsHasBeenSet = false;
  bool m_markerHasBeenSet = false;
  bool m_sortKeyHasBeenSet = false;
};

}  // namespace Model
}  // namespace IAM
}  // namespace Aws
