#include <linux/magic.h>
#include "filesystems.h"

/* PROJECT_FILE, which is used by log_xyz() thus also used by assert_not_reached(), cannot be used in
 * generated files, as the build directory may be outside of the source directory. */
#undef PROJECT_FILE
#define PROJECT_FILE __FILE__

const char* fs_type_to_string(statfs_f_type_t magic) {
        switch (magic) {
        case (statfs_f_type_t) AAFS_MAGIC:
                return "apparmorfs";
        case (statfs_f_type_t) ADFS_SUPER_MAGIC:
                return "adfs";
        case (statfs_f_type_t) AFFS_SUPER_MAGIC:
                return "affs";
        case (statfs_f_type_t) AFS_FS_MAGIC:
        case (statfs_f_type_t) AFS_SUPER_MAGIC:
                return "afs";
        case (statfs_f_type_t) ANON_INODE_FS_MAGIC:
                return "anon_inodefs";
        case (statfs_f_type_t) AUTOFS_SUPER_MAGIC:
                return "autofs";
        case (statfs_f_type_t) BALLOON_KVM_MAGIC:
                return "balloon-kvm";
        case (statfs_f_type_t) BCACHEFS_SUPER_MAGIC:
                return "bcachefs";
        case (statfs_f_type_t) BDEVFS_MAGIC:
                return "bdev";
        case (statfs_f_type_t) BINDERFS_SUPER_MAGIC:
                return "binder";
        case (statfs_f_type_t) BINFMTFS_MAGIC:
                return "binfmt_misc";
        case (statfs_f_type_t) BPF_FS_MAGIC:
                return "bpf";
        case (statfs_f_type_t) BTRFS_SUPER_MAGIC:
                return "btrfs";
        case (statfs_f_type_t) BTRFS_TEST_MAGIC:
                return "btrfs_test_fs";
        case (statfs_f_type_t) CEPH_SUPER_MAGIC:
                return "ceph";
        case (statfs_f_type_t) CGROUP2_SUPER_MAGIC:
                return "cgroup2";
        case (statfs_f_type_t) CGROUP_SUPER_MAGIC:
                return "cgroup";
        case (statfs_f_type_t) CIFS_SUPER_MAGIC:
        case (statfs_f_type_t) SMB2_SUPER_MAGIC:
                return "cifs";
        case (statfs_f_type_t) CODA_SUPER_MAGIC:
                return "coda";
        case (statfs_f_type_t) CONFIGFS_MAGIC:
                return "configfs";
        case (statfs_f_type_t) CRAMFS_MAGIC:
                return "cramfs";
        case (statfs_f_type_t) DAXFS_MAGIC:
                return "dax";
        case (statfs_f_type_t) DEBUGFS_MAGIC:
                return "debugfs";
        case (statfs_f_type_t) DEVMEM_MAGIC:
                return "devmem";
        case (statfs_f_type_t) DEVPTS_SUPER_MAGIC:
                return "devpts";
        case (statfs_f_type_t) DMA_BUF_MAGIC:
                return "dmabuf";
        case (statfs_f_type_t) ECRYPTFS_SUPER_MAGIC:
                return "ecryptfs";
        case (statfs_f_type_t) EFIVARFS_MAGIC:
                return "efivarfs";
        case (statfs_f_type_t) EFS_SUPER_MAGIC:
                return "efs";
        case (statfs_f_type_t) EROFS_SUPER_MAGIC_V1:
                return "erofs";
        case (statfs_f_type_t) EXT4_SUPER_MAGIC:
                return "ext4";
        case (statfs_f_type_t) EXFAT_SUPER_MAGIC:
                return "exfat";
        case (statfs_f_type_t) F2FS_SUPER_MAGIC:
                return "f2fs";
        case (statfs_f_type_t) FUSE_SUPER_MAGIC:
                return "fuse";
        case (statfs_f_type_t) FUSE_CTL_SUPER_MAGIC:
                return "fusectl";
        case (statfs_f_type_t) GFS2_MAGIC:
                return "gfs2";
        case (statfs_f_type_t) GUEST_MEMFD_MAGIC:
                return "gmem";
        case (statfs_f_type_t) HOSTFS_SUPER_MAGIC:
                return "hostfs";
        case (statfs_f_type_t) HPFS_SUPER_MAGIC:
                return "hpfs";
        case (statfs_f_type_t) HUGETLBFS_MAGIC:
                return "hugetlbfs";
        case (statfs_f_type_t) ISOFS_SUPER_MAGIC:
                return "iso9660";
        case (statfs_f_type_t) JFFS2_SUPER_MAGIC:
                return "jffs2";
        case (statfs_f_type_t) MINIX_SUPER_MAGIC:
        case (statfs_f_type_t) MINIX_SUPER_MAGIC2:
        case (statfs_f_type_t) MINIX2_SUPER_MAGIC:
        case (statfs_f_type_t) MINIX2_SUPER_MAGIC2:
        case (statfs_f_type_t) MINIX3_SUPER_MAGIC:
                return "minix";
        case (statfs_f_type_t) MQUEUE_MAGIC:
                return "mqueue";
        case (statfs_f_type_t) NCP_SUPER_MAGIC:
                return "ncpfs";
        case (statfs_f_type_t) NFS_SUPER_MAGIC:
                return "nfs4";
        case (statfs_f_type_t) NILFS_SUPER_MAGIC:
                return "nilfs2";
        case (statfs_f_type_t) NSFS_MAGIC:
                return "nsfs";
        case (statfs_f_type_t) NTFS_SB_MAGIC:
                return "ntfs";
        case (statfs_f_type_t) NTFS3_SUPER_MAGIC:
                return "ntfs3";
        case (statfs_f_type_t) NULL_FS_MAGIC:
                return "nullfs";
        case (statfs_f_type_t) OCFS2_SUPER_MAGIC:
                return "ocfs2";
        case (statfs_f_type_t) OPENPROM_SUPER_MAGIC:
                return "openpromfs";
        case (statfs_f_type_t) ORANGEFS_DEVREQ_MAGIC:
                return "orangefs";
        case (statfs_f_type_t) OVERLAYFS_SUPER_MAGIC:
                return "overlay";
        case (statfs_f_type_t) PID_FS_MAGIC:
                return "pidfs";
        case (statfs_f_type_t) PIPEFS_MAGIC:
                return "pipefs";
        case (statfs_f_type_t) PPC_CMM_MAGIC:
                return "ppc-cmm";
        case (statfs_f_type_t) PROC_SUPER_MAGIC:
                return "proc";
        case (statfs_f_type_t) PSTOREFS_MAGIC:
                return "pstore";
        case (statfs_f_type_t) QNX4_SUPER_MAGIC:
                return "qnx4";
        case (statfs_f_type_t) QNX6_SUPER_MAGIC:
                return "qnx6";
        case (statfs_f_type_t) RAMFS_MAGIC:
                return "ramfs";
        case (statfs_f_type_t) RDTGROUP_SUPER_MAGIC:
                return "resctrl";
        case (statfs_f_type_t) REISERFS_SUPER_MAGIC:
                return "reiserfs";
        case (statfs_f_type_t) RPC_PIPEFS_SUPER_MAGIC:
                return "rpc_pipefs";
        case (statfs_f_type_t) SECRETMEM_MAGIC:
                return "secretmem";
        case (statfs_f_type_t) SECURITYFS_MAGIC:
                return "securityfs";
        case (statfs_f_type_t) SELINUX_MAGIC:
                return "selinuxfs";
        case (statfs_f_type_t) SHIFTFS_MAGIC:
                return "shiftfs";
        case (statfs_f_type_t) SMACK_MAGIC:
                return "smackfs";
        case (statfs_f_type_t) SMB_SUPER_MAGIC:
                return "smbfs";
        case (statfs_f_type_t) SOCKFS_MAGIC:
                return "sockfs";
        case (statfs_f_type_t) SQUASHFS_MAGIC:
                return "squashfs";
        case (statfs_f_type_t) SYSFS_MAGIC:
                return "sysfs";
        case (statfs_f_type_t) TMPFS_MAGIC:
                return "tmpfs";
        case (statfs_f_type_t) TRACEFS_MAGIC:
                return "tracefs";
        case (statfs_f_type_t) UDF_SUPER_MAGIC:
                return "udf";
        case (statfs_f_type_t) USBDEVICE_SUPER_MAGIC:
                return "usbdevfs";
        case (statfs_f_type_t) VBOXSF_SUPER_MAGIC:
                return "vboxsf";
        case (statfs_f_type_t) MSDOS_SUPER_MAGIC:
                return "vfat";
        case (statfs_f_type_t) V9FS_MAGIC:
                return "v9fs";
        case (statfs_f_type_t) XENFS_SUPER_MAGIC:
                return "xenfs";
        case (statfs_f_type_t) XFS_SUPER_MAGIC:
                return "xfs";
        case (statfs_f_type_t) Z3FOLD_MAGIC:
                return "z3fold";
        case (statfs_f_type_t) ZONEFS_MAGIC:
                return "zonefs";
        case (statfs_f_type_t) ZSMALLOC_MAGIC:
                return "zsmalloc";
        }
        return NULL;
}
const FilesystemSet filesystem_sets[_FILESYSTEM_SET_MAX] = {
        [FILESYSTEM_SET_BASIC_API] = {
                .name = "@basic-api",
                .help = "Basic filesystem API",
                .value =
                "cgroup\0"
                "cgroup2\0"
                "devpts\0"
                "devtmpfs\0"
                "mqueue\0"
                "proc\0"
                "sysfs\0"
        },
        [FILESYSTEM_SET_ANONYMOUS] = {
                .name = "@anonymous",
                .help = "Anonymous inodes",
                .value =
                "anon_inodefs\0"
                "pipefs\0"
                "sockfs\0"
        },
        [FILESYSTEM_SET_APPLICATION] = {
                .name = "@application",
                .help = "Application virtual filesystems",
                .value =
                "autofs\0"
                "fuse\0"
                "overlay\0"
        },
        [FILESYSTEM_SET_AUXILIARY_API] = {
                .name = "@auxiliary-api",
                .help = "Auxiliary filesystem API",
                .value =
                "binfmt_misc\0"
                "configfs\0"
                "efivarfs\0"
                "fusectl\0"
                "hugetlbfs\0"
                "rpc_pipefs\0"
                "securityfs\0"
        },
        [FILESYSTEM_SET_COMMON_BLOCK] = {
                .name = "@common-block",
                .help = "Common block device filesystems",
                .value =
                "btrfs\0"
                "erofs\0"
                "exfat\0"
                "ext4\0"
                "f2fs\0"
                "iso9660\0"
                "ntfs3\0"
                "squashfs\0"
                "udf\0"
                "vfat\0"
                "xfs\0"
        },
        [FILESYSTEM_SET_HISTORICAL_BLOCK] = {
                .name = "@historical-block",
                .help = "Historical block device filesystems",
                .value =
                "ext2\0"
                "ext3\0"
                "minix\0"
        },
        [FILESYSTEM_SET_NETWORK] = {
                .name = "@network",
                .help = "Well-known network filesystems",
                .value =
                "afs\0"
                "ceph\0"
                "cifs\0"
                "gfs\0"
                "gfs2\0"
                "ncp\0"
                "ncpfs\0"
                "nfs\0"
                "nfs4\0"
                "ocfs2\0"
                "orangefs\0"
                "pvfs2\0"
                "smb3\0"
                "smbfs\0"
        },
        [FILESYSTEM_SET_PRIVILEGED_API] = {
                .name = "@privileged-api",
                .help = "Privileged filesystem API",
                .value =
                "bpf\0"
                "debugfs\0"
                "pstore\0"
                "tracefs\0"
        },
        [FILESYSTEM_SET_SECURITY] = {
                .name = "@security",
                .help = "Security/MAC API VFS",
                .value =
                "apparmorfs\0"
                "selinuxfs\0"
                "smackfs\0"
        },
        [FILESYSTEM_SET_TEMPORARY] = {
                .name = "@temporary",
                .help = "Temporary filesystems",
                .value =
                "ramfs\0"
                "tmpfs\0"
        },
        [FILESYSTEM_SET_KNOWN] = {
                .name = "@known",
                .help = "All known filesystems declared in the kernel",
                .value =
                "apparmorfs\0"
                "adfs\0"
                "affs\0"
                "afs\0"
                "anon_inodefs\0"
                "autofs\0"
                "balloon-kvm\0"
                "bcachefs\0"
                "bdev\0"
                "binder\0"
                "binfmt_misc\0"
                "bpf\0"
                "btrfs\0"
                "btrfs_test_fs\0"
                "cpuset\0"
                "ceph\0"
                "cgroup2\0"
                "cgroup\0"
                "cifs\0"
                "coda\0"
                "configfs\0"
                "cramfs\0"
                "dax\0"
                "debugfs\0"
                "devmem\0"
                "devpts\0"
                "devtmpfs\0"
                "dmabuf\0"
                "ecryptfs\0"
                "efivarfs\0"
                "efs\0"
                "erofs\0"
                "ext2\0"
                "ext3\0"
                "ext4\0"
                "exfat\0"
                "f2fs\0"
                "fuseblk\0"
                "fuse\0"
                "fusectl\0"
                "gfs\0"
                "gfs2\0"
                "gmem\0"
                "hostfs\0"
                "hpfs\0"
                "hugetlbfs\0"
                "iso9660\0"
                "jffs2\0"
                "minix\0"
                "mqueue\0"
                "msdos\0"
                "ncp\0"
                "ncpfs\0"
                "nfs\0"
                "nfs4\0"
                "nilfs2\0"
                "nsfs\0"
                "ntfs\0"
                "ntfs3\0"
                "nullfs\0"
                "ocfs2\0"
                "openpromfs\0"
                "orangefs\0"
                "overlay\0"
                "pidfs\0"
                "pipefs\0"
                "ppc-cmm\0"
                "proc\0"
                "pstore\0"
                "pvfs2\0"
                "qnx4\0"
                "qnx6\0"
                "ramfs\0"
                "resctrl\0"
                "reiserfs\0"
                "rpc_pipefs\0"
                "secretmem\0"
                "securityfs\0"
                "selinuxfs\0"
                "shiftfs\0"
                "smackfs\0"
                "smb3\0"
                "smbfs\0"
                "sockfs\0"
                "squashfs\0"
                "sysfs\0"
                "tmpfs\0"
                "tracefs\0"
                "udf\0"
                "usbdevfs\0"
                "vboxsf\0"
                "vfat\0"
                "v9fs\0"
                "xenfs\0"
                "xfs\0"
                "z3fold\0"
                "zonefs\0"
                "zsmalloc\0"
        },
};
bool fs_in_group(const struct statfs *st, FilesystemGroups fs_group) {
        switch (fs_group) {
        case FILESYSTEM_SET_BASIC_API:
                return F_TYPE_EQUAL(st->f_type, CGROUP2_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, CGROUP_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, DEVPTS_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, MQUEUE_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, PROC_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, SYSFS_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, TMPFS_MAGIC);
        case FILESYSTEM_SET_ANONYMOUS:
                return F_TYPE_EQUAL(st->f_type, ANON_INODE_FS_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, PIPEFS_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, SOCKFS_MAGIC);
        case FILESYSTEM_SET_APPLICATION:
                return F_TYPE_EQUAL(st->f_type, AUTOFS_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, FUSE_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, OVERLAYFS_SUPER_MAGIC);
        case FILESYSTEM_SET_AUXILIARY_API:
                return F_TYPE_EQUAL(st->f_type, BINFMTFS_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, CONFIGFS_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, EFIVARFS_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, FUSE_CTL_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, HUGETLBFS_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, RPC_PIPEFS_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, SECURITYFS_MAGIC);
        case FILESYSTEM_SET_COMMON_BLOCK:
                return F_TYPE_EQUAL(st->f_type, BTRFS_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, EROFS_SUPER_MAGIC_V1)
                    || F_TYPE_EQUAL(st->f_type, EXFAT_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, EXT4_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, F2FS_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, ISOFS_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, MSDOS_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, NTFS3_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, SQUASHFS_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, UDF_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, XFS_SUPER_MAGIC);
        case FILESYSTEM_SET_HISTORICAL_BLOCK:
                return F_TYPE_EQUAL(st->f_type, EXT2_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, EXT3_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, MINIX2_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, MINIX2_SUPER_MAGIC2)
                    || F_TYPE_EQUAL(st->f_type, MINIX3_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, MINIX_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, MINIX_SUPER_MAGIC2);
        case FILESYSTEM_SET_NETWORK:
                return F_TYPE_EQUAL(st->f_type, AFS_FS_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, AFS_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, CEPH_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, CIFS_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, GFS2_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, NCP_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, NFS_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, OCFS2_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, ORANGEFS_DEVREQ_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, SMB2_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, SMB_SUPER_MAGIC);
        case FILESYSTEM_SET_PRIVILEGED_API:
                return F_TYPE_EQUAL(st->f_type, BPF_FS_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, DEBUGFS_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, PSTOREFS_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, TRACEFS_MAGIC);
        case FILESYSTEM_SET_SECURITY:
                return F_TYPE_EQUAL(st->f_type, AAFS_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, SELINUX_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, SMACK_MAGIC);
        case FILESYSTEM_SET_TEMPORARY:
                return F_TYPE_EQUAL(st->f_type, RAMFS_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, TMPFS_MAGIC);
        case FILESYSTEM_SET_KNOWN:
                return F_TYPE_EQUAL(st->f_type, AAFS_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, ADFS_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, AFFS_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, AFS_FS_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, AFS_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, ANON_INODE_FS_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, AUTOFS_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, BALLOON_KVM_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, BCACHEFS_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, BDEVFS_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, BINDERFS_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, BINFMTFS_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, BPF_FS_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, BTRFS_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, BTRFS_TEST_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, CEPH_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, CGROUP2_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, CGROUP_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, CIFS_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, CODA_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, CONFIGFS_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, CRAMFS_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, DAXFS_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, DEBUGFS_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, DEVMEM_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, DEVPTS_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, DMA_BUF_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, ECRYPTFS_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, EFIVARFS_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, EFS_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, EROFS_SUPER_MAGIC_V1)
                    || F_TYPE_EQUAL(st->f_type, EXFAT_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, EXT2_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, EXT3_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, EXT4_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, F2FS_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, FUSE_CTL_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, FUSE_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, GFS2_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, GUEST_MEMFD_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, HOSTFS_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, HPFS_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, HUGETLBFS_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, ISOFS_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, JFFS2_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, MINIX2_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, MINIX2_SUPER_MAGIC2)
                    || F_TYPE_EQUAL(st->f_type, MINIX3_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, MINIX_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, MINIX_SUPER_MAGIC2)
                    || F_TYPE_EQUAL(st->f_type, MQUEUE_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, MSDOS_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, NCP_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, NFS_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, NILFS_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, NSFS_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, NTFS3_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, NTFS_SB_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, NULL_FS_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, OCFS2_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, OPENPROM_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, ORANGEFS_DEVREQ_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, OVERLAYFS_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, PID_FS_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, PIPEFS_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, PPC_CMM_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, PROC_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, PSTOREFS_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, QNX4_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, QNX6_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, RAMFS_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, RDTGROUP_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, REISERFS_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, RPC_PIPEFS_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, SECRETMEM_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, SECURITYFS_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, SELINUX_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, SHIFTFS_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, SMACK_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, SMB2_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, SMB_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, SOCKFS_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, SQUASHFS_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, SYSFS_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, TMPFS_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, TRACEFS_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, UDF_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, USBDEVICE_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, V9FS_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, VBOXSF_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, XENFS_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, XFS_SUPER_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, Z3FOLD_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, ZONEFS_MAGIC)
                    || F_TYPE_EQUAL(st->f_type, ZSMALLOC_MAGIC);
        default: assert_not_reached();
        }
}
