/*
 * Copyright (C) 2014-2026 CZ.NIC
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations including
 * the two.
 */

#include <QHash> /* ::qHash */
#include <QJsonObject>
#include <QString>
#include <utility> /* ::std::swap */

#include "src/datovka_shared/compat/compiler.h" /* macroStdMove */
#include "src/datovka_shared/json/helper.h"
#include "src/json/message_id.h"

#define DLFT_TEST_ENV Isds::Type::BOOL_NULL
#define nullInt -1
static const QDateTime nullDateTime;

/*!
 * @brief PIMPL Json::MsgId1 class.
 */
class Json::MsgId1Private {
	//Q_DISABLE_COPY(MsgIdPrivate)
public:
	MsgId1Private(void)
	    : m_testEnv(DLFT_TEST_ENV), m_dmId(nullInt)
	{ }

	MsgId1Private &operator=(const MsgId1Private &other) Q_DECL_NOTHROW
	{
		m_testEnv = other.m_testEnv;
		m_dmId = other.m_dmId;

		return *this;
	}

	bool operator==(const MsgId1Private &other) const
	{
		return (m_testEnv == other.m_testEnv) &&
		    (m_dmId == other.m_dmId);
	}

	enum Isds::Type::NilBool m_testEnv; /*!< Test environment. */
	qint64 m_dmId; /*!< Message identifier. */
};

Json::MsgId1::MsgId1(void)
    : Object(),
    d_ptr(Q_NULLPTR)
{
}

Json::MsgId1::MsgId1(const MsgId1 &other)
    : Object(),
    d_ptr((other.d_func() != Q_NULLPTR) ? (new (::std::nothrow) MsgId1Private) : Q_NULLPTR)
{
	Q_D(MsgId1);
	if (d == Q_NULLPTR) {
		return;
	}

	*d = *other.d_func();
}

#ifdef Q_COMPILER_RVALUE_REFS
Json::MsgId1::MsgId1(MsgId1 &&other) Q_DECL_NOEXCEPT
    : Object(),
#if (QT_VERSION >= QT_VERSION_CHECK(5, 12, 0))
    d_ptr(other.d_ptr.release()) //d_ptr(::std::move(other.d_ptr))
#else /* < Qt-5.12 */
    d_ptr(other.d_ptr.take())
#endif /* >= Qt-5.12 */
{
}
#endif /* Q_COMPILER_RVALUE_REFS */

Json::MsgId1::~MsgId1(void)
{
}

Json::MsgId1::MsgId1(enum Isds::Type::NilBool testEnv, qint64 dmId)
    : Object(),
    d_ptr(new (::std::nothrow) MsgId1Private)
{
	Q_D(MsgId1);
	d->m_testEnv = testEnv;
	d->m_dmId = dmId;
}

/*!
 * @brief Ensures MsgId1Private presence.
 *
 * @note Returns if MsgId1Private could not be allocated.
 */
#define ensureMsgId1Private(_x_) \
	do { \
		if (Q_UNLIKELY(d_ptr == Q_NULLPTR)) { \
			MsgId1Private *p = new (::std::nothrow) MsgId1Private; \
			if (Q_UNLIKELY(p == Q_NULLPTR)) { \
				Q_ASSERT(0); \
				return _x_; \
			} \
			d_ptr.reset(p); \
		} \
	} while (0)

Json::MsgId1 &Json::MsgId1::operator=(const MsgId1 &other) Q_DECL_NOTHROW
{
	if (other.d_func() == Q_NULLPTR) {
		d_ptr.reset(Q_NULLPTR);
		return *this;
	}
	ensureMsgId1Private(*this);
	Q_D(MsgId1);

	*d = *other.d_func();

	return *this;
}

#ifdef Q_COMPILER_RVALUE_REFS
Json::MsgId1 &Json::MsgId1::operator=(MsgId1 &&other) Q_DECL_NOTHROW
{
	swap(*this, other);
	return *this;
}
#endif /* Q_COMPILER_RVALUE_REFS */

bool Json::MsgId1::operator==(const MsgId1 &other) const
{
	Q_D(const MsgId1);
	if ((d == Q_NULLPTR) && ((other.d_func() == Q_NULLPTR))) {
		return true;
	} else if ((d == Q_NULLPTR) || ((other.d_func() == Q_NULLPTR))) {
		return false;
	}

	return *d == *other.d_func();
}

bool Json::MsgId1::operator!=(const MsgId1 &other) const
{
	return !operator==(other);
}

bool Json::MsgId1::isNull(void) const
{
	Q_D(const MsgId1);
	return d == Q_NULLPTR;
}

bool Json::MsgId1::isValid(void) const
{
	return !isNull() && (testEnv() != Isds::Type::BOOL_NULL) && (dmId() >= 0);
}

enum Isds::Type::NilBool Json::MsgId1::testEnv(void) const
{
	Q_D(const MsgId1);
	if (Q_UNLIKELY(d == Q_NULLPTR)) {
		return DLFT_TEST_ENV;
	}

	return d->m_testEnv;
}

void Json::MsgId1::setTestEnv(enum Isds::Type::NilBool t)
{
	ensureMsgId1Private();
	Q_D(MsgId1);
	d->m_testEnv = t;
}

qint64 Json::MsgId1::dmId(void) const
{
	Q_D(const MsgId1);
	if (Q_UNLIKELY(d == Q_NULLPTR)) {
		return nullInt;
	}

	return d->m_dmId;
}

void Json::MsgId1::setDmId(qint64 di)
{
	ensureMsgId1Private();
	Q_D(MsgId1);
	d->m_dmId = di;
}

Json::MsgId1 Json::MsgId1::fromJson(const QByteArray &json, bool *ok)
{
	QJsonObject jsonObj;
	if (!Helper::readRootObject(json, jsonObj)) {
		if (ok != Q_NULLPTR) {
			*ok = false;
		}
		return MsgId1();
	}

	return fromJsonVal(jsonObj, ok);
}

static const QString keyTestEnv("testEnv");
static const QString keyMessageId("messageId");
static const QString keyDeliveryTime("deliveryTime");

Json::MsgId1 Json::MsgId1::fromJsonVal(const QJsonValue &jsonVal, bool *ok)
{
	MsgId1 smi;

	if (Q_UNLIKELY(!jsonVal.isObject())) {
		goto fail;
	}

	{
		const QJsonObject jsonObj = jsonVal.toObject();
		{
			enum Isds::Type::NilBool valNilBool = Isds::Type::BOOL_NULL;
			if (Q_UNLIKELY(!Helper::readNilBool(jsonObj,
			        keyTestEnv, valNilBool, Helper::ACCEPT_NULL))) {
				goto fail;
			}
			smi.setTestEnv(valNilBool);
		}
		{
			qint64 valInt = 0;
			if (Q_UNLIKELY(!Helper::readQint64String(jsonObj,
			        keyMessageId, valInt, Helper::ACCEPT_VALID))) {
				goto fail;
			}
			smi.setDmId(valInt);
		}
	}

	if (ok != Q_NULLPTR) {
		*ok = true;
	}
	return smi;

fail:
	if (ok != Q_NULLPTR) {
		*ok = false;
	}
	return MsgId1();
}

bool Json::MsgId1::toJsonVal(QJsonValue &jsonVal) const
{
	QJsonObject jsonObj;

	jsonObj.insert(keyTestEnv, (testEnv() != Isds::Type::BOOL_NULL) ?
	    QJsonValue(testEnv() == Isds::Type::BOOL_TRUE) : QJsonValue());
	jsonObj.insert(keyMessageId, (dmId() >= 0) ? QString::number(dmId()) : QJsonValue());

	jsonVal = jsonObj;
	return true;
}

void Json::swap(MsgId1 &first, MsgId1 &second) Q_DECL_NOTHROW
{
	using ::std::swap;
	swap(first.d_ptr, second.d_ptr);
}

Json::MsgId1List::MsgId1List(void)
    : Object(),
    QList<MsgId1>()
{
}

Json::MsgId1List::MsgId1List(const QList<MsgId1> &other)
    : Object(),
    QList<MsgId1>(other)
{
}

#ifdef Q_COMPILER_RVALUE_REFS
Json::MsgId1List::MsgId1List(QList<MsgId1> &&other)
    : Object(),
    QList<MsgId1>(::std::move(other))
{
}
#endif /* Q_COMPILER_RVALUE_REFS */

Json::MsgId1List Json::MsgId1List::fromJson(const QByteArray &json,
    bool *ok)
{
	QJsonArray jsonArr;
	if (!Helper::readRootArray(json, jsonArr)) {
		if (ok != Q_NULLPTR) {
			*ok = false;
		}
		return MsgId1List();
	}

	return fromJsonVal(jsonArr, ok);
}

Json::MsgId1List Json::MsgId1List::fromJsonVal(const QJsonValue &jsonVal,
    bool *ok)
{
	MsgId1List mil;

	if (Q_UNLIKELY(!jsonVal.isArray())) {
		goto fail;
	}

	{
		const QJsonArray jsonArr = jsonVal.toArray();
		for (const QJsonValue &v : jsonArr) {
			bool iOk = false;
			mil.append(MsgId1::fromJsonVal(v, &iOk));
			if (Q_UNLIKELY(!iOk)) {
				goto fail;
			}
		}
	}

	if (ok != Q_NULLPTR) {
		*ok = true;
	}
	return mil;

fail:
	if (ok != Q_NULLPTR) {
		*ok = false;
	}
	return MsgId1List();
}

bool Json::MsgId1List::toJsonVal(QJsonValue &jsonVal) const
{
	QJsonArray arr;

	for (const MsgId1 &mi : *this) {
		QJsonValue v;
		if (Q_UNLIKELY(!mi.toJsonVal(v))) {
			goto fail;
		}
		arr.append(v);
	}

	jsonVal = arr;
	return true;

fail:
	jsonVal = QJsonArray();
	return false;
}

/*!
 * @brief PIMPL Json::MsgId2 class.
 */
class Json::MsgId2Private {
	//Q_DISABLE_COPY(MsgIdPrivate)
public:
	MsgId2Private(void)
	    : m_testEnv(DLFT_TEST_ENV), m_dmId(nullInt), m_deliveryTime()
	{ }

	MsgId2Private &operator=(const MsgId2Private &other) Q_DECL_NOTHROW
	{
		m_testEnv = other.m_testEnv;
		m_dmId = other.m_dmId;
		m_deliveryTime = other.m_deliveryTime;

		return *this;
	}

	bool operator==(const MsgId2Private &other) const
	{
		return (m_testEnv == other.m_testEnv)
		    && (m_dmId == other.m_dmId)
		    && (m_deliveryTime == other.m_deliveryTime);
	}

	enum Isds::Type::NilBool m_testEnv; /*!< Test environment. */
	qint64 m_dmId; /*!< Message identifier. */
	QDateTime m_deliveryTime; /*!< Message delivery time. */
};

Json::MsgId2::MsgId2(void)
    : Object(),
    d_ptr(Q_NULLPTR)
{
}

Json::MsgId2::MsgId2(const MsgId2 &other)
    : Object(),
    d_ptr((other.d_func() != Q_NULLPTR) ? (new (::std::nothrow) MsgId2Private) : Q_NULLPTR)
{
	Q_D(MsgId2);
	if (d == Q_NULLPTR) {
		return;
	}

	*d = *other.d_func();
}

#ifdef Q_COMPILER_RVALUE_REFS
Json::MsgId2::MsgId2(MsgId2 &&other) Q_DECL_NOEXCEPT
    : Object(),
#if (QT_VERSION >= QT_VERSION_CHECK(5, 12, 0))
    d_ptr(other.d_ptr.release()) //d_ptr(::std::move(other.d_ptr))
#else /* < Qt-5.12 */
    d_ptr(other.d_ptr.take())
#endif /* >= Qt-5.12 */
{
}
#endif /* Q_COMPILER_RVALUE_REFS */

Json::MsgId2::~MsgId2(void)
{
}

Json::MsgId2::MsgId2(enum Isds::Type::NilBool testEnv, qint64 dmId,
    const QDateTime &deliveryTime)
    : Object(),
    d_ptr(new (::std::nothrow) MsgId2Private)
{
	Q_D(MsgId2);
	d->m_testEnv = testEnv;
	d->m_dmId = dmId;
	d->m_deliveryTime = deliveryTime;
}

#ifdef Q_COMPILER_RVALUE_REFS
Json::MsgId2::MsgId2(enum Isds::Type::NilBool testEnv, qint64 dmId,
    QDateTime &&deliveryTime)
    : Object(),
    d_ptr(new (::std::nothrow) MsgId2Private)
{
	Q_D(MsgId2);
	d->m_testEnv = testEnv;
	d->m_dmId = dmId;
	d->m_deliveryTime = ::std::move(deliveryTime);
}
#endif /* Q_COMPILER_RVALUE_REFS */

/*!
 * @brief Ensures MsgId2Private presence.
 *
 * @note Returns if MsgId2Private could not be allocated.
 */
#define ensureMsgId2Private(_x_) \
	do { \
		if (Q_UNLIKELY(d_ptr == Q_NULLPTR)) { \
			MsgId2Private *p = new (::std::nothrow) MsgId2Private; \
			if (Q_UNLIKELY(p == Q_NULLPTR)) { \
				Q_ASSERT(0); \
				return _x_; \
			} \
			d_ptr.reset(p); \
		} \
	} while (0)

Json::MsgId2 &Json::MsgId2::operator=(const MsgId2 &other) Q_DECL_NOTHROW
{
	if (other.d_func() == Q_NULLPTR) {
		d_ptr.reset(Q_NULLPTR);
		return *this;
	}
	ensureMsgId2Private(*this);
	Q_D(MsgId2);

	*d = *other.d_func();

	return *this;
}

#ifdef Q_COMPILER_RVALUE_REFS
Json::MsgId2 &Json::MsgId2::operator=(MsgId2 &&other) Q_DECL_NOTHROW
{
	swap(*this, other);
	return *this;
}
#endif /* Q_COMPILER_RVALUE_REFS */

bool Json::MsgId2::operator==(const MsgId2 &other) const
{
	Q_D(const MsgId2);
	if ((d == Q_NULLPTR) && ((other.d_func() == Q_NULLPTR))) {
		return true;
	} else if ((d == Q_NULLPTR) || ((other.d_func() == Q_NULLPTR))) {
		return false;
	}

	return *d == *other.d_func();
}

bool Json::MsgId2::operator!=(const MsgId2 &other) const
{
	return !operator==(other);
}

bool Json::MsgId2::isNull(void) const
{
	Q_D(const MsgId2);
	return d == Q_NULLPTR;
}

bool Json::MsgId2::isValid(void) const
{
	return !isNull()
	    && (testEnv() != Isds::Type::BOOL_NULL)
	    && (dmId() >= 0)
	    && (deliveryTime().isValid());
}

enum Isds::Type::NilBool Json::MsgId2::testEnv(void) const
{
	Q_D(const MsgId2);
	if (Q_UNLIKELY(d == Q_NULLPTR)) {
		return DLFT_TEST_ENV;
	}

	return d->m_testEnv;
}

void Json::MsgId2::setTestEnv(enum Isds::Type::NilBool t)
{
	ensureMsgId2Private();
	Q_D(MsgId2);
	d->m_testEnv = t;
}

qint64 Json::MsgId2::dmId(void) const
{
	Q_D(const MsgId2);
	if (Q_UNLIKELY(d == Q_NULLPTR)) {
		return nullInt;
	}

	return d->m_dmId;
}

void Json::MsgId2::setDmId(qint64 di)
{
	ensureMsgId2Private();
	Q_D(MsgId2);
	d->m_dmId = di;
}

const QDateTime &Json::MsgId2::deliveryTime(void) const
{
	Q_D(const MsgId2);
	if (Q_UNLIKELY(d == Q_NULLPTR)) {
		return nullDateTime;
	}

	return d->m_deliveryTime;
}

void Json::MsgId2::setDeliveryTime(const QDateTime &dt)
{
	ensureMsgId2Private();
	Q_D(MsgId2);
	d->m_deliveryTime = dt;
}

#ifdef Q_COMPILER_RVALUE_REFS
void Json::MsgId2::setDeliveryTime(QDateTime &&dt)
{
	ensureMsgId2Private();
	Q_D(MsgId2);
	d->m_deliveryTime = ::std::move(dt);
}
#endif /* Q_COMPILER_RVALUE_REFS */

Json::MsgId2 Json::MsgId2::fromJson(const QByteArray &json, bool *ok)
{
	QJsonObject jsonObj;
	if (!Helper::readRootObject(json, jsonObj)) {
		if (ok != Q_NULLPTR) {
			*ok = false;
		}
		return MsgId2();
	}

	return fromJsonVal(jsonObj, ok);
}

Json::MsgId2 Json::MsgId2::fromJsonVal(const QJsonValue &jsonVal, bool *ok)
{
	MsgId2 smi;

	if (Q_UNLIKELY(!jsonVal.isObject())) {
		goto fail;
	}

	{
		const QJsonObject jsonObj = jsonVal.toObject();
		{
			enum Isds::Type::NilBool valNilBool = Isds::Type::BOOL_NULL;
			if (Q_UNLIKELY(!Helper::readNilBool(jsonObj,
			        keyTestEnv, valNilBool, Helper::ACCEPT_NULL))) {
				goto fail;
			}
			smi.setTestEnv(valNilBool);
		}
		{
			qint64 valInt = 0;
			if (Q_UNLIKELY(!Helper::readQint64String(jsonObj,
			        keyMessageId, valInt, Helper::ACCEPT_VALID))) {
				goto fail;
			}
			smi.setDmId(valInt);
		}
		{
			QDateTime valDateTime;
			if (Q_UNLIKELY(!Helper::readQDateTimeString(jsonObj,
			        keyDeliveryTime, valDateTime, Helper::ACCEPT_NULL))) {
				goto fail;
			}
			smi.setDeliveryTime(macroStdMove(valDateTime));
		}
	}

	if (ok != Q_NULLPTR) {
		*ok = true;
	}
	return smi;

fail:
	if (ok != Q_NULLPTR) {
		*ok = false;
	}
	return MsgId2();
}

bool Json::MsgId2::toJsonVal(QJsonValue &jsonVal) const
{
	QJsonObject jsonObj;

	jsonObj.insert(keyTestEnv, (testEnv() != Isds::Type::BOOL_NULL) ?
	    QJsonValue(testEnv() == Isds::Type::BOOL_TRUE) : QJsonValue());
	jsonObj.insert(keyMessageId, (dmId() >= 0) ? QString::number(dmId()) : QJsonValue());
	jsonObj.insert(keyDeliveryTime, deliveryTime().isValid() ? deliveryTime().toUTC().toString(Qt::ISODate) : QJsonValue());

	jsonVal = jsonObj;
	return true;
}

void Json::swap(MsgId2 &first, MsgId2 &second) Q_DECL_NOTHROW
{
	using ::std::swap;
	swap(first.d_ptr, second.d_ptr);
}

Json::MsgId2List::MsgId2List(void)
    : Object(),
    QList<MsgId2>()
{
}

Json::MsgId2List::MsgId2List(const QList<MsgId2> &other)
    : Object(),
    QList<MsgId2>(other)
{
}

#ifdef Q_COMPILER_RVALUE_REFS
Json::MsgId2List::MsgId2List(QList<MsgId2> &&other)
    : Object(),
    QList<MsgId2>(::std::move(other))
{
}
#endif /* Q_COMPILER_RVALUE_REFS */

Json::MsgId2List Json::MsgId2List::fromJson(const QByteArray &json,
    bool *ok)
{
	QJsonArray jsonArr;
	if (!Helper::readRootArray(json, jsonArr)) {
		if (ok != Q_NULLPTR) {
			*ok = false;
		}
		return MsgId2List();
	}

	return fromJsonVal(jsonArr, ok);
}

Json::MsgId2List Json::MsgId2List::fromJsonVal(const QJsonValue &jsonVal,
    bool *ok)
{
	MsgId2List mil;

	if (Q_UNLIKELY(!jsonVal.isArray())) {
		goto fail;
	}

	{
		const QJsonArray jsonArr = jsonVal.toArray();
		for (const QJsonValue &v : jsonArr) {
			bool iOk = false;
			mil.append(MsgId2::fromJsonVal(v, &iOk));
			if (Q_UNLIKELY(!iOk)) {
				goto fail;
			}
		}
	}

	if (ok != Q_NULLPTR) {
		*ok = true;
	}
	return mil;

fail:
	if (ok != Q_NULLPTR) {
		*ok = false;
	}
	return MsgId2List();
}

bool Json::MsgId2List::toJsonVal(QJsonValue &jsonVal) const
{
	QJsonArray arr;

	for (const MsgId2 &mi : *this) {
		QJsonValue v;
		if (Q_UNLIKELY(!mi.toJsonVal(v))) {
			goto fail;
		}
		arr.append(v);
	}

	jsonVal = arr;
	return true;

fail:
	jsonVal = QJsonArray();
	return false;
}

#if (QT_VERSION >= QT_VERSION_CHECK(6, 0, 0))
size_t Json::qHash(const MsgId1 &key, size_t seed)
#else /* < Qt-6.0 */
uint Json::qHash(const MsgId1 &key, uint seed)
#endif /* >= Qt-6.0 */
{
	return ::qHash(((int)key.testEnv()) ^ key.dmId(), seed);
}

#if (QT_VERSION >= QT_VERSION_CHECK(6, 0, 0))
size_t Json::qHash(const MsgId2 &key, size_t seed)
#else /* < Qt-6.0 */
uint Json::qHash(const MsgId2 &key, uint seed)
#endif /* >= Qt-6.0 */
{
	return ::qHash(((int)key.testEnv()) ^ key.dmId() ^ ::qHash(key.deliveryTime(), seed), seed);
}
