{-# LANGUAGE ImplicitParams, RankNTypes, TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- The @GSettings@ class provides a convenient API for storing and retrieving
-- application settings.
-- 
-- Reads and writes can be considered to be non-blocking.  Reading
-- settings with @GSettings@ is typically extremely fast: on
-- approximately the same order of magnitude (but slower than) a
-- t'GHashTable' lookup.  Writing settings is also extremely fast in
-- terms of time to return to your application, but can be extremely expensive
-- for other threads and other processes.  Many settings backends
-- (including dconf) have lazy initialisation which means in the common
-- case of the user using their computer without modifying any settings
-- a lot of work can be avoided.  For dconf, the D-Bus service doesn’t
-- even need to be started in this case.  For this reason, you should
-- only ever modify @GSettings@ keys in response to explicit user action.
-- Particular care should be paid to ensure that modifications are not
-- made during startup — for example, when setting the initial value
-- of preferences widgets.  The built-in 'GI.Gio.Objects.Settings.settingsBind'
-- functionality is careful not to write settings in response to notify signals
-- as a result of modifications that it makes to widgets.
-- 
-- When creating a @GSettings@ instance, you have to specify a schema
-- that describes the keys in your settings and their types and default
-- values, as well as some other information.
-- 
-- Normally, a schema has a fixed path that determines where the settings
-- are stored in the conceptual global tree of settings. However, schemas
-- can also be ‘<http://developer.gnome.org/gio/stable/#relocatable-schemas relocatable>’, i.e. not equipped with
-- a fixed path. This is
-- useful e.g. when the schema describes an ‘account’, and you want to be
-- able to store a arbitrary number of accounts.
-- 
-- Paths must start with and end with a forward slash character (@\/@)
-- and must not contain two sequential slash characters.  Paths should
-- be chosen based on a domain name associated with the program or
-- library to which the settings belong.  Examples of paths are
-- @\/org\/gtk\/settings\/file-chooser\/@ and @\/ca\/desrt\/dconf-editor\/@.
-- Paths should not start with @\/apps\/@, @\/desktop\/@ or @\/system\/@ as
-- they often did in GConf.
-- 
-- Unlike other configuration systems (like GConf), GSettings does not
-- restrict keys to basic types like strings and numbers. GSettings stores
-- values as t'GVariant', and allows any [type/@gLib@/.VariantType] for
-- keys. Key names are restricted to lowercase characters, numbers and @-@.
-- Furthermore, the names must begin with a lowercase character, must not end
-- with a @-@, and must not contain consecutive dashes.
-- 
-- Similar to GConf, the default values in GSettings schemas can be
-- localized, but the localized values are stored in gettext catalogs
-- and looked up with the domain that is specified in the
-- @gettext-domain@ attribute of the @\<schemalist>@ or @\<schema>@
-- elements and the category that is specified in the @l10n@ attribute of
-- the @\<default>@ element. The string which is translated includes all text in
-- the @\<default>@ element, including any surrounding quotation marks.
-- 
-- The @l10n@ attribute must be set to @messages@ or @time@, and sets the
-- <https://www.gnu.org/software/gettext/manual/html_node/Aspects.html#index-locale-categories-1 locale category for
-- translation>.
-- The @messages@ category should be used by default; use @time@ for
-- translatable date or time formats. A translation comment can be added as an
-- XML comment immediately above the @\<default>@ element — it is recommended to
-- add these comments to aid translators understand the meaning and
-- implications of the default value. An optional translation @context@
-- attribute can be set on the @\<default>@ element to disambiguate multiple
-- defaults which use the same string.
-- 
-- For example:
-- 
-- === /xml code/
-- > <!-- Translators: A list of words which are not allowed to be typed, in
-- >      GVariant serialization syntax.
-- >      See: https://developer.gnome.org/glib/stable/gvariant-text.html -->
-- > <default l10n='messages' context='Banned words'>['bad', 'words']</default>
-- 
-- 
-- Translations of default values must remain syntactically valid serialized
-- t'GVariant's (e.g. retaining any surrounding quotation marks) or
-- runtime errors will occur.
-- 
-- GSettings uses schemas in a compact binary form that is created
-- by the <http://developer.gnome.org/gio/stable/glib-compile-schemas.html `glib-compile-schemas`>
-- utility. The input is a schema description in an XML format.
-- 
-- A DTD for the gschema XML format can be found here:
-- <https://gitlab.gnome.org/GNOME/glib/-/blob/HEAD/gio/gschema.dtd gschema.dtd>
-- 
-- The <http://developer.gnome.org/gio/stable/glib-compile-schemas.html `glib-compile-schemas`> tool expects schema
-- files to have the extension @.gschema.xml@.
-- 
-- At runtime, schemas are identified by their ID (as specified in the
-- @id@ attribute of the @\<schema>@ element). The convention for schema
-- IDs is to use a dotted name, similar in style to a D-Bus bus name,
-- e.g. @org.gnome.SessionManager@. In particular, if the settings are
-- for a specific service that owns a D-Bus bus name, the D-Bus bus name
-- and schema ID should match. For schemas which deal with settings not
-- associated with one named application, the ID should not use
-- StudlyCaps, e.g. @org.gnome.font-rendering@.
-- 
-- In addition to t'GVariant' types, keys can have types that have
-- enumerated types. These can be described by a @\<choice>@,
-- @\<enum>@ or @\<flags>@ element, as seen in the
-- second example below. The underlying type of such a key
-- is string, but you can use 'GI.Gio.Objects.Settings.settingsGetEnum',
-- 'GI.Gio.Objects.Settings.settingsSetEnum', 'GI.Gio.Objects.Settings.settingsGetFlags',
-- 'GI.Gio.Objects.Settings.settingsSetFlags' access the numeric values corresponding to
-- the string value of enum and flags keys.
-- 
-- An example for default value:
-- 
-- === /xml code/
-- ><schemalist>
-- >  <schema id="org.gtk.Test" path="/org/gtk/Test/" gettext-domain="test">
-- >
-- >    <key name="greeting" type="s">
-- >      <default l10n="messages">"Hello, earthlings"</default>
-- >      <summary>A greeting</summary>
-- >      <description>
-- >        Greeting of the invading martians
-- >      </description>
-- >    </key>
-- >
-- >    <key name="box" type="(ii)">
-- >      <default>(20,30)</default>
-- >    </key>
-- >
-- >    <key name="empty-string" type="s">
-- >      <default>""</default>
-- >      <summary>Empty strings have to be provided in GVariant form</summary>
-- >    </key>
-- >
-- >  </schema>
-- ></schemalist>
-- 
-- 
-- An example for ranges, choices and enumerated types:
-- 
-- === /xml code/
-- ><schemalist>
-- >
-- >  <enum id="org.gtk.Test.myenum">
-- >    <value nick="first" value="1"/>
-- >    <value nick="second" value="2"/>
-- >  </enum>
-- >
-- >  <flags id="org.gtk.Test.myflags">
-- >    <value nick="flag1" value="1"/>
-- >    <value nick="flag2" value="2"/>
-- >    <value nick="flag3" value="4"/>
-- >  </flags>
-- >
-- >  <schema id="org.gtk.Test">
-- >
-- >    <key name="key-with-range" type="i">
-- >      <range min="1" max="100"/>
-- >      <default>10</default>
-- >    </key>
-- >
-- >    <key name="key-with-choices" type="s">
-- >      <choices>
-- >        <choice value='Elisabeth'/>
-- >        <choice value='Annabeth'/>
-- >        <choice value='Joe'/>
-- >      </choices>
-- >      <aliases>
-- >        <alias value='Anna' target='Annabeth'/>
-- >        <alias value='Beth' target='Elisabeth'/>
-- >      </aliases>
-- >      <default>'Joe'</default>
-- >    </key>
-- >
-- >    <key name='enumerated-key' enum='org.gtk.Test.myenum'>
-- >      <default>'first'</default>
-- >    </key>
-- >
-- >    <key name='flags-key' flags='org.gtk.Test.myflags'>
-- >      <default>["flag1","flag2"]</default>
-- >    </key>
-- >  </schema>
-- ></schemalist>
-- 
-- 
-- == Vendor overrides
-- 
-- Default values are defined in the schemas that get installed by
-- an application. Sometimes, it is necessary for a vendor or distributor
-- to adjust these defaults. Since patching the XML source for the schema
-- is inconvenient and error-prone,
-- <http://developer.gnome.org/gio/stable/glib-compile-schemas.html `glib-compile-schemas`> reads so-called ‘vendor
-- override’ files. These are keyfiles in the same directory as the XML
-- schema sources which can override default values. The schema ID serves
-- as the group name in the key file, and the values are expected in
-- serialized t'GVariant' form, as in the following example:
-- 
-- 	
-- >[org.gtk.Example]
-- >key1='string'
-- >key2=1.5
-- 
-- 
-- @glib-compile-schemas@ expects schema files to have the extension
-- @.gschema.override@.
-- 
-- == Delay-apply mode
-- 
-- By default, values set on a t'GI.Gio.Objects.Settings.Settings' instance immediately start
-- to be written to the backend (although these writes may not complete by the
-- time that t'GI.Gio.Objects.Settings.Settings'.@/set/@()) returns; see 'GI.Gio.Objects.Settings.settingsSync').
-- 
-- In order to allow groups of settings to be changed simultaneously and
-- atomically, GSettings also supports a ‘delay-apply’ mode. In this mode,
-- updated values are kept locally in the t'GI.Gio.Objects.Settings.Settings' instance until
-- they are explicitly applied by calling 'GI.Gio.Objects.Settings.settingsApply'.
-- 
-- For example, this could be useful for a preferences dialog where the
-- preferences all need to be applied simultaneously when the user clicks ‘Save’.
-- 
-- Switching a t'GI.Gio.Objects.Settings.Settings' instance to ‘delay-apply’ mode is a one-time
-- irreversible operation: from that point onwards, *all* changes made to that
-- t'GI.Gio.Objects.Settings.Settings' have to be explicitly applied by calling
-- 'GI.Gio.Objects.Settings.settingsApply'. The ‘delay-apply’ mode is also propagated to any
-- child settings objects subsequently created using
-- 'GI.Gio.Objects.Settings.settingsGetChild'.
-- 
-- At any point, the set of unapplied changes can be queried using
-- [Settings:hasUnapplied]("GI.Gio.Objects.Settings#g:attr:hasUnapplied"), and discarded by calling
-- 'GI.Gio.Objects.Settings.settingsRevert'.
-- 
-- == Binding
-- 
-- A very convenient feature of GSettings lets you bind t'GI.GObject.Objects.Object.Object'
-- properties directly to settings, using 'GI.Gio.Objects.Settings.settingsBind'. Once a
-- t'GI.GObject.Objects.Object.Object' property has been bound to a setting, changes on
-- either side are automatically propagated to the other side. GSettings handles
-- details like mapping between t'GI.GObject.Objects.Object.Object' and t'GVariant'
-- types, and preventing infinite cycles.
-- 
-- This makes it very easy to hook up a preferences dialog to the
-- underlying settings. To make this even more convenient, GSettings
-- looks for a boolean property with the name @sensitivity@ and
-- automatically binds it to the writability of the bound setting.
-- If this ‘magic’ gets in the way, it can be suppressed with the
-- @G_SETTINGS_BIND_NO_SENSITIVITY@ flag.
-- 
-- == Relocatable schemas
-- 
-- A relocatable schema is one with no @path@ attribute specified on its
-- @\<schema>@ element. By using 'GI.Gio.Objects.Settings.settingsNewWithPath', a @GSettings@
-- object can be instantiated for a relocatable schema, assigning a path to the
-- instance. Paths passed to 'GI.Gio.Objects.Settings.settingsNewWithPath' will typically be
-- constructed dynamically from a constant prefix plus some form of instance
-- identifier; but they must still be valid GSettings paths. Paths could also
-- be constant and used with a globally installed schema originating from a
-- dependency library.
-- 
-- For example, a relocatable schema could be used to store geometry information
-- for different windows in an application. If the schema ID was
-- @org.foo.MyApp.Window@, it could be instantiated for paths
-- @\/org\/foo\/MyApp\/main\/@, @\/org\/foo\/MyApp\/document-1\/@,
-- @\/org\/foo\/MyApp\/document-2\/@, etc. If any of the paths are well-known
-- they can be specified as @\<child>@ elements in the parent schema, e.g.:
-- 
-- === /xml code/
-- ><schema id="org.foo.MyApp" path="/org/foo/MyApp/">
-- >  <child name="main" schema="org.foo.MyApp.Window"/>
-- ></schema>
-- 
-- 
-- == Build system integration
-- 
-- === Meson
-- 
-- GSettings is natively supported by Meson’s <https://mesonbuild.com/Gnome-module.html GNOME module>.
-- 
-- You can install the schemas as any other data file:
-- 
-- 
-- 	
-- >install_data(
-- >  'org.foo.MyApp.gschema.xml',
-- >  install_dir: get_option('datadir') / 'glib-2.0/schemas',
-- >)
-- 
-- 
-- You can use @gnome.post_install()@ function to compile the schemas on
-- installation:
-- 
-- 
-- 	
-- >gnome = import('gnome')
-- >gnome.post_install(
-- >  glib_compile_schemas: true,
-- >)
-- 
-- 
-- If an enumerated type defined in a C header file is to be used in a GSettings
-- schema, it can either be defined manually using an @\<enum>@ element in the
-- schema XML, or it can be extracted automatically from the C header. This
-- approach is preferred, as it ensures the two representations are always
-- synchronised. To do so, you will need to use the @gnome.mkenums()@ function
-- with the following templates:
-- 
-- 
-- 	
-- >schemas_enums = gnome.mkenums('org.foo.MyApp.enums.xml',
-- >  comments: '<!-- @comment@ -->',
-- >  fhead: '<schemalist>',
-- >  vhead: '  <@type@ id="org.foo.MyApp.@EnumName@">',
-- >  vprod: '    <value nick="@valuenick@" value="@valuenum@"/>',
-- >  vtail: '  </@type@>',
-- >  ftail: '</schemalist>',
-- >  sources: enum_sources,
-- >  install_header: true,
-- >  install_dir: get_option('datadir') / 'glib-2.0/schemas',
-- >)
-- 
-- 
-- It is recommended to validate your schemas as part of the test suite for
-- your application:
-- 
-- 
-- 	
-- >test('validate-schema',
-- >  find_program('glib-compile-schemas'),
-- >  args: ['--strict', '--dry-run', meson.current_source_dir()],
-- >)
-- 
-- 
-- If your application allows running uninstalled, you should also use the
-- @gnome.compile_schemas()@ function to compile the schemas in the current
-- build directory:
-- 
-- 
-- 	
-- >gnome.compile_schemas()
-- 
-- 
-- === Autotools
-- 
-- GSettings comes with autotools integration to simplify compiling and
-- installing schemas. To add GSettings support to an application, add the
-- following to your @configure.ac@:
-- 
-- 	
-- >GLIB_GSETTINGS
-- 
-- 
-- In the appropriate @Makefile.am@, use the following snippet to compile and
-- install the named schema:
-- 
-- 	
-- >gsettings_SCHEMAS = org.foo.MyApp.gschema.xml
-- >EXTRA_DIST = $(gsettings_SCHEMAS)
-- >
-- >@GSETTINGS_RULES@
-- 
-- 
-- If an enumerated type defined in a C header file is to be used in a GSettings
-- schema, it can either be defined manually using an @\<enum>@ element in the
-- schema XML, or it can be extracted automatically from the C header. This
-- approach is preferred, as it ensures the two representations are always
-- synchronised. To do so, add the following to the relevant @Makefile.am@:
-- 
-- 	
-- >gsettings_ENUM_NAMESPACE = org.foo.MyApp
-- >gsettings_ENUM_FILES = my-app-enums.h my-app-misc.h
-- 
-- 
-- @gsettings_ENUM_NAMESPACE@ specifies the schema namespace for the enum files,
-- which are specified in @gsettings_ENUM_FILES@. This will generate a
-- @org.foo.MyApp.enums.xml@ file containing the extracted enums, which will be
-- automatically included in the schema compilation, install and uninstall
-- rules. It should not be committed to version control or included in
-- @EXTRA_DIST@.
-- 
-- == Localization
-- 
-- No changes are needed to the build system to mark a schema XML file for
-- translation. Assuming it sets the @gettext-domain@ attribute, a schema may
-- be marked for translation by adding it to @POTFILES.in@, assuming gettext
-- 0.19 or newer is in use (the preferred method for translation):
-- 
-- 	
-- >data/org.foo.MyApp.gschema.xml
-- 
-- 
-- Alternatively, if intltool 0.50.1 is in use:
-- 
-- 	
-- >[type: gettext/gsettings]data/org.foo.MyApp.gschema.xml
-- 
-- 
-- GSettings will use gettext to look up translations for the @\<summary>@ and
-- @\<description>@ elements, and also any @\<default>@ elements which have a
-- @l10n@ attribute set.
-- 
-- Translations __must not__ be included in the @.gschema.xml@ file by the build
-- system, for example by using a rule to generate the XML file from a template.

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gio.Objects.Settings
    ( 

-- * Exported types
    Settings(..)                            ,
    IsSettings                              ,
    toSettings                              ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [apply]("GI.Gio.Objects.Settings#g:method:apply"), [bind]("GI.Gio.Objects.Settings#g:method:bind"), [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [bindWithMapping]("GI.Gio.Objects.Settings#g:method:bindWithMapping"), [bindWritable]("GI.Gio.Objects.Settings#g:method:bindWritable"), [createAction]("GI.Gio.Objects.Settings#g:method:createAction"), [delay]("GI.Gio.Objects.Settings#g:method:delay"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [isWritable]("GI.Gio.Objects.Settings#g:method:isWritable"), [listChildren]("GI.Gio.Objects.Settings#g:method:listChildren"), [listKeys]("GI.Gio.Objects.Settings#g:method:listKeys"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [rangeCheck]("GI.Gio.Objects.Settings#g:method:rangeCheck"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [reset]("GI.Gio.Objects.Settings#g:method:reset"), [revert]("GI.Gio.Objects.Settings#g:method:revert"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getBoolean]("GI.Gio.Objects.Settings#g:method:getBoolean"), [getChild]("GI.Gio.Objects.Settings#g:method:getChild"), [getData]("GI.GObject.Objects.Object#g:method:getData"), [getDefaultValue]("GI.Gio.Objects.Settings#g:method:getDefaultValue"), [getDouble]("GI.Gio.Objects.Settings#g:method:getDouble"), [getEnum]("GI.Gio.Objects.Settings#g:method:getEnum"), [getFlags]("GI.Gio.Objects.Settings#g:method:getFlags"), [getHasUnapplied]("GI.Gio.Objects.Settings#g:method:getHasUnapplied"), [getInt]("GI.Gio.Objects.Settings#g:method:getInt"), [getInt64]("GI.Gio.Objects.Settings#g:method:getInt64"), [getMapped]("GI.Gio.Objects.Settings#g:method:getMapped"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata"), [getRange]("GI.Gio.Objects.Settings#g:method:getRange"), [getString]("GI.Gio.Objects.Settings#g:method:getString"), [getStrv]("GI.Gio.Objects.Settings#g:method:getStrv"), [getUint]("GI.Gio.Objects.Settings#g:method:getUint"), [getUint64]("GI.Gio.Objects.Settings#g:method:getUint64"), [getUserValue]("GI.Gio.Objects.Settings#g:method:getUserValue"), [getValue]("GI.Gio.Objects.Settings#g:method:getValue").
-- 
-- ==== Setters
-- [setBoolean]("GI.Gio.Objects.Settings#g:method:setBoolean"), [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setDouble]("GI.Gio.Objects.Settings#g:method:setDouble"), [setEnum]("GI.Gio.Objects.Settings#g:method:setEnum"), [setFlags]("GI.Gio.Objects.Settings#g:method:setFlags"), [setInt]("GI.Gio.Objects.Settings#g:method:setInt"), [setInt64]("GI.Gio.Objects.Settings#g:method:setInt64"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty"), [setString]("GI.Gio.Objects.Settings#g:method:setString"), [setStrv]("GI.Gio.Objects.Settings#g:method:setStrv"), [setUint]("GI.Gio.Objects.Settings#g:method:setUint"), [setUint64]("GI.Gio.Objects.Settings#g:method:setUint64"), [setValue]("GI.Gio.Objects.Settings#g:method:setValue").

#if defined(ENABLE_OVERLOADING)
    ResolveSettingsMethod                   ,
#endif

-- ** apply #method:apply#

#if defined(ENABLE_OVERLOADING)
    SettingsApplyMethodInfo                 ,
#endif
    settingsApply                           ,


-- ** bind #method:bind#

#if defined(ENABLE_OVERLOADING)
    SettingsBindMethodInfo                  ,
#endif
    settingsBind                            ,


-- ** bindWithMapping #method:bindWithMapping#

#if defined(ENABLE_OVERLOADING)
    SettingsBindWithMappingMethodInfo       ,
#endif
    settingsBindWithMapping                 ,


-- ** bindWritable #method:bindWritable#

#if defined(ENABLE_OVERLOADING)
    SettingsBindWritableMethodInfo          ,
#endif
    settingsBindWritable                    ,


-- ** createAction #method:createAction#

#if defined(ENABLE_OVERLOADING)
    SettingsCreateActionMethodInfo          ,
#endif
    settingsCreateAction                    ,


-- ** delay #method:delay#

#if defined(ENABLE_OVERLOADING)
    SettingsDelayMethodInfo                 ,
#endif
    settingsDelay                           ,


-- ** getBoolean #method:getBoolean#

#if defined(ENABLE_OVERLOADING)
    SettingsGetBooleanMethodInfo            ,
#endif
    settingsGetBoolean                      ,


-- ** getChild #method:getChild#

#if defined(ENABLE_OVERLOADING)
    SettingsGetChildMethodInfo              ,
#endif
    settingsGetChild                        ,


-- ** getDefaultValue #method:getDefaultValue#

#if defined(ENABLE_OVERLOADING)
    SettingsGetDefaultValueMethodInfo       ,
#endif
    settingsGetDefaultValue                 ,


-- ** getDouble #method:getDouble#

#if defined(ENABLE_OVERLOADING)
    SettingsGetDoubleMethodInfo             ,
#endif
    settingsGetDouble                       ,


-- ** getEnum #method:getEnum#

#if defined(ENABLE_OVERLOADING)
    SettingsGetEnumMethodInfo               ,
#endif
    settingsGetEnum                         ,


-- ** getFlags #method:getFlags#

#if defined(ENABLE_OVERLOADING)
    SettingsGetFlagsMethodInfo              ,
#endif
    settingsGetFlags                        ,


-- ** getHasUnapplied #method:getHasUnapplied#

#if defined(ENABLE_OVERLOADING)
    SettingsGetHasUnappliedMethodInfo       ,
#endif
    settingsGetHasUnapplied                 ,


-- ** getInt #method:getInt#

#if defined(ENABLE_OVERLOADING)
    SettingsGetIntMethodInfo                ,
#endif
    settingsGetInt                          ,


-- ** getInt64 #method:getInt64#

#if defined(ENABLE_OVERLOADING)
    SettingsGetInt64MethodInfo              ,
#endif
    settingsGetInt64                        ,


-- ** getMapped #method:getMapped#

#if defined(ENABLE_OVERLOADING)
    SettingsGetMappedMethodInfo             ,
#endif
    settingsGetMapped                       ,


-- ** getRange #method:getRange#

#if defined(ENABLE_OVERLOADING)
    SettingsGetRangeMethodInfo              ,
#endif
    settingsGetRange                        ,


-- ** getString #method:getString#

#if defined(ENABLE_OVERLOADING)
    SettingsGetStringMethodInfo             ,
#endif
    settingsGetString                       ,


-- ** getStrv #method:getStrv#

#if defined(ENABLE_OVERLOADING)
    SettingsGetStrvMethodInfo               ,
#endif
    settingsGetStrv                         ,


-- ** getUint #method:getUint#

#if defined(ENABLE_OVERLOADING)
    SettingsGetUintMethodInfo               ,
#endif
    settingsGetUint                         ,


-- ** getUint64 #method:getUint64#

#if defined(ENABLE_OVERLOADING)
    SettingsGetUint64MethodInfo             ,
#endif
    settingsGetUint64                       ,


-- ** getUserValue #method:getUserValue#

#if defined(ENABLE_OVERLOADING)
    SettingsGetUserValueMethodInfo          ,
#endif
    settingsGetUserValue                    ,


-- ** getValue #method:getValue#

#if defined(ENABLE_OVERLOADING)
    SettingsGetValueMethodInfo              ,
#endif
    settingsGetValue                        ,


-- ** isWritable #method:isWritable#

#if defined(ENABLE_OVERLOADING)
    SettingsIsWritableMethodInfo            ,
#endif
    settingsIsWritable                      ,


-- ** listChildren #method:listChildren#

#if defined(ENABLE_OVERLOADING)
    SettingsListChildrenMethodInfo          ,
#endif
    settingsListChildren                    ,


-- ** listKeys #method:listKeys#

#if defined(ENABLE_OVERLOADING)
    SettingsListKeysMethodInfo              ,
#endif
    settingsListKeys                        ,


-- ** listRelocatableSchemas #method:listRelocatableSchemas#

    settingsListRelocatableSchemas          ,


-- ** listSchemas #method:listSchemas#

    settingsListSchemas                     ,


-- ** new #method:new#

    settingsNew                             ,


-- ** newFull #method:newFull#

    settingsNewFull                         ,


-- ** newWithBackend #method:newWithBackend#

    settingsNewWithBackend                  ,


-- ** newWithBackendAndPath #method:newWithBackendAndPath#

    settingsNewWithBackendAndPath           ,


-- ** newWithPath #method:newWithPath#

    settingsNewWithPath                     ,


-- ** rangeCheck #method:rangeCheck#

#if defined(ENABLE_OVERLOADING)
    SettingsRangeCheckMethodInfo            ,
#endif
    settingsRangeCheck                      ,


-- ** reset #method:reset#

#if defined(ENABLE_OVERLOADING)
    SettingsResetMethodInfo                 ,
#endif
    settingsReset                           ,


-- ** revert #method:revert#

#if defined(ENABLE_OVERLOADING)
    SettingsRevertMethodInfo                ,
#endif
    settingsRevert                          ,


-- ** setBoolean #method:setBoolean#

#if defined(ENABLE_OVERLOADING)
    SettingsSetBooleanMethodInfo            ,
#endif
    settingsSetBoolean                      ,


-- ** setDouble #method:setDouble#

#if defined(ENABLE_OVERLOADING)
    SettingsSetDoubleMethodInfo             ,
#endif
    settingsSetDouble                       ,


-- ** setEnum #method:setEnum#

#if defined(ENABLE_OVERLOADING)
    SettingsSetEnumMethodInfo               ,
#endif
    settingsSetEnum                         ,


-- ** setFlags #method:setFlags#

#if defined(ENABLE_OVERLOADING)
    SettingsSetFlagsMethodInfo              ,
#endif
    settingsSetFlags                        ,


-- ** setInt #method:setInt#

#if defined(ENABLE_OVERLOADING)
    SettingsSetIntMethodInfo                ,
#endif
    settingsSetInt                          ,


-- ** setInt64 #method:setInt64#

#if defined(ENABLE_OVERLOADING)
    SettingsSetInt64MethodInfo              ,
#endif
    settingsSetInt64                        ,


-- ** setString #method:setString#

#if defined(ENABLE_OVERLOADING)
    SettingsSetStringMethodInfo             ,
#endif
    settingsSetString                       ,


-- ** setStrv #method:setStrv#

#if defined(ENABLE_OVERLOADING)
    SettingsSetStrvMethodInfo               ,
#endif
    settingsSetStrv                         ,


-- ** setUint #method:setUint#

#if defined(ENABLE_OVERLOADING)
    SettingsSetUintMethodInfo               ,
#endif
    settingsSetUint                         ,


-- ** setUint64 #method:setUint64#

#if defined(ENABLE_OVERLOADING)
    SettingsSetUint64MethodInfo             ,
#endif
    settingsSetUint64                       ,


-- ** setValue #method:setValue#

#if defined(ENABLE_OVERLOADING)
    SettingsSetValueMethodInfo              ,
#endif
    settingsSetValue                        ,


-- ** sync #method:sync#

    settingsSync                            ,


-- ** unbind #method:unbind#

    settingsUnbind                          ,




 -- * Properties


-- ** backend #attr:backend#
-- | The name of the context that the settings are stored in.

#if defined(ENABLE_OVERLOADING)
    SettingsBackendPropertyInfo             ,
#endif
    constructSettingsBackend                ,
    getSettingsBackend                      ,
#if defined(ENABLE_OVERLOADING)
    settingsBackend                         ,
#endif


-- ** delayApply #attr:delayApply#
-- | Whether the t'GI.Gio.Objects.Settings.Settings' object is in
-- <http://developer.gnome.org/gio/stable/class.Settings.html#delay-apply-mode ‘delay-apply’ mode>.
-- 
-- /Since: 2.28/

#if defined(ENABLE_OVERLOADING)
    SettingsDelayApplyPropertyInfo          ,
#endif
    getSettingsDelayApply                   ,
#if defined(ENABLE_OVERLOADING)
    settingsDelayApply                      ,
#endif


-- ** hasUnapplied #attr:hasUnapplied#
-- | Whether the t'GI.Gio.Objects.Settings.Settings' object has outstanding changes.
-- 
-- These changes will be applied when 'GI.Gio.Objects.Settings.settingsApply' is called.

#if defined(ENABLE_OVERLOADING)
    SettingsHasUnappliedPropertyInfo        ,
#endif
    getSettingsHasUnapplied                 ,
#if defined(ENABLE_OVERLOADING)
    settingsHasUnapplied                    ,
#endif


-- ** path #attr:path#
-- | The path within the backend where the settings are stored.

#if defined(ENABLE_OVERLOADING)
    SettingsPathPropertyInfo                ,
#endif
    constructSettingsPath                   ,
    getSettingsPath                         ,
#if defined(ENABLE_OVERLOADING)
    settingsPath                            ,
#endif


-- ** schema #attr:schema#
-- | The name of the schema that describes the types of keys
-- for this t'GI.Gio.Objects.Settings.Settings' object.
-- 
-- The type of this property is *not* t'GI.Gio.Structs.SettingsSchema.SettingsSchema'.
-- t'GI.Gio.Structs.SettingsSchema.SettingsSchema' has only existed since version 2.32 and
-- unfortunately this name was used in previous versions to refer to
-- the schema ID rather than the schema itself.  Take care to use the
-- [Settings:settingsSchema]("GI.Gio.Objects.Settings#g:attr:settingsSchema") property if you wish to pass in a
-- t'GI.Gio.Structs.SettingsSchema.SettingsSchema'.

#if defined(ENABLE_OVERLOADING)
    SettingsSchemaPropertyInfo              ,
#endif
    constructSettingsSchema                 ,
    getSettingsSchema                       ,
#if defined(ENABLE_OVERLOADING)
    settingsSchema                          ,
#endif


-- ** schemaId #attr:schemaId#
-- | The name of the schema that describes the types of keys
-- for this t'GI.Gio.Objects.Settings.Settings' object.

#if defined(ENABLE_OVERLOADING)
    SettingsSchemaIdPropertyInfo            ,
#endif
    constructSettingsSchemaId               ,
    getSettingsSchemaId                     ,
#if defined(ENABLE_OVERLOADING)
    settingsSchemaId                        ,
#endif


-- ** settingsSchema #attr:settingsSchema#
-- | The t'GI.Gio.Structs.SettingsSchema.SettingsSchema' describing the types of keys for this
-- t'GI.Gio.Objects.Settings.Settings' object.
-- 
-- Ideally, this property would be called [Settings:schema]("GI.Gio.Objects.Settings#g:attr:schema").
-- t'GI.Gio.Structs.SettingsSchema.SettingsSchema'
-- has only existed since version 2.32, however, and before then the
-- [Settings:schema]("GI.Gio.Objects.Settings#g:attr:schema") property was used to refer to the ID of the schema rather
-- than the schema itself.  Take care.

#if defined(ENABLE_OVERLOADING)
    SettingsSettingsSchemaPropertyInfo      ,
#endif
    constructSettingsSettingsSchema         ,
    getSettingsSettingsSchema               ,
#if defined(ENABLE_OVERLOADING)
    settingsSettingsSchema                  ,
#endif




 -- * Signals


-- ** changeEvent #signal:changeEvent#

    SettingsChangeEventCallback             ,
#if defined(ENABLE_OVERLOADING)
    SettingsChangeEventSignalInfo           ,
#endif
    afterSettingsChangeEvent                ,
    onSettingsChangeEvent                   ,


-- ** changed #signal:changed#

    SettingsChangedCallback                 ,
#if defined(ENABLE_OVERLOADING)
    SettingsChangedSignalInfo               ,
#endif
    afterSettingsChanged                    ,
    onSettingsChanged                       ,


-- ** writableChangeEvent #signal:writableChangeEvent#

    SettingsWritableChangeEventCallback     ,
#if defined(ENABLE_OVERLOADING)
    SettingsWritableChangeEventSignalInfo   ,
#endif
    afterSettingsWritableChangeEvent        ,
    onSettingsWritableChangeEvent           ,


-- ** writableChanged #signal:writableChanged#

    SettingsWritableChangedCallback         ,
#if defined(ENABLE_OVERLOADING)
    SettingsWritableChangedSignalInfo       ,
#endif
    afterSettingsWritableChanged            ,
    onSettingsWritableChanged               ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.GLib.Structs.Tree as GLib.Tree
import qualified GI.GLib.Structs.VariantType as GLib.VariantType
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gio.Callbacks as Gio.Callbacks
import {-# SOURCE #-} qualified GI.Gio.Flags as Gio.Flags
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Action as Gio.Action
import {-# SOURCE #-} qualified GI.Gio.Objects.SettingsBackend as Gio.SettingsBackend
import {-# SOURCE #-} qualified GI.Gio.Structs.SettingsSchema as Gio.SettingsSchema
import {-# SOURCE #-} qualified GI.Gio.Structs.SettingsSchemaKey as Gio.SettingsSchemaKey

#else
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gio.Callbacks as Gio.Callbacks
import {-# SOURCE #-} qualified GI.Gio.Flags as Gio.Flags
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Action as Gio.Action
import {-# SOURCE #-} qualified GI.Gio.Objects.SettingsBackend as Gio.SettingsBackend
import {-# SOURCE #-} qualified GI.Gio.Structs.SettingsSchema as Gio.SettingsSchema

#endif

-- | Memory-managed wrapper type.
newtype Settings = Settings (SP.ManagedPtr Settings)
    deriving (Eq)

instance SP.ManagedPtrNewtype Settings where
    toManagedPtr (Settings p) = p

foreign import ccall "g_settings_get_type"
    c_g_settings_get_type :: IO B.Types.GType

instance B.Types.TypedObject Settings where
    glibType = c_g_settings_get_type

instance B.Types.GObject Settings

-- | Type class for types which can be safely cast to t'Settings', for instance with `toSettings`.
class (SP.GObject o, O.IsDescendantOf Settings o) => IsSettings o
instance (SP.GObject o, O.IsDescendantOf Settings o) => IsSettings o

instance O.HasParentTypes Settings
type instance O.ParentTypes Settings = '[GObject.Object.Object]

-- | Cast to t'Settings', for types for which this is known to be safe. For general casts, use 'Data.GI.Base.ManagedPtr.castTo'.
toSettings :: (MIO.MonadIO m, IsSettings o) => o -> m Settings
toSettings = MIO.liftIO . B.ManagedPtr.unsafeCastTo Settings

-- | Convert t'Settings' to and from t'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe Settings) where
    gvalueGType_ = c_g_settings_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr Settings)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr Settings)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject Settings ptr
        else return P.Nothing
        
    

#if defined(ENABLE_OVERLOADING)
type family ResolveSettingsMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveSettingsMethod "apply" o = SettingsApplyMethodInfo
    ResolveSettingsMethod "bind" o = SettingsBindMethodInfo
    ResolveSettingsMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolveSettingsMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolveSettingsMethod "bindWithMapping" o = SettingsBindWithMappingMethodInfo
    ResolveSettingsMethod "bindWritable" o = SettingsBindWritableMethodInfo
    ResolveSettingsMethod "createAction" o = SettingsCreateActionMethodInfo
    ResolveSettingsMethod "delay" o = SettingsDelayMethodInfo
    ResolveSettingsMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolveSettingsMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolveSettingsMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolveSettingsMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolveSettingsMethod "isWritable" o = SettingsIsWritableMethodInfo
    ResolveSettingsMethod "listChildren" o = SettingsListChildrenMethodInfo
    ResolveSettingsMethod "listKeys" o = SettingsListKeysMethodInfo
    ResolveSettingsMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolveSettingsMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolveSettingsMethod "rangeCheck" o = SettingsRangeCheckMethodInfo
    ResolveSettingsMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolveSettingsMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolveSettingsMethod "reset" o = SettingsResetMethodInfo
    ResolveSettingsMethod "revert" o = SettingsRevertMethodInfo
    ResolveSettingsMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolveSettingsMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolveSettingsMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolveSettingsMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolveSettingsMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolveSettingsMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolveSettingsMethod "getBoolean" o = SettingsGetBooleanMethodInfo
    ResolveSettingsMethod "getChild" o = SettingsGetChildMethodInfo
    ResolveSettingsMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolveSettingsMethod "getDefaultValue" o = SettingsGetDefaultValueMethodInfo
    ResolveSettingsMethod "getDouble" o = SettingsGetDoubleMethodInfo
    ResolveSettingsMethod "getEnum" o = SettingsGetEnumMethodInfo
    ResolveSettingsMethod "getFlags" o = SettingsGetFlagsMethodInfo
    ResolveSettingsMethod "getHasUnapplied" o = SettingsGetHasUnappliedMethodInfo
    ResolveSettingsMethod "getInt" o = SettingsGetIntMethodInfo
    ResolveSettingsMethod "getInt64" o = SettingsGetInt64MethodInfo
    ResolveSettingsMethod "getMapped" o = SettingsGetMappedMethodInfo
    ResolveSettingsMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolveSettingsMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolveSettingsMethod "getRange" o = SettingsGetRangeMethodInfo
    ResolveSettingsMethod "getString" o = SettingsGetStringMethodInfo
    ResolveSettingsMethod "getStrv" o = SettingsGetStrvMethodInfo
    ResolveSettingsMethod "getUint" o = SettingsGetUintMethodInfo
    ResolveSettingsMethod "getUint64" o = SettingsGetUint64MethodInfo
    ResolveSettingsMethod "getUserValue" o = SettingsGetUserValueMethodInfo
    ResolveSettingsMethod "getValue" o = SettingsGetValueMethodInfo
    ResolveSettingsMethod "setBoolean" o = SettingsSetBooleanMethodInfo
    ResolveSettingsMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolveSettingsMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolveSettingsMethod "setDouble" o = SettingsSetDoubleMethodInfo
    ResolveSettingsMethod "setEnum" o = SettingsSetEnumMethodInfo
    ResolveSettingsMethod "setFlags" o = SettingsSetFlagsMethodInfo
    ResolveSettingsMethod "setInt" o = SettingsSetIntMethodInfo
    ResolveSettingsMethod "setInt64" o = SettingsSetInt64MethodInfo
    ResolveSettingsMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolveSettingsMethod "setString" o = SettingsSetStringMethodInfo
    ResolveSettingsMethod "setStrv" o = SettingsSetStrvMethodInfo
    ResolveSettingsMethod "setUint" o = SettingsSetUintMethodInfo
    ResolveSettingsMethod "setUint64" o = SettingsSetUint64MethodInfo
    ResolveSettingsMethod "setValue" o = SettingsSetValueMethodInfo
    ResolveSettingsMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveSettingsMethod t Settings, O.OverloadedMethod info Settings p) => OL.IsLabel t (Settings -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveSettingsMethod t Settings, O.OverloadedMethod info Settings p, R.HasField t Settings p) => R.HasField t Settings p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveSettingsMethod t Settings, O.OverloadedMethodInfo info Settings) => OL.IsLabel t (O.MethodProxy info Settings) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

-- signal Settings::change-event
-- | Emitted once per change event that affects this settings object.
-- 
-- You should connect to this signal
-- only if you are interested in viewing groups of changes before they
-- are split out into multiple emissions of the [Settings::changed]("GI.Gio.Objects.Settings#g:signal:changed") signal.
-- For most use cases it is more appropriate to use the [Settings::changed]("GI.Gio.Objects.Settings#g:signal:changed") signal.
-- 
-- In the event that the change event applies to one or more specified
-- keys, /@keys@/ will be an array of [alias/@gLib@/.Quark]s of length /@nKeys@/.  In the
-- event that the change event applies to the t'GI.Gio.Objects.Settings.Settings' object as a
-- whole (ie: potentially every key has been changed) then /@keys@/ will
-- be @NULL@ and /@nKeys@/ will be @0@.
-- 
-- The default handler for this signal invokes the [Settings::changed]("GI.Gio.Objects.Settings#g:signal:changed") signal
-- for each affected key.  If any other connected handler returns
-- true then this default functionality will be suppressed.
type SettingsChangeEventCallback =
    Maybe [Word32]
    -- ^ /@keys@/: array of the
    --   keys which have changed
    -> IO Bool
    -- ^ __Returns:__ true to stop other handlers from being invoked for the
    --   event, false to propagate the event further

type C_SettingsChangeEventCallback =
    Ptr Settings ->                         -- object
    Ptr Word32 ->
    Int32 ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_SettingsChangeEventCallback`.
foreign import ccall "wrapper"
    mk_SettingsChangeEventCallback :: C_SettingsChangeEventCallback -> IO (FunPtr C_SettingsChangeEventCallback)

wrap_SettingsChangeEventCallback :: 
    GObject a => (a -> SettingsChangeEventCallback) ->
    C_SettingsChangeEventCallback
wrap_SettingsChangeEventCallback gi'cb gi'selfPtr keys nKeys _ = do
    maybeKeys <-
        if keys == FP.nullPtr
        then return Nothing
        else do
            keys' <- (unpackStorableArrayWithLength nKeys) keys
            return $ Just keys'
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  maybeKeys
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [changeEvent](#signal:changeEvent) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' settings #changeEvent callback
-- @
-- 
-- 
onSettingsChangeEvent :: (IsSettings a, MonadIO m) => a -> ((?self :: a) => SettingsChangeEventCallback) -> m SignalHandlerId
onSettingsChangeEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_SettingsChangeEventCallback wrapped
    wrapped'' <- mk_SettingsChangeEventCallback wrapped'
    connectSignalFunPtr obj "change-event" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [changeEvent](#signal:changeEvent) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' settings #changeEvent callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterSettingsChangeEvent :: (IsSettings a, MonadIO m) => a -> ((?self :: a) => SettingsChangeEventCallback) -> m SignalHandlerId
afterSettingsChangeEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_SettingsChangeEventCallback wrapped
    wrapped'' <- mk_SettingsChangeEventCallback wrapped'
    connectSignalFunPtr obj "change-event" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data SettingsChangeEventSignalInfo
instance SignalInfo SettingsChangeEventSignalInfo where
    type HaskellCallbackType SettingsChangeEventSignalInfo = SettingsChangeEventCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_SettingsChangeEventCallback cb
        cb'' <- mk_SettingsChangeEventCallback cb'
        connectSignalFunPtr obj "change-event" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings::change-event"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#g:signal:changeEvent"})

#endif

-- signal Settings::changed
-- | Emitted when a key has potentially changed.
-- 
-- You should call one of the t'GI.Gio.Objects.Settings.Settings'.@/get/@() calls to check the new
-- value.
-- 
-- This signal supports detailed connections.  You can connect to the
-- detailed signal @changed::x@ in order to only receive callbacks
-- when key @x@ changes.
-- 
-- Note that /@settings@/ only emits this signal if you have read /@key@/ at
-- least once while a signal handler was already connected for /@key@/.
type SettingsChangedCallback =
    T.Text
    -- ^ /@key@/: the name of the key that changed
    -> IO ()

type C_SettingsChangedCallback =
    Ptr Settings ->                         -- object
    CString ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_SettingsChangedCallback`.
foreign import ccall "wrapper"
    mk_SettingsChangedCallback :: C_SettingsChangedCallback -> IO (FunPtr C_SettingsChangedCallback)

wrap_SettingsChangedCallback :: 
    GObject a => (a -> SettingsChangedCallback) ->
    C_SettingsChangedCallback
wrap_SettingsChangedCallback gi'cb gi'selfPtr key _ = do
    key' <- cstringToText key
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  key'


-- | Connect a signal handler for the [changed](#signal:changed) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' settings #changed callback
-- @
-- 
-- This signal admits a optional parameter @detail@.
-- If it's not @Nothing@, we will connect to “@changed::detail@” instead.
-- 
onSettingsChanged :: (IsSettings a, MonadIO m) => a -> P.Maybe T.Text -> ((?self :: a) => SettingsChangedCallback) -> m SignalHandlerId
onSettingsChanged obj detail cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_SettingsChangedCallback wrapped
    wrapped'' <- mk_SettingsChangedCallback wrapped'
    connectSignalFunPtr obj "changed" wrapped'' SignalConnectBefore detail

-- | Connect a signal handler for the [changed](#signal:changed) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' settings #changed callback
-- @
-- 
-- This signal admits a optional parameter @detail@.
-- If it's not @Nothing@, we will connect to “@changed::detail@” instead.
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterSettingsChanged :: (IsSettings a, MonadIO m) => a -> P.Maybe T.Text -> ((?self :: a) => SettingsChangedCallback) -> m SignalHandlerId
afterSettingsChanged obj detail cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_SettingsChangedCallback wrapped
    wrapped'' <- mk_SettingsChangedCallback wrapped'
    connectSignalFunPtr obj "changed" wrapped'' SignalConnectAfter detail


#if defined(ENABLE_OVERLOADING)
data SettingsChangedSignalInfo
instance SignalInfo SettingsChangedSignalInfo where
    type HaskellCallbackType SettingsChangedSignalInfo = SettingsChangedCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_SettingsChangedCallback cb
        cb'' <- mk_SettingsChangedCallback cb'
        connectSignalFunPtr obj "changed" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings::changed"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#g:signal:changed"})

#endif

-- signal Settings::writable-change-event
-- | Emitted once per writability change event that affects this settings object.
-- 
-- You should connect
-- to this signal if you are interested in viewing groups of changes
-- before they are split out into multiple emissions of the
-- [Settings::writableChanged]("GI.Gio.Objects.Settings#g:signal:writableChanged") signal.  For most use cases it is more
-- appropriate to use the [Settings::writableChanged]("GI.Gio.Objects.Settings#g:signal:writableChanged") signal.
-- 
-- In the event that the writability change applies only to a single
-- key, /@key@/ will be set to the [alias/@gLib@/.Quark] for that key.  In the event
-- that the writability change affects the entire settings object,
-- /@key@/ will be @0@.
-- 
-- The default handler for this signal invokes the [Settings::writableChanged]("GI.Gio.Objects.Settings#g:signal:writableChanged")
-- and [Settings::changed]("GI.Gio.Objects.Settings#g:signal:changed") signals for each affected key.  This is done because
-- changes in writability might also imply changes in value (if for
-- example, a new mandatory setting is introduced).  If any other
-- connected handler returns true then this default functionality
-- will be suppressed.
type SettingsWritableChangeEventCallback =
    Word32
    -- ^ /@key@/: the quark of the key, or @0@
    -> IO Bool
    -- ^ __Returns:__ true to stop other handlers from being invoked for the
    --   event, false to propagate the event further

type C_SettingsWritableChangeEventCallback =
    Ptr Settings ->                         -- object
    Word32 ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_SettingsWritableChangeEventCallback`.
foreign import ccall "wrapper"
    mk_SettingsWritableChangeEventCallback :: C_SettingsWritableChangeEventCallback -> IO (FunPtr C_SettingsWritableChangeEventCallback)

wrap_SettingsWritableChangeEventCallback :: 
    GObject a => (a -> SettingsWritableChangeEventCallback) ->
    C_SettingsWritableChangeEventCallback
wrap_SettingsWritableChangeEventCallback gi'cb gi'selfPtr key _ = do
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  key
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [writableChangeEvent](#signal:writableChangeEvent) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' settings #writableChangeEvent callback
-- @
-- 
-- 
onSettingsWritableChangeEvent :: (IsSettings a, MonadIO m) => a -> ((?self :: a) => SettingsWritableChangeEventCallback) -> m SignalHandlerId
onSettingsWritableChangeEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_SettingsWritableChangeEventCallback wrapped
    wrapped'' <- mk_SettingsWritableChangeEventCallback wrapped'
    connectSignalFunPtr obj "writable-change-event" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [writableChangeEvent](#signal:writableChangeEvent) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' settings #writableChangeEvent callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterSettingsWritableChangeEvent :: (IsSettings a, MonadIO m) => a -> ((?self :: a) => SettingsWritableChangeEventCallback) -> m SignalHandlerId
afterSettingsWritableChangeEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_SettingsWritableChangeEventCallback wrapped
    wrapped'' <- mk_SettingsWritableChangeEventCallback wrapped'
    connectSignalFunPtr obj "writable-change-event" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data SettingsWritableChangeEventSignalInfo
instance SignalInfo SettingsWritableChangeEventSignalInfo where
    type HaskellCallbackType SettingsWritableChangeEventSignalInfo = SettingsWritableChangeEventCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_SettingsWritableChangeEventCallback cb
        cb'' <- mk_SettingsWritableChangeEventCallback cb'
        connectSignalFunPtr obj "writable-change-event" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings::writable-change-event"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#g:signal:writableChangeEvent"})

#endif

-- signal Settings::writable-changed
-- | Emitted when the writability of a key has potentially changed.
-- 
-- You should call 'GI.Gio.Objects.Settings.settingsIsWritable' in order to determine the
-- new status.
-- 
-- This signal supports detailed connections.  You can connect to the
-- detailed signal @writable-changed::x@ in order to only receive
-- callbacks when the writability of @x@ changes.
type SettingsWritableChangedCallback =
    T.Text
    -- ^ /@key@/: the key
    -> IO ()

type C_SettingsWritableChangedCallback =
    Ptr Settings ->                         -- object
    CString ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_SettingsWritableChangedCallback`.
foreign import ccall "wrapper"
    mk_SettingsWritableChangedCallback :: C_SettingsWritableChangedCallback -> IO (FunPtr C_SettingsWritableChangedCallback)

wrap_SettingsWritableChangedCallback :: 
    GObject a => (a -> SettingsWritableChangedCallback) ->
    C_SettingsWritableChangedCallback
wrap_SettingsWritableChangedCallback gi'cb gi'selfPtr key _ = do
    key' <- cstringToText key
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  key'


-- | Connect a signal handler for the [writableChanged](#signal:writableChanged) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' settings #writableChanged callback
-- @
-- 
-- This signal admits a optional parameter @detail@.
-- If it's not @Nothing@, we will connect to “@writable-changed::detail@” instead.
-- 
onSettingsWritableChanged :: (IsSettings a, MonadIO m) => a -> P.Maybe T.Text -> ((?self :: a) => SettingsWritableChangedCallback) -> m SignalHandlerId
onSettingsWritableChanged obj detail cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_SettingsWritableChangedCallback wrapped
    wrapped'' <- mk_SettingsWritableChangedCallback wrapped'
    connectSignalFunPtr obj "writable-changed" wrapped'' SignalConnectBefore detail

-- | Connect a signal handler for the [writableChanged](#signal:writableChanged) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' settings #writableChanged callback
-- @
-- 
-- This signal admits a optional parameter @detail@.
-- If it's not @Nothing@, we will connect to “@writable-changed::detail@” instead.
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterSettingsWritableChanged :: (IsSettings a, MonadIO m) => a -> P.Maybe T.Text -> ((?self :: a) => SettingsWritableChangedCallback) -> m SignalHandlerId
afterSettingsWritableChanged obj detail cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_SettingsWritableChangedCallback wrapped
    wrapped'' <- mk_SettingsWritableChangedCallback wrapped'
    connectSignalFunPtr obj "writable-changed" wrapped'' SignalConnectAfter detail


#if defined(ENABLE_OVERLOADING)
data SettingsWritableChangedSignalInfo
instance SignalInfo SettingsWritableChangedSignalInfo where
    type HaskellCallbackType SettingsWritableChangedSignalInfo = SettingsWritableChangedCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_SettingsWritableChangedCallback cb
        cb'' <- mk_SettingsWritableChangedCallback cb'
        connectSignalFunPtr obj "writable-changed" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings::writable-changed"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#g:signal:writableChanged"})

#endif

-- VVV Prop "backend"
   -- Type: TInterface (Name {namespace = "Gio", name = "SettingsBackend"})
   -- Flags: [PropertyReadable,PropertyWritable,PropertyConstructOnly]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@backend@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #backend
-- @
getSettingsBackend :: (MonadIO m, IsSettings o) => o -> m (Maybe Gio.SettingsBackend.SettingsBackend)
getSettingsBackend obj = MIO.liftIO $ B.Properties.getObjectPropertyObject obj "backend" Gio.SettingsBackend.SettingsBackend

-- | Construct a t'GValueConstruct' with valid value for the “@backend@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsBackend :: (IsSettings o, MIO.MonadIO m, Gio.SettingsBackend.IsSettingsBackend a) => a -> m (GValueConstruct o)
constructSettingsBackend val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyObject "backend" (P.Just val)

#if defined(ENABLE_OVERLOADING)
data SettingsBackendPropertyInfo
instance AttrInfo SettingsBackendPropertyInfo where
    type AttrAllowedOps SettingsBackendPropertyInfo = '[ 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint SettingsBackendPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsBackendPropertyInfo = Gio.SettingsBackend.IsSettingsBackend
    type AttrTransferTypeConstraint SettingsBackendPropertyInfo = Gio.SettingsBackend.IsSettingsBackend
    type AttrTransferType SettingsBackendPropertyInfo = Gio.SettingsBackend.SettingsBackend
    type AttrGetType SettingsBackendPropertyInfo = (Maybe Gio.SettingsBackend.SettingsBackend)
    type AttrLabel SettingsBackendPropertyInfo = "backend"
    type AttrOrigin SettingsBackendPropertyInfo = Settings
    attrGet = getSettingsBackend
    attrSet = undefined
    attrTransfer _ v = do
        unsafeCastTo Gio.SettingsBackend.SettingsBackend v
    attrConstruct = constructSettingsBackend
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.backend"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#g:attr:backend"
        })
#endif

-- VVV Prop "delay-apply"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@delay-apply@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #delayApply
-- @
getSettingsDelayApply :: (MonadIO m, IsSettings o) => o -> m Bool
getSettingsDelayApply obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "delay-apply"

#if defined(ENABLE_OVERLOADING)
data SettingsDelayApplyPropertyInfo
instance AttrInfo SettingsDelayApplyPropertyInfo where
    type AttrAllowedOps SettingsDelayApplyPropertyInfo = '[ 'AttrGet]
    type AttrBaseTypeConstraint SettingsDelayApplyPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsDelayApplyPropertyInfo = (~) ()
    type AttrTransferTypeConstraint SettingsDelayApplyPropertyInfo = (~) ()
    type AttrTransferType SettingsDelayApplyPropertyInfo = ()
    type AttrGetType SettingsDelayApplyPropertyInfo = Bool
    type AttrLabel SettingsDelayApplyPropertyInfo = "delay-apply"
    type AttrOrigin SettingsDelayApplyPropertyInfo = Settings
    attrGet = getSettingsDelayApply
    attrSet = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.delayApply"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#g:attr:delayApply"
        })
#endif

-- VVV Prop "has-unapplied"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable]
   -- Nullable: (Just False,Nothing)

-- | Get the value of the “@has-unapplied@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #hasUnapplied
-- @
getSettingsHasUnapplied :: (MonadIO m, IsSettings o) => o -> m Bool
getSettingsHasUnapplied obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "has-unapplied"

#if defined(ENABLE_OVERLOADING)
data SettingsHasUnappliedPropertyInfo
instance AttrInfo SettingsHasUnappliedPropertyInfo where
    type AttrAllowedOps SettingsHasUnappliedPropertyInfo = '[ 'AttrGet]
    type AttrBaseTypeConstraint SettingsHasUnappliedPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsHasUnappliedPropertyInfo = (~) ()
    type AttrTransferTypeConstraint SettingsHasUnappliedPropertyInfo = (~) ()
    type AttrTransferType SettingsHasUnappliedPropertyInfo = ()
    type AttrGetType SettingsHasUnappliedPropertyInfo = Bool
    type AttrLabel SettingsHasUnappliedPropertyInfo = "has-unapplied"
    type AttrOrigin SettingsHasUnappliedPropertyInfo = Settings
    attrGet = getSettingsHasUnapplied
    attrSet = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.hasUnapplied"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#g:attr:hasUnapplied"
        })
#endif

-- VVV Prop "path"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable,PropertyWritable,PropertyConstructOnly]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@path@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #path
-- @
getSettingsPath :: (MonadIO m, IsSettings o) => o -> m (Maybe T.Text)
getSettingsPath obj = MIO.liftIO $ B.Properties.getObjectPropertyString obj "path"

-- | Construct a t'GValueConstruct' with valid value for the “@path@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsPath :: (IsSettings o, MIO.MonadIO m) => T.Text -> m (GValueConstruct o)
constructSettingsPath val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyString "path" (P.Just val)

#if defined(ENABLE_OVERLOADING)
data SettingsPathPropertyInfo
instance AttrInfo SettingsPathPropertyInfo where
    type AttrAllowedOps SettingsPathPropertyInfo = '[ 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint SettingsPathPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsPathPropertyInfo = (~) T.Text
    type AttrTransferTypeConstraint SettingsPathPropertyInfo = (~) T.Text
    type AttrTransferType SettingsPathPropertyInfo = T.Text
    type AttrGetType SettingsPathPropertyInfo = (Maybe T.Text)
    type AttrLabel SettingsPathPropertyInfo = "path"
    type AttrOrigin SettingsPathPropertyInfo = Settings
    attrGet = getSettingsPath
    attrSet = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsPath
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.path"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#g:attr:path"
        })
#endif

-- VVV Prop "schema"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable,PropertyWritable,PropertyConstructOnly]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@schema@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #schema
-- @
getSettingsSchema :: (MonadIO m, IsSettings o) => o -> m (Maybe T.Text)
getSettingsSchema obj = MIO.liftIO $ B.Properties.getObjectPropertyString obj "schema"

-- | Construct a t'GValueConstruct' with valid value for the “@schema@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsSchema :: (IsSettings o, MIO.MonadIO m) => T.Text -> m (GValueConstruct o)
constructSettingsSchema val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyString "schema" (P.Just val)

#if defined(ENABLE_OVERLOADING)
data SettingsSchemaPropertyInfo
instance AttrInfo SettingsSchemaPropertyInfo where
    type AttrAllowedOps SettingsSchemaPropertyInfo = '[ 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint SettingsSchemaPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsSchemaPropertyInfo = (~) T.Text
    type AttrTransferTypeConstraint SettingsSchemaPropertyInfo = (~) T.Text
    type AttrTransferType SettingsSchemaPropertyInfo = T.Text
    type AttrGetType SettingsSchemaPropertyInfo = (Maybe T.Text)
    type AttrLabel SettingsSchemaPropertyInfo = "schema"
    type AttrOrigin SettingsSchemaPropertyInfo = Settings
    attrGet = getSettingsSchema
    attrSet = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsSchema
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.schema"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#g:attr:schema"
        })
#endif

-- VVV Prop "schema-id"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable,PropertyWritable,PropertyConstructOnly]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@schema-id@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #schemaId
-- @
getSettingsSchemaId :: (MonadIO m, IsSettings o) => o -> m (Maybe T.Text)
getSettingsSchemaId obj = MIO.liftIO $ B.Properties.getObjectPropertyString obj "schema-id"

-- | Construct a t'GValueConstruct' with valid value for the “@schema-id@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsSchemaId :: (IsSettings o, MIO.MonadIO m) => T.Text -> m (GValueConstruct o)
constructSettingsSchemaId val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyString "schema-id" (P.Just val)

#if defined(ENABLE_OVERLOADING)
data SettingsSchemaIdPropertyInfo
instance AttrInfo SettingsSchemaIdPropertyInfo where
    type AttrAllowedOps SettingsSchemaIdPropertyInfo = '[ 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint SettingsSchemaIdPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsSchemaIdPropertyInfo = (~) T.Text
    type AttrTransferTypeConstraint SettingsSchemaIdPropertyInfo = (~) T.Text
    type AttrTransferType SettingsSchemaIdPropertyInfo = T.Text
    type AttrGetType SettingsSchemaIdPropertyInfo = (Maybe T.Text)
    type AttrLabel SettingsSchemaIdPropertyInfo = "schema-id"
    type AttrOrigin SettingsSchemaIdPropertyInfo = Settings
    attrGet = getSettingsSchemaId
    attrSet = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsSchemaId
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.schemaId"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#g:attr:schemaId"
        })
#endif

-- VVV Prop "settings-schema"
   -- Type: TInterface (Name {namespace = "Gio", name = "SettingsSchema"})
   -- Flags: [PropertyReadable,PropertyWritable,PropertyConstructOnly]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@settings-schema@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' settings #settingsSchema
-- @
getSettingsSettingsSchema :: (MonadIO m, IsSettings o) => o -> m (Maybe Gio.SettingsSchema.SettingsSchema)
getSettingsSettingsSchema obj = MIO.liftIO $ B.Properties.getObjectPropertyBoxed obj "settings-schema" Gio.SettingsSchema.SettingsSchema

-- | Construct a t'GValueConstruct' with valid value for the “@settings-schema@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSettingsSettingsSchema :: (IsSettings o, MIO.MonadIO m) => Gio.SettingsSchema.SettingsSchema -> m (GValueConstruct o)
constructSettingsSettingsSchema val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBoxed "settings-schema" (P.Just val)

#if defined(ENABLE_OVERLOADING)
data SettingsSettingsSchemaPropertyInfo
instance AttrInfo SettingsSettingsSchemaPropertyInfo where
    type AttrAllowedOps SettingsSettingsSchemaPropertyInfo = '[ 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint SettingsSettingsSchemaPropertyInfo = IsSettings
    type AttrSetTypeConstraint SettingsSettingsSchemaPropertyInfo = (~) Gio.SettingsSchema.SettingsSchema
    type AttrTransferTypeConstraint SettingsSettingsSchemaPropertyInfo = (~) Gio.SettingsSchema.SettingsSchema
    type AttrTransferType SettingsSettingsSchemaPropertyInfo = Gio.SettingsSchema.SettingsSchema
    type AttrGetType SettingsSettingsSchemaPropertyInfo = (Maybe Gio.SettingsSchema.SettingsSchema)
    type AttrLabel SettingsSettingsSchemaPropertyInfo = "settings-schema"
    type AttrOrigin SettingsSettingsSchemaPropertyInfo = Settings
    attrGet = getSettingsSettingsSchema
    attrSet = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructSettingsSettingsSchema
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.settingsSchema"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#g:attr:settingsSchema"
        })
#endif

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList Settings
type instance O.AttributeList Settings = SettingsAttributeList
type SettingsAttributeList = ('[ '("backend", SettingsBackendPropertyInfo), '("delayApply", SettingsDelayApplyPropertyInfo), '("hasUnapplied", SettingsHasUnappliedPropertyInfo), '("path", SettingsPathPropertyInfo), '("schema", SettingsSchemaPropertyInfo), '("schemaId", SettingsSchemaIdPropertyInfo), '("settingsSchema", SettingsSettingsSchemaPropertyInfo)] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
settingsBackend :: AttrLabelProxy "backend"
settingsBackend = AttrLabelProxy

settingsDelayApply :: AttrLabelProxy "delayApply"
settingsDelayApply = AttrLabelProxy

settingsHasUnapplied :: AttrLabelProxy "hasUnapplied"
settingsHasUnapplied = AttrLabelProxy

settingsPath :: AttrLabelProxy "path"
settingsPath = AttrLabelProxy

settingsSchema :: AttrLabelProxy "schema"
settingsSchema = AttrLabelProxy

settingsSchemaId :: AttrLabelProxy "schemaId"
settingsSchemaId = AttrLabelProxy

settingsSettingsSchema :: AttrLabelProxy "settingsSchema"
settingsSettingsSchema = AttrLabelProxy

#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList Settings = SettingsSignalList
type SettingsSignalList = ('[ '("changeEvent", SettingsChangeEventSignalInfo), '("changed", SettingsChangedSignalInfo), '("notify", GObject.Object.ObjectNotifySignalInfo), '("writableChangeEvent", SettingsWritableChangeEventSignalInfo), '("writableChanged", SettingsWritableChangedSignalInfo)] :: [(Symbol, DK.Type)])

#endif

-- method Settings::new
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "schema_id"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the ID of the schema"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "Settings" })
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_new" g_settings_new :: 
    CString ->                              -- schema_id : TBasicType TUTF8
    IO (Ptr Settings)

-- | Creates a new t'GI.Gio.Objects.Settings.Settings' object with the schema specified by
-- /@schemaId@/.
-- 
-- It is an error for the schema to not exist: schemas are an
-- essential part of a program, as they provide type information.
-- If schemas need to be dynamically loaded (for example, from an
-- optional runtime dependency), 'GI.Gio.Structs.SettingsSchemaSource.settingsSchemaSourceLookup'
-- can be used to test for their existence before loading them.
-- 
-- Signals on the newly created t'GI.Gio.Objects.Settings.Settings' object will be dispatched
-- via the thread-default t'GI.GLib.Structs.MainContext.MainContext' in effect at the time of the
-- call to 'GI.Gio.Objects.Settings.settingsNew'.  The new t'GI.Gio.Objects.Settings.Settings' will hold a reference
-- on the context.  See 'GI.GLib.Structs.MainContext.mainContextPushThreadDefault'.
-- 
-- /Since: 2.26/
settingsNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@schemaId@/: the ID of the schema
    -> m Settings
    -- ^ __Returns:__ a new t'GI.Gio.Objects.Settings.Settings' object
settingsNew schemaId = liftIO $ do
    schemaId' <- textToCString schemaId
    result <- g_settings_new schemaId'
    checkUnexpectedReturnNULL "settingsNew" result
    result' <- (wrapObject Settings) result
    freeMem schemaId'
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method Settings::new_full
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "schema"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "SettingsSchema" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the schema describing the settings"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "backend"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "SettingsBackend" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings backend to use"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "path"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the path to use" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "Settings" })
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_new_full" g_settings_new_full :: 
    Ptr Gio.SettingsSchema.SettingsSchema -> -- schema : TInterface (Name {namespace = "Gio", name = "SettingsSchema"})
    Ptr Gio.SettingsBackend.SettingsBackend -> -- backend : TInterface (Name {namespace = "Gio", name = "SettingsBackend"})
    CString ->                              -- path : TBasicType TUTF8
    IO (Ptr Settings)

-- | Creates a new t'GI.Gio.Objects.Settings.Settings' object with a given schema, backend and
-- path.
-- 
-- It should be extremely rare that you ever want to use this function.
-- It is made available for advanced use-cases (such as plugin systems
-- that want to provide access to schemas loaded from custom locations,
-- etc).
-- 
-- At the most basic level, a t'GI.Gio.Objects.Settings.Settings' object is a pure composition of
-- four things: a t'GI.Gio.Structs.SettingsSchema.SettingsSchema', a t'GI.Gio.Objects.SettingsBackend.SettingsBackend', a path within that
-- backend, and a t'GI.GLib.Structs.MainContext.MainContext' to which signals are dispatched.
-- 
-- This constructor therefore gives you full control over constructing
-- t'GI.Gio.Objects.Settings.Settings' instances.  The first 3 parameters are given directly as
-- /@schema@/, /@backend@/ and /@path@/, and the main context is taken from the
-- thread-default (as per 'GI.Gio.Objects.Settings.settingsNew').
-- 
-- If /@backend@/ is @NULL@ then the default backend is used.
-- 
-- If /@path@/ is @NULL@ then the path from the schema is used.  It is an
-- error if /@path@/ is @NULL@ and the schema has no path of its own or if
-- /@path@/ is non-@NULL@ and not equal to the path that the schema does
-- have.
-- 
-- /Since: 2.32/
settingsNewFull ::
    (B.CallStack.HasCallStack, MonadIO m, Gio.SettingsBackend.IsSettingsBackend a) =>
    Gio.SettingsSchema.SettingsSchema
    -- ^ /@schema@/: the schema describing the settings
    -> Maybe (a)
    -- ^ /@backend@/: the settings backend to use
    -> Maybe (T.Text)
    -- ^ /@path@/: the path to use
    -> m Settings
    -- ^ __Returns:__ a new t'GI.Gio.Objects.Settings.Settings' object
settingsNewFull schema backend path = liftIO $ do
    schema' <- unsafeManagedPtrGetPtr schema
    maybeBackend <- case backend of
        Nothing -> return FP.nullPtr
        Just jBackend -> do
            jBackend' <- unsafeManagedPtrCastPtr jBackend
            return jBackend'
    maybePath <- case path of
        Nothing -> return FP.nullPtr
        Just jPath -> do
            jPath' <- textToCString jPath
            return jPath'
    result <- g_settings_new_full schema' maybeBackend maybePath
    checkUnexpectedReturnNULL "settingsNewFull" result
    result' <- (wrapObject Settings) result
    touchManagedPtr schema
    whenJust backend touchManagedPtr
    freeMem maybePath
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method Settings::new_with_backend
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "schema_id"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the ID of the schema"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "backend"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "SettingsBackend" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings backend to use"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "Settings" })
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_new_with_backend" g_settings_new_with_backend :: 
    CString ->                              -- schema_id : TBasicType TUTF8
    Ptr Gio.SettingsBackend.SettingsBackend -> -- backend : TInterface (Name {namespace = "Gio", name = "SettingsBackend"})
    IO (Ptr Settings)

-- | Creates a new t'GI.Gio.Objects.Settings.Settings' object with the schema specified by
-- /@schemaId@/ and a given t'GI.Gio.Objects.SettingsBackend.SettingsBackend'.
-- 
-- Creating a t'GI.Gio.Objects.Settings.Settings' object with a different backend allows accessing
-- settings from a database other than the usual one. For example, it may make
-- sense to pass a backend corresponding to the ‘defaults’ settings database on
-- the system to get a settings object that modifies the system default
-- settings instead of the settings for this user.
-- 
-- /Since: 2.26/
settingsNewWithBackend ::
    (B.CallStack.HasCallStack, MonadIO m, Gio.SettingsBackend.IsSettingsBackend a) =>
    T.Text
    -- ^ /@schemaId@/: the ID of the schema
    -> a
    -- ^ /@backend@/: the settings backend to use
    -> m Settings
    -- ^ __Returns:__ a new t'GI.Gio.Objects.Settings.Settings' object
settingsNewWithBackend schemaId backend = liftIO $ do
    schemaId' <- textToCString schemaId
    backend' <- unsafeManagedPtrCastPtr backend
    result <- g_settings_new_with_backend schemaId' backend'
    checkUnexpectedReturnNULL "settingsNewWithBackend" result
    result' <- (wrapObject Settings) result
    touchManagedPtr backend
    freeMem schemaId'
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method Settings::new_with_backend_and_path
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "schema_id"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the ID of the schema"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "backend"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "SettingsBackend" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings backend to use"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "path"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the path to use" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "Settings" })
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_new_with_backend_and_path" g_settings_new_with_backend_and_path :: 
    CString ->                              -- schema_id : TBasicType TUTF8
    Ptr Gio.SettingsBackend.SettingsBackend -> -- backend : TInterface (Name {namespace = "Gio", name = "SettingsBackend"})
    CString ->                              -- path : TBasicType TUTF8
    IO (Ptr Settings)

-- | Creates a new t'GI.Gio.Objects.Settings.Settings' object with the schema specified by
-- /@schemaId@/ and a given t'GI.Gio.Objects.SettingsBackend.SettingsBackend' and path.
-- 
-- This is a mix of 'GI.Gio.Objects.Settings.settingsNewWithBackend' and
-- 'GI.Gio.Objects.Settings.settingsNewWithPath'.
-- 
-- /Since: 2.26/
settingsNewWithBackendAndPath ::
    (B.CallStack.HasCallStack, MonadIO m, Gio.SettingsBackend.IsSettingsBackend a) =>
    T.Text
    -- ^ /@schemaId@/: the ID of the schema
    -> a
    -- ^ /@backend@/: the settings backend to use
    -> T.Text
    -- ^ /@path@/: the path to use
    -> m Settings
    -- ^ __Returns:__ a new t'GI.Gio.Objects.Settings.Settings' object
settingsNewWithBackendAndPath schemaId backend path = liftIO $ do
    schemaId' <- textToCString schemaId
    backend' <- unsafeManagedPtrCastPtr backend
    path' <- textToCString path
    result <- g_settings_new_with_backend_and_path schemaId' backend' path'
    checkUnexpectedReturnNULL "settingsNewWithBackendAndPath" result
    result' <- (wrapObject Settings) result
    touchManagedPtr backend
    freeMem schemaId'
    freeMem path'
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method Settings::new_with_path
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "schema_id"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the ID of the schema"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "path"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the path to use" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "Settings" })
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_new_with_path" g_settings_new_with_path :: 
    CString ->                              -- schema_id : TBasicType TUTF8
    CString ->                              -- path : TBasicType TUTF8
    IO (Ptr Settings)

-- | Creates a new t'GI.Gio.Objects.Settings.Settings' object with the relocatable schema specified
-- by /@schemaId@/ and a given path.
-- 
-- You only need to do this if you want to directly create a settings
-- object with a schema that doesn’t have a specified path of its own.
-- That’s quite rare.
-- 
-- It is a programmer error to call this function for a schema that
-- has an explicitly specified path.
-- 
-- It is a programmer error if /@path@/ is not a valid path.  A valid path
-- begins and ends with @\/@ and does not contain two consecutive @\/@
-- characters.
-- 
-- /Since: 2.26/
settingsNewWithPath ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@schemaId@/: the ID of the schema
    -> T.Text
    -- ^ /@path@/: the path to use
    -> m Settings
    -- ^ __Returns:__ a new t'GI.Gio.Objects.Settings.Settings' object
settingsNewWithPath schemaId path = liftIO $ do
    schemaId' <- textToCString schemaId
    path' <- textToCString path
    result <- g_settings_new_with_path schemaId' path'
    checkUnexpectedReturnNULL "settingsNewWithPath" result
    result' <- (wrapObject Settings) result
    freeMem schemaId'
    freeMem path'
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method Settings::apply
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "settings"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Settings" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_apply" g_settings_apply :: 
    Ptr Settings ->                         -- settings : TInterface (Name {namespace = "Gio", name = "Settings"})
    IO ()

-- | Applies any changes that have been made to the settings.
-- 
-- This function does nothing unless /@settings@/ is in
-- <http://developer.gnome.org/gio/stable/class.Settings.html#delay-apply-mode ‘delay-apply’ mode>.  In the normal
-- case settings are always applied immediately.
settingsApply ::
    (B.CallStack.HasCallStack, MonadIO m, IsSettings a) =>
    a
    -- ^ /@settings@/: the settings object
    -> m ()
settingsApply settings = liftIO $ do
    settings' <- unsafeManagedPtrCastPtr settings
    g_settings_apply settings'
    touchManagedPtr settings
    return ()

#if defined(ENABLE_OVERLOADING)
data SettingsApplyMethodInfo
instance (signature ~ (m ()), MonadIO m, IsSettings a) => O.OverloadedMethod SettingsApplyMethodInfo a signature where
    overloadedMethod = settingsApply

instance O.OverloadedMethodInfo SettingsApplyMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.settingsApply",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#v:settingsApply"
        })


#endif

-- method Settings::bind
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "settings"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Settings" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the key to bind" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "object"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "Object" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the object with property to bind"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "property"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the name of the property to bind"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "SettingsBindFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "flags for the binding"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_bind" g_settings_bind :: 
    Ptr Settings ->                         -- settings : TInterface (Name {namespace = "Gio", name = "Settings"})
    CString ->                              -- key : TBasicType TUTF8
    Ptr GObject.Object.Object ->            -- object : TInterface (Name {namespace = "GObject", name = "Object"})
    CString ->                              -- property : TBasicType TUTF8
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "SettingsBindFlags"})
    IO ()

-- | Create a binding between the /@key@/ in the /@settings@/ object
-- and the property /@property@/ of /@object@/.
-- 
-- The binding uses the default GIO mapping functions to map
-- between the settings and property values. These functions
-- handle booleans, numeric types and string types in a
-- straightforward way. Use 'GI.Gio.Objects.Settings.settingsBindWithMapping' if
-- you need a custom mapping, or map between types that are not
-- supported by the default mapping functions.
-- 
-- Unless the /@flags@/ include [flags/@gio@/.SettingsBindFlags.NO_SENSITIVITY], this
-- function also establishes a binding between the writability of
-- /@key@/ and the @sensitive@ property of /@object@/ (if /@object@/ has
-- a boolean property by that name). See 'GI.Gio.Objects.Settings.settingsBindWritable'
-- for more details about writable bindings.
-- 
-- Note that the lifecycle of the binding is tied to /@object@/,
-- and that you can have only one binding per object property.
-- If you bind the same property twice on the same object, the second
-- binding overrides the first one.
-- 
-- /Since: 2.26/
settingsBind ::
    (B.CallStack.HasCallStack, MonadIO m, IsSettings a, GObject.Object.IsObject b) =>
    a
    -- ^ /@settings@/: the settings object
    -> T.Text
    -- ^ /@key@/: the key to bind
    -> b
    -- ^ /@object@/: the object with property to bind
    -> T.Text
    -- ^ /@property@/: the name of the property to bind
    -> [Gio.Flags.SettingsBindFlags]
    -- ^ /@flags@/: flags for the binding
    -> m ()
settingsBind settings key object property flags = liftIO $ do
    settings' <- unsafeManagedPtrCastPtr settings
    key' <- textToCString key
    object' <- unsafeManagedPtrCastPtr object
    property' <- textToCString property
    let flags' = gflagsToWord flags
    g_settings_bind settings' key' object' property' flags'
    touchManagedPtr settings
    touchManagedPtr object
    freeMem key'
    freeMem property'
    return ()

#if defined(ENABLE_OVERLOADING)
data SettingsBindMethodInfo
instance (signature ~ (T.Text -> b -> T.Text -> [Gio.Flags.SettingsBindFlags] -> m ()), MonadIO m, IsSettings a, GObject.Object.IsObject b) => O.OverloadedMethod SettingsBindMethodInfo a signature where
    overloadedMethod = settingsBind

instance O.OverloadedMethodInfo SettingsBindMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.settingsBind",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#v:settingsBind"
        })


#endif

-- method Settings::bind_with_mapping
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "settings"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Settings" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the key to bind" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "object"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "Object" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the object with property to bind"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "property"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the name of the property to bind"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "SettingsBindFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "flags for the binding"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "get_mapping"
--           , argType = TGClosure Nothing
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a function that gets called to convert values\n  from @settings to @object, or `NULL` to use the default GIO mapping"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "set_mapping"
--           , argType = TGClosure Nothing
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a function that gets called to convert values\n  from @object to @settings, or `NULL` to use the default GIO mapping"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_bind_with_mapping_closures" g_settings_bind_with_mapping_closures :: 
    Ptr Settings ->                         -- settings : TInterface (Name {namespace = "Gio", name = "Settings"})
    CString ->                              -- key : TBasicType TUTF8
    Ptr GObject.Object.Object ->            -- object : TInterface (Name {namespace = "GObject", name = "Object"})
    CString ->                              -- property : TBasicType TUTF8
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "SettingsBindFlags"})
    Ptr (GClosure ()) ->                    -- get_mapping : TGClosure Nothing
    Ptr (GClosure ()) ->                    -- set_mapping : TGClosure Nothing
    IO ()

-- | Version of 'GI.Gio.Objects.Settings.settingsBindWithMapping' using closures instead of
-- callbacks for easier binding in other languages.
-- 
-- /Since: 2.82/
settingsBindWithMapping ::
    (B.CallStack.HasCallStack, MonadIO m, IsSettings a, GObject.Object.IsObject b) =>
    a
    -- ^ /@settings@/: the settings object
    -> T.Text
    -- ^ /@key@/: the key to bind
    -> b
    -- ^ /@object@/: the object with property to bind
    -> T.Text
    -- ^ /@property@/: the name of the property to bind
    -> [Gio.Flags.SettingsBindFlags]
    -- ^ /@flags@/: flags for the binding
    -> Maybe (GClosure c)
    -- ^ /@getMapping@/: a function that gets called to convert values
    --   from /@settings@/ to /@object@/, or @NULL@ to use the default GIO mapping
    -> Maybe (GClosure d)
    -- ^ /@setMapping@/: a function that gets called to convert values
    --   from /@object@/ to /@settings@/, or @NULL@ to use the default GIO mapping
    -> m ()
settingsBindWithMapping settings key object property flags getMapping setMapping = liftIO $ do
    settings' <- unsafeManagedPtrCastPtr settings
    key' <- textToCString key
    object' <- unsafeManagedPtrCastPtr object
    property' <- textToCString property
    let flags' = gflagsToWord flags
    maybeGetMapping <- case getMapping of
        Nothing -> return FP.nullPtr
        Just jGetMapping -> do
            jGetMapping' <- unsafeManagedPtrCastPtr jGetMapping
            return jGetMapping'
    maybeSetMapping <- case setMapping of
        Nothing -> return FP.nullPtr
        Just jSetMapping -> do
            jSetMapping' <- unsafeManagedPtrCastPtr jSetMapping
            return jSetMapping'
    g_settings_bind_with_mapping_closures settings' key' object' property' flags' maybeGetMapping maybeSetMapping
    touchManagedPtr settings
    touchManagedPtr object
    whenJust getMapping touchManagedPtr
    whenJust setMapping touchManagedPtr
    freeMem key'
    freeMem property'
    return ()

#if defined(ENABLE_OVERLOADING)
data SettingsBindWithMappingMethodInfo
instance (signature ~ (T.Text -> b -> T.Text -> [Gio.Flags.SettingsBindFlags] -> Maybe (GClosure c) -> Maybe (GClosure d) -> m ()), MonadIO m, IsSettings a, GObject.Object.IsObject b) => O.OverloadedMethod SettingsBindWithMappingMethodInfo a signature where
    overloadedMethod = settingsBindWithMapping

instance O.OverloadedMethodInfo SettingsBindWithMappingMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.settingsBindWithMapping",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#v:settingsBindWithMapping"
        })


#endif

-- method Settings::bind_writable
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "settings"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Settings" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the key to bind" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "object"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "Object" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the object with property to bind"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "property"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the name of a boolean property to bind"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "inverted"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "whether to \8216invert\8217 the value"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_bind_writable" g_settings_bind_writable :: 
    Ptr Settings ->                         -- settings : TInterface (Name {namespace = "Gio", name = "Settings"})
    CString ->                              -- key : TBasicType TUTF8
    Ptr GObject.Object.Object ->            -- object : TInterface (Name {namespace = "GObject", name = "Object"})
    CString ->                              -- property : TBasicType TUTF8
    CInt ->                                 -- inverted : TBasicType TBoolean
    IO ()

-- | Create a binding between the writability of /@key@/ in the
-- /@settings@/ object and the property /@property@/ of /@object@/.
-- 
-- The property must be boolean; @sensitive@ or @visible@
-- properties of widgets are the most likely candidates.
-- 
-- Writable bindings are always uni-directional; changes of the
-- writability of the setting will be propagated to the object
-- property, not the other way.
-- 
-- When the /@inverted@/ argument is true, the binding inverts the
-- value as it passes from the setting to the object, i.e. /@property@/
-- will be set to true if the key is not writable.
-- 
-- Note that the lifecycle of the binding is tied to /@object@/,
-- and that you can have only one binding per object property.
-- If you bind the same property twice on the same object, the second
-- binding overrides the first one.
-- 
-- /Since: 2.26/
settingsBindWritable ::
    (B.CallStack.HasCallStack, MonadIO m, IsSettings a, GObject.Object.IsObject b) =>
    a
    -- ^ /@settings@/: the settings object
    -> T.Text
    -- ^ /@key@/: the key to bind
    -> b
    -- ^ /@object@/: the object with property to bind
    -> T.Text
    -- ^ /@property@/: the name of a boolean property to bind
    -> Bool
    -- ^ /@inverted@/: whether to ‘invert’ the value
    -> m ()
settingsBindWritable settings key object property inverted = liftIO $ do
    settings' <- unsafeManagedPtrCastPtr settings
    key' <- textToCString key
    object' <- unsafeManagedPtrCastPtr object
    property' <- textToCString property
    let inverted' = (P.fromIntegral . P.fromEnum) inverted
    g_settings_bind_writable settings' key' object' property' inverted'
    touchManagedPtr settings
    touchManagedPtr object
    freeMem key'
    freeMem property'
    return ()

#if defined(ENABLE_OVERLOADING)
data SettingsBindWritableMethodInfo
instance (signature ~ (T.Text -> b -> T.Text -> Bool -> m ()), MonadIO m, IsSettings a, GObject.Object.IsObject b) => O.OverloadedMethod SettingsBindWritableMethodInfo a signature where
    overloadedMethod = settingsBindWritable

instance O.OverloadedMethodInfo SettingsBindWritableMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.settingsBindWritable",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#v:settingsBindWritable"
        })


#endif

-- method Settings::create_action
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "settings"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Settings" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the name of a key in @settings"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "Action" })
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_create_action" g_settings_create_action :: 
    Ptr Settings ->                         -- settings : TInterface (Name {namespace = "Gio", name = "Settings"})
    CString ->                              -- key : TBasicType TUTF8
    IO (Ptr Gio.Action.Action)

-- | Creates a t'GI.Gio.Interfaces.Action.Action' corresponding to a given t'GI.Gio.Objects.Settings.Settings' key.
-- 
-- The action has the same name as the key.
-- 
-- The value of the key becomes the state of the action and the action
-- is enabled when the key is writable.  Changing the state of the
-- action results in the key being written to.  Changes to the value or
-- writability of the key cause appropriate change notifications to be
-- emitted for the action.
-- 
-- For boolean-valued keys, action activations take no parameter and
-- result in the toggling of the value.  For all other types,
-- activations take the new value for the key (which must have the
-- correct type).
-- 
-- /Since: 2.32/
settingsCreateAction ::
    (B.CallStack.HasCallStack, MonadIO m, IsSettings a) =>
    a
    -- ^ /@settings@/: the settings object
    -> T.Text
    -- ^ /@key@/: the name of a key in /@settings@/
    -> m Gio.Action.Action
    -- ^ __Returns:__ a new t'GI.Gio.Interfaces.Action.Action'
settingsCreateAction settings key = liftIO $ do
    settings' <- unsafeManagedPtrCastPtr settings
    key' <- textToCString key
    result <- g_settings_create_action settings' key'
    checkUnexpectedReturnNULL "settingsCreateAction" result
    result' <- (wrapObject Gio.Action.Action) result
    touchManagedPtr settings
    freeMem key'
    return result'

#if defined(ENABLE_OVERLOADING)
data SettingsCreateActionMethodInfo
instance (signature ~ (T.Text -> m Gio.Action.Action), MonadIO m, IsSettings a) => O.OverloadedMethod SettingsCreateActionMethodInfo a signature where
    overloadedMethod = settingsCreateAction

instance O.OverloadedMethodInfo SettingsCreateActionMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.settingsCreateAction",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#v:settingsCreateAction"
        })


#endif

-- method Settings::delay
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "settings"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Settings" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_delay" g_settings_delay :: 
    Ptr Settings ->                         -- settings : TInterface (Name {namespace = "Gio", name = "Settings"})
    IO ()

-- | Changes the t'GI.Gio.Objects.Settings.Settings' object into
-- <http://developer.gnome.org/gio/stable/class.Settings.html#delay-apply-mode ‘delay-apply’ mode>.
-- 
-- In this
-- mode, changes to /@settings@/ are not immediately propagated to the
-- backend, but kept locally until 'GI.Gio.Objects.Settings.settingsApply' is called.
-- 
-- /Since: 2.26/
settingsDelay ::
    (B.CallStack.HasCallStack, MonadIO m, IsSettings a) =>
    a
    -- ^ /@settings@/: the settings object
    -> m ()
settingsDelay settings = liftIO $ do
    settings' <- unsafeManagedPtrCastPtr settings
    g_settings_delay settings'
    touchManagedPtr settings
    return ()

#if defined(ENABLE_OVERLOADING)
data SettingsDelayMethodInfo
instance (signature ~ (m ()), MonadIO m, IsSettings a) => O.OverloadedMethod SettingsDelayMethodInfo a signature where
    overloadedMethod = settingsDelay

instance O.OverloadedMethodInfo SettingsDelayMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.settingsDelay",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#v:settingsDelay"
        })


#endif

-- method Settings::get_boolean
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "settings"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Settings" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the key to get the value for"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_get_boolean" g_settings_get_boolean :: 
    Ptr Settings ->                         -- settings : TInterface (Name {namespace = "Gio", name = "Settings"})
    CString ->                              -- key : TBasicType TUTF8
    IO CInt

-- | Gets the value that is stored at /@key@/ in /@settings@/.
-- 
-- A convenience variant of t'GI.Gio.Objects.Settings.Settings'.@/get/@() for booleans.
-- 
-- It is a programmer error to give a /@key@/ that isn’t specified as
-- having a @b@ type in the schema for /@settings@/ (see t'GI.GLib.Structs.VariantType.VariantType').
-- 
-- /Since: 2.26/
settingsGetBoolean ::
    (B.CallStack.HasCallStack, MonadIO m, IsSettings a) =>
    a
    -- ^ /@settings@/: the settings object
    -> T.Text
    -- ^ /@key@/: the key to get the value for
    -> m Bool
    -- ^ __Returns:__ a boolean
settingsGetBoolean settings key = liftIO $ do
    settings' <- unsafeManagedPtrCastPtr settings
    key' <- textToCString key
    result <- g_settings_get_boolean settings' key'
    let result' = (/= 0) result
    touchManagedPtr settings
    freeMem key'
    return result'

#if defined(ENABLE_OVERLOADING)
data SettingsGetBooleanMethodInfo
instance (signature ~ (T.Text -> m Bool), MonadIO m, IsSettings a) => O.OverloadedMethod SettingsGetBooleanMethodInfo a signature where
    overloadedMethod = settingsGetBoolean

instance O.OverloadedMethodInfo SettingsGetBooleanMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.settingsGetBoolean",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#v:settingsGetBoolean"
        })


#endif

-- method Settings::get_child
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "settings"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Settings" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the name of the child schema"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "Settings" })
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_get_child" g_settings_get_child :: 
    Ptr Settings ->                         -- settings : TInterface (Name {namespace = "Gio", name = "Settings"})
    CString ->                              -- name : TBasicType TUTF8
    IO (Ptr Settings)

-- | Creates a child settings object which has a base path of
-- @base-path\/name@, where @base-path@ is the base path of
-- /@settings@/ and @name@ is as specified by the caller.
-- 
-- The schema for the child settings object must have been declared
-- in the schema of /@settings@/ using a @\<child>@ element.
-- 
-- The created child settings object will inherit the
-- [Settings:delayApply]("GI.Gio.Objects.Settings#g:attr:delayApply") mode from /@settings@/.
-- 
-- /Since: 2.26/
settingsGetChild ::
    (B.CallStack.HasCallStack, MonadIO m, IsSettings a) =>
    a
    -- ^ /@settings@/: the settings object
    -> T.Text
    -- ^ /@name@/: the name of the child schema
    -> m Settings
    -- ^ __Returns:__ a ‘child’ settings object
settingsGetChild settings name = liftIO $ do
    settings' <- unsafeManagedPtrCastPtr settings
    name' <- textToCString name
    result <- g_settings_get_child settings' name'
    checkUnexpectedReturnNULL "settingsGetChild" result
    result' <- (wrapObject Settings) result
    touchManagedPtr settings
    freeMem name'
    return result'

#if defined(ENABLE_OVERLOADING)
data SettingsGetChildMethodInfo
instance (signature ~ (T.Text -> m Settings), MonadIO m, IsSettings a) => O.OverloadedMethod SettingsGetChildMethodInfo a signature where
    overloadedMethod = settingsGetChild

instance O.OverloadedMethodInfo SettingsGetChildMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.settingsGetChild",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#v:settingsGetChild"
        })


#endif

-- method Settings::get_default_value
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "settings"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Settings" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the key to get the default value for"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just TVariant
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_get_default_value" g_settings_get_default_value :: 
    Ptr Settings ->                         -- settings : TInterface (Name {namespace = "Gio", name = "Settings"})
    CString ->                              -- key : TBasicType TUTF8
    IO (Ptr GVariant)

-- | Gets the ‘default value’ of a key.
-- 
-- This is the value that would be read if 'GI.Gio.Objects.Settings.settingsReset' were to be
-- called on the key.
-- 
-- Note that this may be a different value than returned by
-- 'GI.Gio.Structs.SettingsSchemaKey.settingsSchemaKeyGetDefaultValue' if the system administrator
-- has provided a default value.
-- 
-- Comparing the return values of 'GI.Gio.Objects.Settings.settingsGetDefaultValue' and
-- 'GI.Gio.Objects.Settings.settingsGetValue' is not sufficient for determining if a value
-- has been set because the user may have explicitly set the value to
-- something that happens to be equal to the default.  The difference
-- here is that if the default changes in the future, the user’s key
-- will still be set.
-- 
-- This function may be useful for adding an indication to a UI of what
-- the default value was before the user set it.
-- 
-- It is a programmer error to give a /@key@/ that isn’t contained in the
-- schema for /@settings@/.
-- 
-- /Since: 2.40/
settingsGetDefaultValue ::
    (B.CallStack.HasCallStack, MonadIO m, IsSettings a) =>
    a
    -- ^ /@settings@/: the settings object
    -> T.Text
    -- ^ /@key@/: the key to get the default value for
    -> m (Maybe GVariant)
    -- ^ __Returns:__ the default value
settingsGetDefaultValue settings key = liftIO $ do
    settings' <- unsafeManagedPtrCastPtr settings
    key' <- textToCString key
    result <- g_settings_get_default_value settings' key'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- B.GVariant.wrapGVariantPtr result'
        return result''
    touchManagedPtr settings
    freeMem key'
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data SettingsGetDefaultValueMethodInfo
instance (signature ~ (T.Text -> m (Maybe GVariant)), MonadIO m, IsSettings a) => O.OverloadedMethod SettingsGetDefaultValueMethodInfo a signature where
    overloadedMethod = settingsGetDefaultValue

instance O.OverloadedMethodInfo SettingsGetDefaultValueMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.settingsGetDefaultValue",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#v:settingsGetDefaultValue"
        })


#endif

-- method Settings::get_double
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "settings"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Settings" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the key to get the value for"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TDouble)
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_get_double" g_settings_get_double :: 
    Ptr Settings ->                         -- settings : TInterface (Name {namespace = "Gio", name = "Settings"})
    CString ->                              -- key : TBasicType TUTF8
    IO CDouble

-- | Gets the value that is stored at /@key@/ in /@settings@/.
-- 
-- A convenience variant of t'GI.Gio.Objects.Settings.Settings'.@/get/@() for doubles.
-- 
-- It is a programmer error to give a /@key@/ that isn’t specified as
-- having a @d@ type in the schema for /@settings@/ (see t'GI.GLib.Structs.VariantType.VariantType').
-- 
-- /Since: 2.26/
settingsGetDouble ::
    (B.CallStack.HasCallStack, MonadIO m, IsSettings a) =>
    a
    -- ^ /@settings@/: the settings object
    -> T.Text
    -- ^ /@key@/: the key to get the value for
    -> m Double
    -- ^ __Returns:__ a double
settingsGetDouble settings key = liftIO $ do
    settings' <- unsafeManagedPtrCastPtr settings
    key' <- textToCString key
    result <- g_settings_get_double settings' key'
    let result' = realToFrac result
    touchManagedPtr settings
    freeMem key'
    return result'

#if defined(ENABLE_OVERLOADING)
data SettingsGetDoubleMethodInfo
instance (signature ~ (T.Text -> m Double), MonadIO m, IsSettings a) => O.OverloadedMethod SettingsGetDoubleMethodInfo a signature where
    overloadedMethod = settingsGetDouble

instance O.OverloadedMethodInfo SettingsGetDoubleMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.settingsGetDouble",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#v:settingsGetDouble"
        })


#endif

-- method Settings::get_enum
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "settings"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Settings" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the key to get the value for"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_get_enum" g_settings_get_enum :: 
    Ptr Settings ->                         -- settings : TInterface (Name {namespace = "Gio", name = "Settings"})
    CString ->                              -- key : TBasicType TUTF8
    IO Int32

-- | Gets the value that is stored in /@settings@/ for /@key@/ and converts it
-- to the enum value that it represents.
-- 
-- In order to use this function the type of the value must be a string
-- and it must be marked in the schema file as an enumerated type.
-- 
-- It is a programmer error to give a /@key@/ that isn’t contained in the
-- schema for /@settings@/ or is not marked as an enumerated type.
-- 
-- If the value stored in the configuration database is not a valid
-- value for the enumerated type then this function will return the
-- default value.
-- 
-- /Since: 2.26/
settingsGetEnum ::
    (B.CallStack.HasCallStack, MonadIO m, IsSettings a) =>
    a
    -- ^ /@settings@/: the settings object
    -> T.Text
    -- ^ /@key@/: the key to get the value for
    -> m Int32
    -- ^ __Returns:__ the enum value
settingsGetEnum settings key = liftIO $ do
    settings' <- unsafeManagedPtrCastPtr settings
    key' <- textToCString key
    result <- g_settings_get_enum settings' key'
    touchManagedPtr settings
    freeMem key'
    return result

#if defined(ENABLE_OVERLOADING)
data SettingsGetEnumMethodInfo
instance (signature ~ (T.Text -> m Int32), MonadIO m, IsSettings a) => O.OverloadedMethod SettingsGetEnumMethodInfo a signature where
    overloadedMethod = settingsGetEnum

instance O.OverloadedMethodInfo SettingsGetEnumMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.settingsGetEnum",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#v:settingsGetEnum"
        })


#endif

-- method Settings::get_flags
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "settings"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Settings" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the key to get the value for"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_get_flags" g_settings_get_flags :: 
    Ptr Settings ->                         -- settings : TInterface (Name {namespace = "Gio", name = "Settings"})
    CString ->                              -- key : TBasicType TUTF8
    IO Word32

-- | Gets the value that is stored in /@settings@/ for /@key@/ and converts it
-- to the flags value that it represents.
-- 
-- In order to use this function the type of the value must be an array
-- of strings and it must be marked in the schema file as a flags type.
-- 
-- It is a programmer error to give a /@key@/ that isn’t contained in the
-- schema for /@settings@/ or is not marked as a flags type.
-- 
-- If the value stored in the configuration database is not a valid
-- value for the flags type then this function will return the default
-- value.
-- 
-- /Since: 2.26/
settingsGetFlags ::
    (B.CallStack.HasCallStack, MonadIO m, IsSettings a) =>
    a
    -- ^ /@settings@/: the settings object
    -> T.Text
    -- ^ /@key@/: the key to get the value for
    -> m Word32
    -- ^ __Returns:__ the flags value
settingsGetFlags settings key = liftIO $ do
    settings' <- unsafeManagedPtrCastPtr settings
    key' <- textToCString key
    result <- g_settings_get_flags settings' key'
    touchManagedPtr settings
    freeMem key'
    return result

#if defined(ENABLE_OVERLOADING)
data SettingsGetFlagsMethodInfo
instance (signature ~ (T.Text -> m Word32), MonadIO m, IsSettings a) => O.OverloadedMethod SettingsGetFlagsMethodInfo a signature where
    overloadedMethod = settingsGetFlags

instance O.OverloadedMethodInfo SettingsGetFlagsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.settingsGetFlags",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#v:settingsGetFlags"
        })


#endif

-- method Settings::get_has_unapplied
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "settings"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Settings" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_get_has_unapplied" g_settings_get_has_unapplied :: 
    Ptr Settings ->                         -- settings : TInterface (Name {namespace = "Gio", name = "Settings"})
    IO CInt

-- | Returns whether the t'GI.Gio.Objects.Settings.Settings' object has any unapplied
-- changes.
-- 
-- This can only be the case if it is in
-- <http://developer.gnome.org/gio/stable/class.Settings.html#delay-apply-mode ‘delay-apply’ mode>.
-- 
-- /Since: 2.26/
settingsGetHasUnapplied ::
    (B.CallStack.HasCallStack, MonadIO m, IsSettings a) =>
    a
    -- ^ /@settings@/: the settings object
    -> m Bool
    -- ^ __Returns:__ true if /@settings@/ has unapplied changes, false otherwise
settingsGetHasUnapplied settings = liftIO $ do
    settings' <- unsafeManagedPtrCastPtr settings
    result <- g_settings_get_has_unapplied settings'
    let result' = (/= 0) result
    touchManagedPtr settings
    return result'

#if defined(ENABLE_OVERLOADING)
data SettingsGetHasUnappliedMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsSettings a) => O.OverloadedMethod SettingsGetHasUnappliedMethodInfo a signature where
    overloadedMethod = settingsGetHasUnapplied

instance O.OverloadedMethodInfo SettingsGetHasUnappliedMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.settingsGetHasUnapplied",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#v:settingsGetHasUnapplied"
        })


#endif

-- method Settings::get_int
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "settings"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Settings" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the key to get the value for"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_get_int" g_settings_get_int :: 
    Ptr Settings ->                         -- settings : TInterface (Name {namespace = "Gio", name = "Settings"})
    CString ->                              -- key : TBasicType TUTF8
    IO Int32

-- | Gets the value that is stored at /@key@/ in /@settings@/.
-- 
-- A convenience variant of t'GI.Gio.Objects.Settings.Settings'.@/get/@() for 32-bit integers.
-- 
-- It is a programmer error to give a /@key@/ that isn’t specified as
-- having an @i@ type in the schema for /@settings@/ (see t'GI.GLib.Structs.VariantType.VariantType').
-- 
-- /Since: 2.26/
settingsGetInt ::
    (B.CallStack.HasCallStack, MonadIO m, IsSettings a) =>
    a
    -- ^ /@settings@/: the settings object
    -> T.Text
    -- ^ /@key@/: the key to get the value for
    -> m Int32
    -- ^ __Returns:__ an integer
settingsGetInt settings key = liftIO $ do
    settings' <- unsafeManagedPtrCastPtr settings
    key' <- textToCString key
    result <- g_settings_get_int settings' key'
    touchManagedPtr settings
    freeMem key'
    return result

#if defined(ENABLE_OVERLOADING)
data SettingsGetIntMethodInfo
instance (signature ~ (T.Text -> m Int32), MonadIO m, IsSettings a) => O.OverloadedMethod SettingsGetIntMethodInfo a signature where
    overloadedMethod = settingsGetInt

instance O.OverloadedMethodInfo SettingsGetIntMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.settingsGetInt",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#v:settingsGetInt"
        })


#endif

-- method Settings::get_int64
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "settings"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Settings" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the key to get the value for"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt64)
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_get_int64" g_settings_get_int64 :: 
    Ptr Settings ->                         -- settings : TInterface (Name {namespace = "Gio", name = "Settings"})
    CString ->                              -- key : TBasicType TUTF8
    IO Int64

-- | Gets the value that is stored at /@key@/ in /@settings@/.
-- 
-- A convenience variant of t'GI.Gio.Objects.Settings.Settings'.@/get/@() for 64-bit integers.
-- 
-- It is a programmer error to give a /@key@/ that isn’t specified as
-- having an @x@ type in the schema for /@settings@/ (see t'GI.GLib.Structs.VariantType.VariantType').
-- 
-- /Since: 2.50/
settingsGetInt64 ::
    (B.CallStack.HasCallStack, MonadIO m, IsSettings a) =>
    a
    -- ^ /@settings@/: the settings object
    -> T.Text
    -- ^ /@key@/: the key to get the value for
    -> m Int64
    -- ^ __Returns:__ a 64-bit integer
settingsGetInt64 settings key = liftIO $ do
    settings' <- unsafeManagedPtrCastPtr settings
    key' <- textToCString key
    result <- g_settings_get_int64 settings' key'
    touchManagedPtr settings
    freeMem key'
    return result

#if defined(ENABLE_OVERLOADING)
data SettingsGetInt64MethodInfo
instance (signature ~ (T.Text -> m Int64), MonadIO m, IsSettings a) => O.OverloadedMethod SettingsGetInt64MethodInfo a signature where
    overloadedMethod = settingsGetInt64

instance O.OverloadedMethodInfo SettingsGetInt64MethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.settingsGetInt64",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#v:settingsGetInt64"
        })


#endif

-- method Settings::get_mapped
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "settings"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Settings" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the key to get the value for"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "mapping"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "SettingsGetMapping" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the function to map the value in the\n  settings database to the value used by the application"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeCall
--           , argClosure = 3
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "user data for @mapping"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TPtr)
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_get_mapped" g_settings_get_mapped :: 
    Ptr Settings ->                         -- settings : TInterface (Name {namespace = "Gio", name = "Settings"})
    CString ->                              -- key : TBasicType TUTF8
    FunPtr Gio.Callbacks.C_SettingsGetMapping -> -- mapping : TInterface (Name {namespace = "Gio", name = "SettingsGetMapping"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO (Ptr ())

-- | Gets the value that is stored at /@key@/ in /@settings@/, subject to
-- application-level validation\/mapping.
-- 
-- You should use this function when the application needs to perform
-- some processing on the value of the key (for example, parsing).  The
-- /@mapping@/ function performs that processing.  If the function
-- indicates that the processing was unsuccessful (due to a parse error,
-- for example) then the mapping is tried again with another value.
-- 
-- This allows a robust ‘fall back to defaults’ behaviour to be
-- implemented somewhat automatically.
-- 
-- The first value that is tried is the user’s setting for the key.  If
-- the mapping function fails to map this value, other values may be
-- tried in an unspecified order (system or site defaults, translated
-- schema default values, untranslated schema default values, etc).
-- 
-- If the mapping function fails for all possible values, one additional
-- attempt is made: the mapping function is called with a @NULL@ value.
-- If the mapping function still indicates failure at this point then
-- the application will be aborted.
-- 
-- The result parameter for the /@mapping@/ function is pointed to a
-- @gpointer@ which is initially set to @NULL@.  The same pointer is given
-- to each invocation of /@mapping@/.  The final value of that @gpointer@ is
-- what is returned by this function.  @NULL@ is valid; it is returned
-- just as any other value would be.
settingsGetMapped ::
    (B.CallStack.HasCallStack, MonadIO m, IsSettings a) =>
    a
    -- ^ /@settings@/: the settings object
    -> T.Text
    -- ^ /@key@/: the key to get the value for
    -> Gio.Callbacks.SettingsGetMapping
    -- ^ /@mapping@/: the function to map the value in the
    --   settings database to the value used by the application
    -> m (Ptr ())
    -- ^ __Returns:__ the result, which may be @NULL@
settingsGetMapped settings key mapping = liftIO $ do
    settings' <- unsafeManagedPtrCastPtr settings
    key' <- textToCString key
    mapping' <- Gio.Callbacks.mk_SettingsGetMapping (Gio.Callbacks.wrap_SettingsGetMapping Nothing (Gio.Callbacks.drop_closures_SettingsGetMapping mapping))
    let userData = nullPtr
    result <- g_settings_get_mapped settings' key' mapping' userData
    safeFreeFunPtr $ castFunPtrToPtr mapping'
    touchManagedPtr settings
    freeMem key'
    return result

#if defined(ENABLE_OVERLOADING)
data SettingsGetMappedMethodInfo
instance (signature ~ (T.Text -> Gio.Callbacks.SettingsGetMapping -> m (Ptr ())), MonadIO m, IsSettings a) => O.OverloadedMethod SettingsGetMappedMethodInfo a signature where
    overloadedMethod = settingsGetMapped

instance O.OverloadedMethodInfo SettingsGetMappedMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.settingsGetMapped",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#v:settingsGetMapped"
        })


#endif

-- method Settings::get_range
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "settings"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Settings" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the key to query the range of"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just TVariant
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_get_range" g_settings_get_range :: 
    Ptr Settings ->                         -- settings : TInterface (Name {namespace = "Gio", name = "Settings"})
    CString ->                              -- key : TBasicType TUTF8
    IO (Ptr GVariant)

{-# DEPRECATED settingsGetRange ["(Since version 2.40)","Use 'GI.Gio.Structs.SettingsSchemaKey.settingsSchemaKeyGetRange' instead."] #-}
-- | Queries the range of a key.
-- 
-- /Since: 2.28/
settingsGetRange ::
    (B.CallStack.HasCallStack, MonadIO m, IsSettings a) =>
    a
    -- ^ /@settings@/: the settings object
    -> T.Text
    -- ^ /@key@/: the key to query the range of
    -> m GVariant
settingsGetRange settings key = liftIO $ do
    settings' <- unsafeManagedPtrCastPtr settings
    key' <- textToCString key
    result <- g_settings_get_range settings' key'
    checkUnexpectedReturnNULL "settingsGetRange" result
    result' <- B.GVariant.wrapGVariantPtr result
    touchManagedPtr settings
    freeMem key'
    return result'

#if defined(ENABLE_OVERLOADING)
data SettingsGetRangeMethodInfo
instance (signature ~ (T.Text -> m GVariant), MonadIO m, IsSettings a) => O.OverloadedMethod SettingsGetRangeMethodInfo a signature where
    overloadedMethod = settingsGetRange

instance O.OverloadedMethodInfo SettingsGetRangeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.settingsGetRange",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#v:settingsGetRange"
        })


#endif

-- method Settings::get_string
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "settings"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Settings" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the key to get the value for"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_get_string" g_settings_get_string :: 
    Ptr Settings ->                         -- settings : TInterface (Name {namespace = "Gio", name = "Settings"})
    CString ->                              -- key : TBasicType TUTF8
    IO CString

-- | Gets the value that is stored at /@key@/ in /@settings@/.
-- 
-- A convenience variant of t'GI.Gio.Objects.Settings.Settings'.@/get/@() for strings.
-- 
-- It is a programmer error to give a /@key@/ that isn’t specified as
-- having an @s@ type in the schema for /@settings@/ (see t'GI.GLib.Structs.VariantType.VariantType').
-- 
-- /Since: 2.26/
settingsGetString ::
    (B.CallStack.HasCallStack, MonadIO m, IsSettings a) =>
    a
    -- ^ /@settings@/: the settings object
    -> T.Text
    -- ^ /@key@/: the key to get the value for
    -> m T.Text
    -- ^ __Returns:__ a newly-allocated string
settingsGetString settings key = liftIO $ do
    settings' <- unsafeManagedPtrCastPtr settings
    key' <- textToCString key
    result <- g_settings_get_string settings' key'
    checkUnexpectedReturnNULL "settingsGetString" result
    result' <- cstringToText result
    freeMem result
    touchManagedPtr settings
    freeMem key'
    return result'

#if defined(ENABLE_OVERLOADING)
data SettingsGetStringMethodInfo
instance (signature ~ (T.Text -> m T.Text), MonadIO m, IsSettings a) => O.OverloadedMethod SettingsGetStringMethodInfo a signature where
    overloadedMethod = settingsGetString

instance O.OverloadedMethodInfo SettingsGetStringMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.settingsGetString",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#v:settingsGetString"
        })


#endif

-- method Settings::get_strv
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "settings"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Settings" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the key to get the value for"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TCArray True (-1) (-1) (TBasicType TUTF8))
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_get_strv" g_settings_get_strv :: 
    Ptr Settings ->                         -- settings : TInterface (Name {namespace = "Gio", name = "Settings"})
    CString ->                              -- key : TBasicType TUTF8
    IO (Ptr CString)

-- | A convenience variant of t'GI.Gio.Objects.Settings.Settings'.@/get/@() for string arrays.
-- 
-- It is a programmer error to give a /@key@/ that isn’t specified as
-- having an @as@ type in the schema for /@settings@/ (see t'GI.GLib.Structs.VariantType.VariantType').
-- 
-- /Since: 2.26/
settingsGetStrv ::
    (B.CallStack.HasCallStack, MonadIO m, IsSettings a) =>
    a
    -- ^ /@settings@/: the settings object
    -> T.Text
    -- ^ /@key@/: the key to get the value for
    -> m [T.Text]
    -- ^ __Returns:__ a
    --   newly-allocated, @NULL@-terminated array of strings, the value that
    --   is stored at /@key@/ in /@settings@/.
settingsGetStrv settings key = liftIO $ do
    settings' <- unsafeManagedPtrCastPtr settings
    key' <- textToCString key
    result <- g_settings_get_strv settings' key'
    checkUnexpectedReturnNULL "settingsGetStrv" result
    result' <- unpackZeroTerminatedUTF8CArray result
    mapZeroTerminatedCArray freeMem result
    freeMem result
    touchManagedPtr settings
    freeMem key'
    return result'

#if defined(ENABLE_OVERLOADING)
data SettingsGetStrvMethodInfo
instance (signature ~ (T.Text -> m [T.Text]), MonadIO m, IsSettings a) => O.OverloadedMethod SettingsGetStrvMethodInfo a signature where
    overloadedMethod = settingsGetStrv

instance O.OverloadedMethodInfo SettingsGetStrvMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.settingsGetStrv",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#v:settingsGetStrv"
        })


#endif

-- method Settings::get_uint
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "settings"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Settings" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the key to get the value for"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_get_uint" g_settings_get_uint :: 
    Ptr Settings ->                         -- settings : TInterface (Name {namespace = "Gio", name = "Settings"})
    CString ->                              -- key : TBasicType TUTF8
    IO Word32

-- | Gets the value that is stored at /@key@/ in /@settings@/.
-- 
-- A convenience variant of t'GI.Gio.Objects.Settings.Settings'.@/get/@() for 32-bit unsigned
-- integers.
-- 
-- It is a programmer error to give a /@key@/ that isn’t specified as
-- having a @u@ type in the schema for /@settings@/ (see t'GI.GLib.Structs.VariantType.VariantType').
-- 
-- /Since: 2.30/
settingsGetUint ::
    (B.CallStack.HasCallStack, MonadIO m, IsSettings a) =>
    a
    -- ^ /@settings@/: the settings object
    -> T.Text
    -- ^ /@key@/: the key to get the value for
    -> m Word32
    -- ^ __Returns:__ an unsigned integer
settingsGetUint settings key = liftIO $ do
    settings' <- unsafeManagedPtrCastPtr settings
    key' <- textToCString key
    result <- g_settings_get_uint settings' key'
    touchManagedPtr settings
    freeMem key'
    return result

#if defined(ENABLE_OVERLOADING)
data SettingsGetUintMethodInfo
instance (signature ~ (T.Text -> m Word32), MonadIO m, IsSettings a) => O.OverloadedMethod SettingsGetUintMethodInfo a signature where
    overloadedMethod = settingsGetUint

instance O.OverloadedMethodInfo SettingsGetUintMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.settingsGetUint",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#v:settingsGetUint"
        })


#endif

-- method Settings::get_uint64
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "settings"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Settings" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the key to get the value for"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUInt64)
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_get_uint64" g_settings_get_uint64 :: 
    Ptr Settings ->                         -- settings : TInterface (Name {namespace = "Gio", name = "Settings"})
    CString ->                              -- key : TBasicType TUTF8
    IO Word64

-- | Gets the value that is stored at /@key@/ in /@settings@/.
-- 
-- A convenience variant of t'GI.Gio.Objects.Settings.Settings'.@/get/@() for 64-bit unsigned
-- integers.
-- 
-- It is a programmer error to give a /@key@/ that isn’t specified as
-- having a @t@ type in the schema for /@settings@/ (see t'GI.GLib.Structs.VariantType.VariantType').
-- 
-- /Since: 2.50/
settingsGetUint64 ::
    (B.CallStack.HasCallStack, MonadIO m, IsSettings a) =>
    a
    -- ^ /@settings@/: the settings object
    -> T.Text
    -- ^ /@key@/: the key to get the value for
    -> m Word64
    -- ^ __Returns:__ a 64-bit unsigned integer
settingsGetUint64 settings key = liftIO $ do
    settings' <- unsafeManagedPtrCastPtr settings
    key' <- textToCString key
    result <- g_settings_get_uint64 settings' key'
    touchManagedPtr settings
    freeMem key'
    return result

#if defined(ENABLE_OVERLOADING)
data SettingsGetUint64MethodInfo
instance (signature ~ (T.Text -> m Word64), MonadIO m, IsSettings a) => O.OverloadedMethod SettingsGetUint64MethodInfo a signature where
    overloadedMethod = settingsGetUint64

instance O.OverloadedMethodInfo SettingsGetUint64MethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.settingsGetUint64",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#v:settingsGetUint64"
        })


#endif

-- method Settings::get_user_value
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "settings"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Settings" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the key to get the user value for"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just TVariant
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_get_user_value" g_settings_get_user_value :: 
    Ptr Settings ->                         -- settings : TInterface (Name {namespace = "Gio", name = "Settings"})
    CString ->                              -- key : TBasicType TUTF8
    IO (Ptr GVariant)

-- | Checks the ‘user value’ of a key, if there is one.
-- 
-- The user value of a key is the last value that was set by the user.
-- 
-- After calling 'GI.Gio.Objects.Settings.settingsReset' this function should always return
-- @NULL@ (assuming something is not wrong with the system
-- configuration).
-- 
-- It is possible that 'GI.Gio.Objects.Settings.settingsGetValue' will return a different
-- value than this function.  This can happen in the case that the user
-- set a value for a key that was subsequently locked down by the system
-- administrator — this function will return the user’s old value.
-- 
-- This function may be useful for adding a ‘reset’ option to a UI or
-- for providing indication that a particular value has been changed.
-- 
-- It is a programmer error to give a /@key@/ that isn’t contained in the
-- schema for /@settings@/.
-- 
-- /Since: 2.40/
settingsGetUserValue ::
    (B.CallStack.HasCallStack, MonadIO m, IsSettings a) =>
    a
    -- ^ /@settings@/: the settings object
    -> T.Text
    -- ^ /@key@/: the key to get the user value for
    -> m (Maybe GVariant)
    -- ^ __Returns:__ the user’s value, if set
settingsGetUserValue settings key = liftIO $ do
    settings' <- unsafeManagedPtrCastPtr settings
    key' <- textToCString key
    result <- g_settings_get_user_value settings' key'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- B.GVariant.wrapGVariantPtr result'
        return result''
    touchManagedPtr settings
    freeMem key'
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data SettingsGetUserValueMethodInfo
instance (signature ~ (T.Text -> m (Maybe GVariant)), MonadIO m, IsSettings a) => O.OverloadedMethod SettingsGetUserValueMethodInfo a signature where
    overloadedMethod = settingsGetUserValue

instance O.OverloadedMethodInfo SettingsGetUserValueMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.settingsGetUserValue",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#v:settingsGetUserValue"
        })


#endif

-- method Settings::get_value
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "settings"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Settings" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the key to get the value for"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just TVariant
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_get_value" g_settings_get_value :: 
    Ptr Settings ->                         -- settings : TInterface (Name {namespace = "Gio", name = "Settings"})
    CString ->                              -- key : TBasicType TUTF8
    IO (Ptr GVariant)

-- | Gets the value that is stored in /@settings@/ for /@key@/.
-- 
-- It is a programmer error to give a /@key@/ that isn’t contained in the
-- schema for /@settings@/.
-- 
-- /Since: 2.26/
settingsGetValue ::
    (B.CallStack.HasCallStack, MonadIO m, IsSettings a) =>
    a
    -- ^ /@settings@/: the settings object
    -> T.Text
    -- ^ /@key@/: the key to get the value for
    -> m GVariant
    -- ^ __Returns:__ a new t'GVariant'
settingsGetValue settings key = liftIO $ do
    settings' <- unsafeManagedPtrCastPtr settings
    key' <- textToCString key
    result <- g_settings_get_value settings' key'
    checkUnexpectedReturnNULL "settingsGetValue" result
    result' <- B.GVariant.wrapGVariantPtr result
    touchManagedPtr settings
    freeMem key'
    return result'

#if defined(ENABLE_OVERLOADING)
data SettingsGetValueMethodInfo
instance (signature ~ (T.Text -> m GVariant), MonadIO m, IsSettings a) => O.OverloadedMethod SettingsGetValueMethodInfo a signature where
    overloadedMethod = settingsGetValue

instance O.OverloadedMethodInfo SettingsGetValueMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.settingsGetValue",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#v:settingsGetValue"
        })


#endif

-- method Settings::is_writable
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "settings"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Settings" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the name of a key" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_is_writable" g_settings_is_writable :: 
    Ptr Settings ->                         -- settings : TInterface (Name {namespace = "Gio", name = "Settings"})
    CString ->                              -- name : TBasicType TUTF8
    IO CInt

-- | Finds out if a key can be written.
-- 
-- /Since: 2.26/
settingsIsWritable ::
    (B.CallStack.HasCallStack, MonadIO m, IsSettings a) =>
    a
    -- ^ /@settings@/: the settings object
    -> T.Text
    -- ^ /@name@/: the name of a key
    -> m Bool
    -- ^ __Returns:__ true if the key /@name@/ is writable, false otherwise
settingsIsWritable settings name = liftIO $ do
    settings' <- unsafeManagedPtrCastPtr settings
    name' <- textToCString name
    result <- g_settings_is_writable settings' name'
    let result' = (/= 0) result
    touchManagedPtr settings
    freeMem name'
    return result'

#if defined(ENABLE_OVERLOADING)
data SettingsIsWritableMethodInfo
instance (signature ~ (T.Text -> m Bool), MonadIO m, IsSettings a) => O.OverloadedMethod SettingsIsWritableMethodInfo a signature where
    overloadedMethod = settingsIsWritable

instance O.OverloadedMethodInfo SettingsIsWritableMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.settingsIsWritable",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#v:settingsIsWritable"
        })


#endif

-- method Settings::list_children
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "settings"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Settings" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TCArray True (-1) (-1) (TBasicType TUTF8))
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_list_children" g_settings_list_children :: 
    Ptr Settings ->                         -- settings : TInterface (Name {namespace = "Gio", name = "Settings"})
    IO (Ptr CString)

-- | Gets the list of children on /@settings@/.
-- 
-- The list is exactly the list of strings for which it is not an error
-- to call 'GI.Gio.Objects.Settings.settingsGetChild'.
-- 
-- There is little reason to call this function from ‘normal’ code, since
-- you should already know what children are in your schema. This function
-- may still be useful there for introspection reasons, however.
-- 
-- You should free the return value with 'GI.GLib.Functions.strfreev' when you are done
-- with it.
settingsListChildren ::
    (B.CallStack.HasCallStack, MonadIO m, IsSettings a) =>
    a
    -- ^ /@settings@/: the settings object
    -> m [T.Text]
    -- ^ __Returns:__ a list of the children
    --   on /@settings@/, in no defined order
settingsListChildren settings = liftIO $ do
    settings' <- unsafeManagedPtrCastPtr settings
    result <- g_settings_list_children settings'
    checkUnexpectedReturnNULL "settingsListChildren" result
    result' <- unpackZeroTerminatedUTF8CArray result
    mapZeroTerminatedCArray freeMem result
    freeMem result
    touchManagedPtr settings
    return result'

#if defined(ENABLE_OVERLOADING)
data SettingsListChildrenMethodInfo
instance (signature ~ (m [T.Text]), MonadIO m, IsSettings a) => O.OverloadedMethod SettingsListChildrenMethodInfo a signature where
    overloadedMethod = settingsListChildren

instance O.OverloadedMethodInfo SettingsListChildrenMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.settingsListChildren",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#v:settingsListChildren"
        })


#endif

-- method Settings::list_keys
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "settings"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Settings" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TCArray True (-1) (-1) (TBasicType TUTF8))
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_list_keys" g_settings_list_keys :: 
    Ptr Settings ->                         -- settings : TInterface (Name {namespace = "Gio", name = "Settings"})
    IO (Ptr CString)

{-# DEPRECATED settingsListKeys ["(Since version 2.46)","Use 'GI.Gio.Structs.SettingsSchema.settingsSchemaListKeys' instead."] #-}
-- | Introspects the list of keys on /@settings@/.
-- 
-- You should probably not be calling this function from ‘normal’ code
-- (since you should already know what keys are in your schema).  This
-- function is intended for introspection reasons.
-- 
-- You should free the return value with 'GI.GLib.Functions.strfreev' when you are done
-- with it.
settingsListKeys ::
    (B.CallStack.HasCallStack, MonadIO m, IsSettings a) =>
    a
    -- ^ /@settings@/: the settings object
    -> m [T.Text]
    -- ^ __Returns:__ a list
    --   of the keys on /@settings@/, in no defined order
settingsListKeys settings = liftIO $ do
    settings' <- unsafeManagedPtrCastPtr settings
    result <- g_settings_list_keys settings'
    checkUnexpectedReturnNULL "settingsListKeys" result
    result' <- unpackZeroTerminatedUTF8CArray result
    mapZeroTerminatedCArray freeMem result
    freeMem result
    touchManagedPtr settings
    return result'

#if defined(ENABLE_OVERLOADING)
data SettingsListKeysMethodInfo
instance (signature ~ (m [T.Text]), MonadIO m, IsSettings a) => O.OverloadedMethod SettingsListKeysMethodInfo a signature where
    overloadedMethod = settingsListKeys

instance O.OverloadedMethodInfo SettingsListKeysMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.settingsListKeys",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#v:settingsListKeys"
        })


#endif

-- method Settings::range_check
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "settings"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Settings" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the key to check" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "value"
--           , argType = TVariant
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the value to check" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_range_check" g_settings_range_check :: 
    Ptr Settings ->                         -- settings : TInterface (Name {namespace = "Gio", name = "Settings"})
    CString ->                              -- key : TBasicType TUTF8
    Ptr GVariant ->                         -- value : TVariant
    IO CInt

{-# DEPRECATED settingsRangeCheck ["(Since version 2.40)","Use 'GI.Gio.Structs.SettingsSchemaKey.settingsSchemaKeyRangeCheck' instead."] #-}
-- | Checks if the given /@value@/ is of the correct type and within the
-- permitted range for /@key@/.
-- 
-- /Since: 2.28/
settingsRangeCheck ::
    (B.CallStack.HasCallStack, MonadIO m, IsSettings a) =>
    a
    -- ^ /@settings@/: the settings object
    -> T.Text
    -- ^ /@key@/: the key to check
    -> GVariant
    -- ^ /@value@/: the value to check
    -> m Bool
    -- ^ __Returns:__ true if /@value@/ is valid for /@key@/, false otherwise
settingsRangeCheck settings key value = liftIO $ do
    settings' <- unsafeManagedPtrCastPtr settings
    key' <- textToCString key
    value' <- unsafeManagedPtrGetPtr value
    result <- g_settings_range_check settings' key' value'
    let result' = (/= 0) result
    touchManagedPtr settings
    touchManagedPtr value
    freeMem key'
    return result'

#if defined(ENABLE_OVERLOADING)
data SettingsRangeCheckMethodInfo
instance (signature ~ (T.Text -> GVariant -> m Bool), MonadIO m, IsSettings a) => O.OverloadedMethod SettingsRangeCheckMethodInfo a signature where
    overloadedMethod = settingsRangeCheck

instance O.OverloadedMethodInfo SettingsRangeCheckMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.settingsRangeCheck",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#v:settingsRangeCheck"
        })


#endif

-- method Settings::reset
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "settings"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Settings" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the name of a key" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_reset" g_settings_reset :: 
    Ptr Settings ->                         -- settings : TInterface (Name {namespace = "Gio", name = "Settings"})
    CString ->                              -- key : TBasicType TUTF8
    IO ()

-- | Resets /@key@/ to its default value.
-- 
-- This call resets the key, as much as possible, to its default value.
-- That might be the value specified in the schema or the one set by the
-- administrator.
settingsReset ::
    (B.CallStack.HasCallStack, MonadIO m, IsSettings a) =>
    a
    -- ^ /@settings@/: the settings object
    -> T.Text
    -- ^ /@key@/: the name of a key
    -> m ()
settingsReset settings key = liftIO $ do
    settings' <- unsafeManagedPtrCastPtr settings
    key' <- textToCString key
    g_settings_reset settings' key'
    touchManagedPtr settings
    freeMem key'
    return ()

#if defined(ENABLE_OVERLOADING)
data SettingsResetMethodInfo
instance (signature ~ (T.Text -> m ()), MonadIO m, IsSettings a) => O.OverloadedMethod SettingsResetMethodInfo a signature where
    overloadedMethod = settingsReset

instance O.OverloadedMethodInfo SettingsResetMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.settingsReset",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#v:settingsReset"
        })


#endif

-- method Settings::revert
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "settings"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Settings" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_revert" g_settings_revert :: 
    Ptr Settings ->                         -- settings : TInterface (Name {namespace = "Gio", name = "Settings"})
    IO ()

-- | Reverts all unapplied changes to the settings.
-- 
-- This function does nothing unless /@settings@/ is in
-- <http://developer.gnome.org/gio/stable/class.Settings.html#delay-apply-mode ‘delay-apply’ mode>.  In the normal
-- case settings are always applied immediately.
-- 
-- Change notifications will be emitted for affected keys.
settingsRevert ::
    (B.CallStack.HasCallStack, MonadIO m, IsSettings a) =>
    a
    -- ^ /@settings@/: the settings object
    -> m ()
settingsRevert settings = liftIO $ do
    settings' <- unsafeManagedPtrCastPtr settings
    g_settings_revert settings'
    touchManagedPtr settings
    return ()

#if defined(ENABLE_OVERLOADING)
data SettingsRevertMethodInfo
instance (signature ~ (m ()), MonadIO m, IsSettings a) => O.OverloadedMethod SettingsRevertMethodInfo a signature where
    overloadedMethod = settingsRevert

instance O.OverloadedMethodInfo SettingsRevertMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.settingsRevert",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#v:settingsRevert"
        })


#endif

-- method Settings::set_boolean
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "settings"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Settings" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the key to set the value for"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "value"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the value to set it to"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_set_boolean" g_settings_set_boolean :: 
    Ptr Settings ->                         -- settings : TInterface (Name {namespace = "Gio", name = "Settings"})
    CString ->                              -- key : TBasicType TUTF8
    CInt ->                                 -- value : TBasicType TBoolean
    IO CInt

-- | Sets /@key@/ in /@settings@/ to /@value@/.
-- 
-- A convenience variant of t'GI.Gio.Objects.Settings.Settings'.@/set/@() for booleans.
-- 
-- It is a programmer error to give a /@key@/ that isn’t specified as
-- having a @b@ type in the schema for /@settings@/ (see t'GI.GLib.Structs.VariantType.VariantType').
-- 
-- /Since: 2.26/
settingsSetBoolean ::
    (B.CallStack.HasCallStack, MonadIO m, IsSettings a) =>
    a
    -- ^ /@settings@/: the settings object
    -> T.Text
    -- ^ /@key@/: the key to set the value for
    -> Bool
    -- ^ /@value@/: the value to set it to
    -> m Bool
    -- ^ __Returns:__ true if setting the key succeeded,
    --   false if the key was not writable
settingsSetBoolean settings key value = liftIO $ do
    settings' <- unsafeManagedPtrCastPtr settings
    key' <- textToCString key
    let value' = (P.fromIntegral . P.fromEnum) value
    result <- g_settings_set_boolean settings' key' value'
    let result' = (/= 0) result
    touchManagedPtr settings
    freeMem key'
    return result'

#if defined(ENABLE_OVERLOADING)
data SettingsSetBooleanMethodInfo
instance (signature ~ (T.Text -> Bool -> m Bool), MonadIO m, IsSettings a) => O.OverloadedMethod SettingsSetBooleanMethodInfo a signature where
    overloadedMethod = settingsSetBoolean

instance O.OverloadedMethodInfo SettingsSetBooleanMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.settingsSetBoolean",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#v:settingsSetBoolean"
        })


#endif

-- method Settings::set_double
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "settings"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Settings" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the key to set the value for"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "value"
--           , argType = TBasicType TDouble
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the value to set it to"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_set_double" g_settings_set_double :: 
    Ptr Settings ->                         -- settings : TInterface (Name {namespace = "Gio", name = "Settings"})
    CString ->                              -- key : TBasicType TUTF8
    CDouble ->                              -- value : TBasicType TDouble
    IO CInt

-- | Sets /@key@/ in /@settings@/ to /@value@/.
-- 
-- A convenience variant of t'GI.Gio.Objects.Settings.Settings'.@/set/@() for doubles.
-- 
-- It is a programmer error to give a /@key@/ that isn’t specified as
-- having a @d@ type in the schema for /@settings@/ (see t'GI.GLib.Structs.VariantType.VariantType').
-- 
-- /Since: 2.26/
settingsSetDouble ::
    (B.CallStack.HasCallStack, MonadIO m, IsSettings a) =>
    a
    -- ^ /@settings@/: the settings object
    -> T.Text
    -- ^ /@key@/: the key to set the value for
    -> Double
    -- ^ /@value@/: the value to set it to
    -> m Bool
    -- ^ __Returns:__ true if setting the key succeeded,
    --   false if the key was not writable
settingsSetDouble settings key value = liftIO $ do
    settings' <- unsafeManagedPtrCastPtr settings
    key' <- textToCString key
    let value' = realToFrac value
    result <- g_settings_set_double settings' key' value'
    let result' = (/= 0) result
    touchManagedPtr settings
    freeMem key'
    return result'

#if defined(ENABLE_OVERLOADING)
data SettingsSetDoubleMethodInfo
instance (signature ~ (T.Text -> Double -> m Bool), MonadIO m, IsSettings a) => O.OverloadedMethod SettingsSetDoubleMethodInfo a signature where
    overloadedMethod = settingsSetDouble

instance O.OverloadedMethodInfo SettingsSetDoubleMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.settingsSetDouble",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#v:settingsSetDouble"
        })


#endif

-- method Settings::set_enum
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "settings"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Settings" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the key to set the value for"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "value"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an enumerated value"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_set_enum" g_settings_set_enum :: 
    Ptr Settings ->                         -- settings : TInterface (Name {namespace = "Gio", name = "Settings"})
    CString ->                              -- key : TBasicType TUTF8
    Int32 ->                                -- value : TBasicType TInt
    IO CInt

-- | Looks up the enumerated type nick for /@value@/ and writes it to /@key@/,
-- within /@settings@/.
-- 
-- It is a programmer error to give a /@key@/ that isn’t contained in the
-- schema for /@settings@/ or is not marked as an enumerated type, or for
-- /@value@/ not to be a valid value for the named type.
-- 
-- After performing the write, accessing /@key@/ directly with
-- 'GI.Gio.Objects.Settings.settingsGetString' will return the ‘nick’ associated with
-- /@value@/.
settingsSetEnum ::
    (B.CallStack.HasCallStack, MonadIO m, IsSettings a) =>
    a
    -- ^ /@settings@/: the settings object
    -> T.Text
    -- ^ /@key@/: the key to set the value for
    -> Int32
    -- ^ /@value@/: an enumerated value
    -> m Bool
    -- ^ __Returns:__ true if the set succeeds, false otherwise
settingsSetEnum settings key value = liftIO $ do
    settings' <- unsafeManagedPtrCastPtr settings
    key' <- textToCString key
    result <- g_settings_set_enum settings' key' value
    let result' = (/= 0) result
    touchManagedPtr settings
    freeMem key'
    return result'

#if defined(ENABLE_OVERLOADING)
data SettingsSetEnumMethodInfo
instance (signature ~ (T.Text -> Int32 -> m Bool), MonadIO m, IsSettings a) => O.OverloadedMethod SettingsSetEnumMethodInfo a signature where
    overloadedMethod = settingsSetEnum

instance O.OverloadedMethodInfo SettingsSetEnumMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.settingsSetEnum",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#v:settingsSetEnum"
        })


#endif

-- method Settings::set_flags
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "settings"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Settings" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the key to set the value for"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "value"
--           , argType = TBasicType TUInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a flags value" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_set_flags" g_settings_set_flags :: 
    Ptr Settings ->                         -- settings : TInterface (Name {namespace = "Gio", name = "Settings"})
    CString ->                              -- key : TBasicType TUTF8
    Word32 ->                               -- value : TBasicType TUInt
    IO CInt

-- | Looks up the flags type nicks for the bits specified by /@value@/, puts
-- them in an array of strings and writes the array to /@key@/, within
-- /@settings@/.
-- 
-- It is a programmer error to give a /@key@/ that isn’t contained in the
-- schema for /@settings@/ or is not marked as a flags type, or for /@value@/
-- to contain any bits that are not value for the named type.
-- 
-- After performing the write, accessing /@key@/ directly with
-- 'GI.Gio.Objects.Settings.settingsGetStrv' will return an array of ‘nicks’; one for each
-- bit in /@value@/.
settingsSetFlags ::
    (B.CallStack.HasCallStack, MonadIO m, IsSettings a) =>
    a
    -- ^ /@settings@/: the settings object
    -> T.Text
    -- ^ /@key@/: the key to set the value for
    -> Word32
    -- ^ /@value@/: a flags value
    -> m Bool
    -- ^ __Returns:__ true if the set succeeds, false otherwise
settingsSetFlags settings key value = liftIO $ do
    settings' <- unsafeManagedPtrCastPtr settings
    key' <- textToCString key
    result <- g_settings_set_flags settings' key' value
    let result' = (/= 0) result
    touchManagedPtr settings
    freeMem key'
    return result'

#if defined(ENABLE_OVERLOADING)
data SettingsSetFlagsMethodInfo
instance (signature ~ (T.Text -> Word32 -> m Bool), MonadIO m, IsSettings a) => O.OverloadedMethod SettingsSetFlagsMethodInfo a signature where
    overloadedMethod = settingsSetFlags

instance O.OverloadedMethodInfo SettingsSetFlagsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.settingsSetFlags",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#v:settingsSetFlags"
        })


#endif

-- method Settings::set_int
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "settings"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Settings" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the key to set the value for"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "value"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the value to set it to"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_set_int" g_settings_set_int :: 
    Ptr Settings ->                         -- settings : TInterface (Name {namespace = "Gio", name = "Settings"})
    CString ->                              -- key : TBasicType TUTF8
    Int32 ->                                -- value : TBasicType TInt
    IO CInt

-- | Sets /@key@/ in /@settings@/ to /@value@/.
-- 
-- A convenience variant of t'GI.Gio.Objects.Settings.Settings'.@/set/@() for 32-bit integers.
-- 
-- It is a programmer error to give a /@key@/ that isn’t specified as
-- having an @i@ type in the schema for /@settings@/ (see t'GI.GLib.Structs.VariantType.VariantType').
-- 
-- /Since: 2.26/
settingsSetInt ::
    (B.CallStack.HasCallStack, MonadIO m, IsSettings a) =>
    a
    -- ^ /@settings@/: the settings object
    -> T.Text
    -- ^ /@key@/: the key to set the value for
    -> Int32
    -- ^ /@value@/: the value to set it to
    -> m Bool
    -- ^ __Returns:__ true if setting the key succeeded,
    --   false if the key was not writable
settingsSetInt settings key value = liftIO $ do
    settings' <- unsafeManagedPtrCastPtr settings
    key' <- textToCString key
    result <- g_settings_set_int settings' key' value
    let result' = (/= 0) result
    touchManagedPtr settings
    freeMem key'
    return result'

#if defined(ENABLE_OVERLOADING)
data SettingsSetIntMethodInfo
instance (signature ~ (T.Text -> Int32 -> m Bool), MonadIO m, IsSettings a) => O.OverloadedMethod SettingsSetIntMethodInfo a signature where
    overloadedMethod = settingsSetInt

instance O.OverloadedMethodInfo SettingsSetIntMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.settingsSetInt",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#v:settingsSetInt"
        })


#endif

-- method Settings::set_int64
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "settings"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Settings" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the key to set the value for"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "value"
--           , argType = TBasicType TInt64
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the value to set it to"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_set_int64" g_settings_set_int64 :: 
    Ptr Settings ->                         -- settings : TInterface (Name {namespace = "Gio", name = "Settings"})
    CString ->                              -- key : TBasicType TUTF8
    Int64 ->                                -- value : TBasicType TInt64
    IO CInt

-- | Sets /@key@/ in /@settings@/ to /@value@/.
-- 
-- A convenience variant of t'GI.Gio.Objects.Settings.Settings'.@/set/@() for 64-bit integers.
-- 
-- It is a programmer error to give a /@key@/ that isn’t specified as
-- having an @x@ type in the schema for /@settings@/ (see t'GI.GLib.Structs.VariantType.VariantType').
-- 
-- /Since: 2.50/
settingsSetInt64 ::
    (B.CallStack.HasCallStack, MonadIO m, IsSettings a) =>
    a
    -- ^ /@settings@/: the settings object
    -> T.Text
    -- ^ /@key@/: the key to set the value for
    -> Int64
    -- ^ /@value@/: the value to set it to
    -> m Bool
    -- ^ __Returns:__ true if setting the key succeeded,
    --   false if the key was not writable
settingsSetInt64 settings key value = liftIO $ do
    settings' <- unsafeManagedPtrCastPtr settings
    key' <- textToCString key
    result <- g_settings_set_int64 settings' key' value
    let result' = (/= 0) result
    touchManagedPtr settings
    freeMem key'
    return result'

#if defined(ENABLE_OVERLOADING)
data SettingsSetInt64MethodInfo
instance (signature ~ (T.Text -> Int64 -> m Bool), MonadIO m, IsSettings a) => O.OverloadedMethod SettingsSetInt64MethodInfo a signature where
    overloadedMethod = settingsSetInt64

instance O.OverloadedMethodInfo SettingsSetInt64MethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.settingsSetInt64",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#v:settingsSetInt64"
        })


#endif

-- method Settings::set_string
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "settings"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Settings" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the key to set the value for"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "value"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the value to set it to"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_set_string" g_settings_set_string :: 
    Ptr Settings ->                         -- settings : TInterface (Name {namespace = "Gio", name = "Settings"})
    CString ->                              -- key : TBasicType TUTF8
    CString ->                              -- value : TBasicType TUTF8
    IO CInt

-- | Sets /@key@/ in /@settings@/ to /@value@/.
-- 
-- A convenience variant of t'GI.Gio.Objects.Settings.Settings'.@/set/@() for strings.
-- 
-- It is a programmer error to give a /@key@/ that isn’t specified as
-- having an @s@ type in the schema for /@settings@/ (see t'GI.GLib.Structs.VariantType.VariantType').
-- 
-- /Since: 2.26/
settingsSetString ::
    (B.CallStack.HasCallStack, MonadIO m, IsSettings a) =>
    a
    -- ^ /@settings@/: the settings object
    -> T.Text
    -- ^ /@key@/: the key to set the value for
    -> T.Text
    -- ^ /@value@/: the value to set it to
    -> m Bool
    -- ^ __Returns:__ true if setting the key succeeded,
    --   false if the key was not writable
settingsSetString settings key value = liftIO $ do
    settings' <- unsafeManagedPtrCastPtr settings
    key' <- textToCString key
    value' <- textToCString value
    result <- g_settings_set_string settings' key' value'
    let result' = (/= 0) result
    touchManagedPtr settings
    freeMem key'
    freeMem value'
    return result'

#if defined(ENABLE_OVERLOADING)
data SettingsSetStringMethodInfo
instance (signature ~ (T.Text -> T.Text -> m Bool), MonadIO m, IsSettings a) => O.OverloadedMethod SettingsSetStringMethodInfo a signature where
    overloadedMethod = settingsSetString

instance O.OverloadedMethodInfo SettingsSetStringMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.settingsSetString",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#v:settingsSetString"
        })


#endif

-- method Settings::set_strv
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "settings"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Settings" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the key to set the value for"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "value"
--           , argType = TCArray True (-1) (-1) (TBasicType TUTF8)
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the value to set it to"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_set_strv" g_settings_set_strv :: 
    Ptr Settings ->                         -- settings : TInterface (Name {namespace = "Gio", name = "Settings"})
    CString ->                              -- key : TBasicType TUTF8
    Ptr CString ->                          -- value : TCArray True (-1) (-1) (TBasicType TUTF8)
    IO CInt

-- | Sets /@key@/ in /@settings@/ to /@value@/.
-- 
-- A convenience variant of t'GI.Gio.Objects.Settings.Settings'.@/set/@() for string arrays.  If
-- /@value@/ is @NULL@, then /@key@/ is set to be the empty array.
-- 
-- It is a programmer error to give a /@key@/ that isn’t specified as
-- having an @as@ type in the schema for /@settings@/ (see t'GI.GLib.Structs.VariantType.VariantType').
-- 
-- /Since: 2.26/
settingsSetStrv ::
    (B.CallStack.HasCallStack, MonadIO m, IsSettings a) =>
    a
    -- ^ /@settings@/: the settings object
    -> T.Text
    -- ^ /@key@/: the key to set the value for
    -> Maybe ([T.Text])
    -- ^ /@value@/: the value to set it to
    -> m Bool
    -- ^ __Returns:__ true if setting the key succeeded,
    --   false if the key was not writable
settingsSetStrv settings key value = liftIO $ do
    settings' <- unsafeManagedPtrCastPtr settings
    key' <- textToCString key
    maybeValue <- case value of
        Nothing -> return FP.nullPtr
        Just jValue -> do
            jValue' <- packZeroTerminatedUTF8CArray jValue
            return jValue'
    result <- g_settings_set_strv settings' key' maybeValue
    let result' = (/= 0) result
    touchManagedPtr settings
    freeMem key'
    mapZeroTerminatedCArray freeMem maybeValue
    freeMem maybeValue
    return result'

#if defined(ENABLE_OVERLOADING)
data SettingsSetStrvMethodInfo
instance (signature ~ (T.Text -> Maybe ([T.Text]) -> m Bool), MonadIO m, IsSettings a) => O.OverloadedMethod SettingsSetStrvMethodInfo a signature where
    overloadedMethod = settingsSetStrv

instance O.OverloadedMethodInfo SettingsSetStrvMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.settingsSetStrv",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#v:settingsSetStrv"
        })


#endif

-- method Settings::set_uint
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "settings"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Settings" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the key to set the value for"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "value"
--           , argType = TBasicType TUInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the value to set it to"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_set_uint" g_settings_set_uint :: 
    Ptr Settings ->                         -- settings : TInterface (Name {namespace = "Gio", name = "Settings"})
    CString ->                              -- key : TBasicType TUTF8
    Word32 ->                               -- value : TBasicType TUInt
    IO CInt

-- | Sets /@key@/ in /@settings@/ to /@value@/.
-- 
-- A convenience variant of t'GI.Gio.Objects.Settings.Settings'.@/set/@() for 32-bit unsigned
-- integers.
-- 
-- It is a programmer error to give a /@key@/ that isn’t specified as
-- having a @u@ type in the schema for /@settings@/ (see t'GI.GLib.Structs.VariantType.VariantType').
-- 
-- /Since: 2.30/
settingsSetUint ::
    (B.CallStack.HasCallStack, MonadIO m, IsSettings a) =>
    a
    -- ^ /@settings@/: the settings object
    -> T.Text
    -- ^ /@key@/: the key to set the value for
    -> Word32
    -- ^ /@value@/: the value to set it to
    -> m Bool
    -- ^ __Returns:__ true if setting the key succeeded,
    --   false if the key was not writable
settingsSetUint settings key value = liftIO $ do
    settings' <- unsafeManagedPtrCastPtr settings
    key' <- textToCString key
    result <- g_settings_set_uint settings' key' value
    let result' = (/= 0) result
    touchManagedPtr settings
    freeMem key'
    return result'

#if defined(ENABLE_OVERLOADING)
data SettingsSetUintMethodInfo
instance (signature ~ (T.Text -> Word32 -> m Bool), MonadIO m, IsSettings a) => O.OverloadedMethod SettingsSetUintMethodInfo a signature where
    overloadedMethod = settingsSetUint

instance O.OverloadedMethodInfo SettingsSetUintMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.settingsSetUint",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#v:settingsSetUint"
        })


#endif

-- method Settings::set_uint64
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "settings"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Settings" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the key to set the value for"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "value"
--           , argType = TBasicType TUInt64
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the value to set it to"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_set_uint64" g_settings_set_uint64 :: 
    Ptr Settings ->                         -- settings : TInterface (Name {namespace = "Gio", name = "Settings"})
    CString ->                              -- key : TBasicType TUTF8
    Word64 ->                               -- value : TBasicType TUInt64
    IO CInt

-- | Sets /@key@/ in /@settings@/ to /@value@/.
-- 
-- A convenience variant of t'GI.Gio.Objects.Settings.Settings'.@/set/@() for 64-bit unsigned
-- integers.
-- 
-- It is a programmer error to give a /@key@/ that isn’t specified as
-- having a @t@ type in the schema for /@settings@/ (see t'GI.GLib.Structs.VariantType.VariantType').
-- 
-- /Since: 2.50/
settingsSetUint64 ::
    (B.CallStack.HasCallStack, MonadIO m, IsSettings a) =>
    a
    -- ^ /@settings@/: the settings object
    -> T.Text
    -- ^ /@key@/: the key to set the value for
    -> Word64
    -- ^ /@value@/: the value to set it to
    -> m Bool
    -- ^ __Returns:__ true if setting the key succeeded,
    --   false if the key was not writable
settingsSetUint64 settings key value = liftIO $ do
    settings' <- unsafeManagedPtrCastPtr settings
    key' <- textToCString key
    result <- g_settings_set_uint64 settings' key' value
    let result' = (/= 0) result
    touchManagedPtr settings
    freeMem key'
    return result'

#if defined(ENABLE_OVERLOADING)
data SettingsSetUint64MethodInfo
instance (signature ~ (T.Text -> Word64 -> m Bool), MonadIO m, IsSettings a) => O.OverloadedMethod SettingsSetUint64MethodInfo a signature where
    overloadedMethod = settingsSetUint64

instance O.OverloadedMethodInfo SettingsSetUint64MethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.settingsSetUint64",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#v:settingsSetUint64"
        })


#endif

-- method Settings::set_value
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "settings"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Settings" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the settings object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the key to set the value for"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "value"
--           , argType = TVariant
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a [struct@GLib.Variant] of the correct type"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_set_value" g_settings_set_value :: 
    Ptr Settings ->                         -- settings : TInterface (Name {namespace = "Gio", name = "Settings"})
    CString ->                              -- key : TBasicType TUTF8
    Ptr GVariant ->                         -- value : TVariant
    IO CInt

-- | Sets /@key@/ in /@settings@/ to /@value@/.
-- 
-- It is a programmer error to give a /@key@/ that isn’t contained in the
-- schema for /@settings@/ or for /@value@/ to have the incorrect type, per
-- the schema.
-- 
-- If /@value@/ is floating then this function consumes the reference.
-- 
-- /Since: 2.26/
settingsSetValue ::
    (B.CallStack.HasCallStack, MonadIO m, IsSettings a) =>
    a
    -- ^ /@settings@/: the settings object
    -> T.Text
    -- ^ /@key@/: the key to set the value for
    -> GVariant
    -- ^ /@value@/: a t'GVariant' of the correct type
    -> m Bool
    -- ^ __Returns:__ true if setting the key succeeded,
    --   false if the key was not writable
settingsSetValue settings key value = liftIO $ do
    settings' <- unsafeManagedPtrCastPtr settings
    key' <- textToCString key
    value' <- unsafeManagedPtrGetPtr value
    result <- g_settings_set_value settings' key' value'
    let result' = (/= 0) result
    touchManagedPtr settings
    touchManagedPtr value
    freeMem key'
    return result'

#if defined(ENABLE_OVERLOADING)
data SettingsSetValueMethodInfo
instance (signature ~ (T.Text -> GVariant -> m Bool), MonadIO m, IsSettings a) => O.OverloadedMethod SettingsSetValueMethodInfo a signature where
    overloadedMethod = settingsSetValue

instance O.OverloadedMethodInfo SettingsSetValueMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Settings.settingsSetValue",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Settings.html#v:settingsSetValue"
        })


#endif

-- method Settings::list_relocatable_schemas
-- method type : MemberFunction
-- Args: []
-- Lengths: []
-- returnType: Just (TCArray True (-1) (-1) (TBasicType TUTF8))
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_list_relocatable_schemas" g_settings_list_relocatable_schemas :: 
    IO (Ptr CString)

{-# DEPRECATED settingsListRelocatableSchemas ["(Since version 2.40)","Use 'GI.Gio.Structs.SettingsSchemaSource.settingsSchemaSourceListSchemas' instead"] #-}
-- | Deprecated.
-- 
-- /Since: 2.28/
settingsListRelocatableSchemas ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m [T.Text]
    -- ^ __Returns:__ a list of
    --   relocatable t'GI.Gio.Objects.Settings.Settings' schemas that are available, in no defined order.
    --   The list must not be modified or freed.
settingsListRelocatableSchemas  = liftIO $ do
    result <- g_settings_list_relocatable_schemas
    checkUnexpectedReturnNULL "settingsListRelocatableSchemas" result
    result' <- unpackZeroTerminatedUTF8CArray result
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method Settings::list_schemas
-- method type : MemberFunction
-- Args: []
-- Lengths: []
-- returnType: Just (TCArray True (-1) (-1) (TBasicType TUTF8))
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_list_schemas" g_settings_list_schemas :: 
    IO (Ptr CString)

{-# DEPRECATED settingsListSchemas ["(Since version 2.40)","Use 'GI.Gio.Structs.SettingsSchemaSource.settingsSchemaSourceListSchemas' instead.","If you used 'GI.Gio.Objects.Settings.settingsListSchemas' to check for the presence of","a particular schema, use 'GI.Gio.Structs.SettingsSchemaSource.settingsSchemaSourceLookup' instead","of your whole loop."] #-}
-- | Deprecated.
-- 
-- /Since: 2.26/
settingsListSchemas ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m [T.Text]
    -- ^ __Returns:__ a list of
    --   t'GI.Gio.Objects.Settings.Settings' schemas that are available, in no defined order.  The list
    --   must not be modified or freed.
settingsListSchemas  = liftIO $ do
    result <- g_settings_list_schemas
    checkUnexpectedReturnNULL "settingsListSchemas" result
    result' <- unpackZeroTerminatedUTF8CArray result
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method Settings::sync
-- method type : MemberFunction
-- Args: []
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_sync" g_settings_sync :: 
    IO ()

-- | Ensures that all pending operations are complete for the default backend.
-- 
-- Writes made to a t'GI.Gio.Objects.Settings.Settings' are handled asynchronously.  For this
-- reason, it is very unlikely that the changes have it to disk by the
-- time t'GI.Gio.Objects.Settings.Settings'.@/set/@() returns.
-- 
-- This call will block until all of the writes have made it to the
-- backend.  Since the main loop is not running, no change notifications
-- will be dispatched during this call (but some may be queued by the
-- time the call is done).
settingsSync ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m ()
settingsSync  = liftIO $ do
    g_settings_sync
    return ()

#if defined(ENABLE_OVERLOADING)
#endif

-- method Settings::unbind
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "object"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "Object" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the object with property to unbind"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "property"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the property whose binding is removed"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_settings_unbind" g_settings_unbind :: 
    Ptr GObject.Object.Object ->            -- object : TInterface (Name {namespace = "GObject", name = "Object"})
    CString ->                              -- property : TBasicType TUTF8
    IO ()

-- | Removes an existing binding for /@property@/ on /@object@/.
-- 
-- Note that bindings are automatically removed when the
-- object is finalized, so it is rarely necessary to call this
-- function.
-- 
-- /Since: 2.26/
settingsUnbind ::
    (B.CallStack.HasCallStack, MonadIO m, GObject.Object.IsObject a) =>
    a
    -- ^ /@object@/: the object with property to unbind
    -> T.Text
    -- ^ /@property@/: the property whose binding is removed
    -> m ()
settingsUnbind object property = liftIO $ do
    object' <- unsafeManagedPtrCastPtr object
    property' <- textToCString property
    g_settings_unbind object' property'
    touchManagedPtr object
    freeMem property'
    return ()

#if defined(ENABLE_OVERLOADING)
#endif


