{-# LANGUAGE ImplicitParams, RankNTypes, TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- @GDBusProxy@ is a base class used for proxies to access a D-Bus
-- interface on a remote object. A @GDBusProxy@ can be constructed for
-- both well-known and unique names.
-- 
-- By default, @GDBusProxy@ will cache all properties (and listen to
-- changes) of the remote object, and proxy all signals that get
-- emitted. This behaviour can be changed by passing suitable
-- [flags/@gio@/.DBusProxyFlags] when the proxy is created. If the proxy is for a
-- well-known name, the property cache is flushed when the name owner
-- vanishes and reloaded when a name owner appears.
-- 
-- The unique name owner of the proxy’s name is tracked and can be read from
-- [DBusProxy:gNameOwner]("GI.Gio.Objects.DBusProxy#g:attr:gNameOwner"). Connect to the
-- [Object::notify]("GI.GObject.Objects.Object#g:signal:notify") signal to get notified of changes.
-- Additionally, only signals and property changes emitted from the current name
-- owner are considered and calls are always sent to the current name owner.
-- This avoids a number of race conditions when the name is lost by one owner
-- and claimed by another. However, if no name owner currently exists,
-- then calls will be sent to the well-known name which may result in
-- the message bus launching an owner (unless
-- @G_DBUS_PROXY_FLAGS_DO_NOT_AUTO_START@ is set).
-- 
-- If the proxy is for a stateless D-Bus service, where the name owner may
-- be started and stopped between calls, the
-- [DBusProxy:gNameOwner]("GI.Gio.Objects.DBusProxy#g:attr:gNameOwner") tracking of @GDBusProxy@ will cause the
-- proxy to drop signal and property changes from the service after it has
-- restarted for the first time. When interacting with a stateless D-Bus
-- service, do not use @GDBusProxy@ — use direct D-Bus method calls and signal
-- connections.
-- 
-- The generic [DBusProxy::gPropertiesChanged]("GI.Gio.Objects.DBusProxy#g:signal:gPropertiesChanged") and
-- [DBusProxy::gSignal]("GI.Gio.Objects.DBusProxy#g:signal:gSignal") signals are not very convenient to work
-- with. Therefore, the recommended way of working with proxies is to subclass
-- @GDBusProxy@, and have more natural properties and signals in your derived
-- class. This <http://developer.gnome.org/gio/stable/migrating-gdbus.html#using-gdbus-codegen example> shows how
-- this can easily be done using the <http://developer.gnome.org/gio/stable/gdbus-codegen.html `gdbus-codegen`> tool.
-- 
-- A @GDBusProxy@ instance can be used from multiple threads but note
-- that all signals (e.g. [DBusProxy::gSignal]("GI.Gio.Objects.DBusProxy#g:signal:gSignal"),
-- [DBusProxy::gPropertiesChanged]("GI.Gio.Objects.DBusProxy#g:signal:gPropertiesChanged") and
-- [Object::notify]("GI.GObject.Objects.Object#g:signal:notify")) are emitted in the thread-default main
-- context (see 'GI.GLib.Structs.MainContext.mainContextPushThreadDefault') of the thread
-- where the instance was constructed.
-- 
-- 
-- == A watch proxy example
-- An example using a proxy for a well-known name can be found in
-- <https://gitlab.gnome.org/GNOME/glib/-/blob/HEAD/gio/tests/gdbus-example-watch-proxy.c `gdbus-example-watch-proxy.c`>.
-- 
-- /Since: 2.26/

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gio.Objects.DBusProxy
    ( 

-- * Exported types
    DBusProxy(..)                           ,
    IsDBusProxy                             ,
    toDBusProxy                             ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [call]("GI.Gio.Objects.DBusProxy#g:method:call"), [callFinish]("GI.Gio.Objects.DBusProxy#g:method:callFinish"), [callSync]("GI.Gio.Objects.DBusProxy#g:method:callSync"), [callWithUnixFdList]("GI.Gio.Objects.DBusProxy#g:method:callWithUnixFdList"), [callWithUnixFdListFinish]("GI.Gio.Objects.DBusProxy#g:method:callWithUnixFdListFinish"), [callWithUnixFdListSync]("GI.Gio.Objects.DBusProxy#g:method:callWithUnixFdListSync"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [init]("GI.Gio.Interfaces.Initable#g:method:init"), [initAsync]("GI.Gio.Interfaces.AsyncInitable#g:method:initAsync"), [initFinish]("GI.Gio.Interfaces.AsyncInitable#g:method:initFinish"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getCachedProperty]("GI.Gio.Objects.DBusProxy#g:method:getCachedProperty"), [getCachedPropertyNames]("GI.Gio.Objects.DBusProxy#g:method:getCachedPropertyNames"), [getConnection]("GI.Gio.Objects.DBusProxy#g:method:getConnection"), [getData]("GI.GObject.Objects.Object#g:method:getData"), [getDefaultTimeout]("GI.Gio.Objects.DBusProxy#g:method:getDefaultTimeout"), [getFlags]("GI.Gio.Objects.DBusProxy#g:method:getFlags"), [getInfo]("GI.Gio.Interfaces.DBusInterface#g:method:getInfo"), [getInterfaceInfo]("GI.Gio.Objects.DBusProxy#g:method:getInterfaceInfo"), [getInterfaceName]("GI.Gio.Objects.DBusProxy#g:method:getInterfaceName"), [getName]("GI.Gio.Objects.DBusProxy#g:method:getName"), [getNameOwner]("GI.Gio.Objects.DBusProxy#g:method:getNameOwner"), [getObject]("GI.Gio.Interfaces.DBusInterface#g:method:getObject"), [getObjectPath]("GI.Gio.Objects.DBusProxy#g:method:getObjectPath"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata").
-- 
-- ==== Setters
-- [setCachedProperty]("GI.Gio.Objects.DBusProxy#g:method:setCachedProperty"), [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setDefaultTimeout]("GI.Gio.Objects.DBusProxy#g:method:setDefaultTimeout"), [setInterfaceInfo]("GI.Gio.Objects.DBusProxy#g:method:setInterfaceInfo"), [setObject]("GI.Gio.Interfaces.DBusInterface#g:method:setObject"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty").

#if defined(ENABLE_OVERLOADING)
    ResolveDBusProxyMethod                  ,
#endif

-- ** call #method:call#

#if defined(ENABLE_OVERLOADING)
    DBusProxyCallMethodInfo                 ,
#endif
    dBusProxyCall                           ,


-- ** callFinish #method:callFinish#

#if defined(ENABLE_OVERLOADING)
    DBusProxyCallFinishMethodInfo           ,
#endif
    dBusProxyCallFinish                     ,


-- ** callSync #method:callSync#

#if defined(ENABLE_OVERLOADING)
    DBusProxyCallSyncMethodInfo             ,
#endif
    dBusProxyCallSync                       ,


-- ** callWithUnixFdList #method:callWithUnixFdList#

#if defined(ENABLE_OVERLOADING)
    DBusProxyCallWithUnixFdListMethodInfo   ,
#endif
    dBusProxyCallWithUnixFdList             ,


-- ** callWithUnixFdListFinish #method:callWithUnixFdListFinish#

#if defined(ENABLE_OVERLOADING)
    DBusProxyCallWithUnixFdListFinishMethodInfo,
#endif
    dBusProxyCallWithUnixFdListFinish       ,


-- ** callWithUnixFdListSync #method:callWithUnixFdListSync#

#if defined(ENABLE_OVERLOADING)
    DBusProxyCallWithUnixFdListSyncMethodInfo,
#endif
    dBusProxyCallWithUnixFdListSync         ,


-- ** getCachedProperty #method:getCachedProperty#

#if defined(ENABLE_OVERLOADING)
    DBusProxyGetCachedPropertyMethodInfo    ,
#endif
    dBusProxyGetCachedProperty              ,


-- ** getCachedPropertyNames #method:getCachedPropertyNames#

#if defined(ENABLE_OVERLOADING)
    DBusProxyGetCachedPropertyNamesMethodInfo,
#endif
    dBusProxyGetCachedPropertyNames         ,


-- ** getConnection #method:getConnection#

#if defined(ENABLE_OVERLOADING)
    DBusProxyGetConnectionMethodInfo        ,
#endif
    dBusProxyGetConnection                  ,


-- ** getDefaultTimeout #method:getDefaultTimeout#

#if defined(ENABLE_OVERLOADING)
    DBusProxyGetDefaultTimeoutMethodInfo    ,
#endif
    dBusProxyGetDefaultTimeout              ,


-- ** getFlags #method:getFlags#

#if defined(ENABLE_OVERLOADING)
    DBusProxyGetFlagsMethodInfo             ,
#endif
    dBusProxyGetFlags                       ,


-- ** getInterfaceInfo #method:getInterfaceInfo#

#if defined(ENABLE_OVERLOADING)
    DBusProxyGetInterfaceInfoMethodInfo     ,
#endif
    dBusProxyGetInterfaceInfo               ,


-- ** getInterfaceName #method:getInterfaceName#

#if defined(ENABLE_OVERLOADING)
    DBusProxyGetInterfaceNameMethodInfo     ,
#endif
    dBusProxyGetInterfaceName               ,


-- ** getName #method:getName#

#if defined(ENABLE_OVERLOADING)
    DBusProxyGetNameMethodInfo              ,
#endif
    dBusProxyGetName                        ,


-- ** getNameOwner #method:getNameOwner#

#if defined(ENABLE_OVERLOADING)
    DBusProxyGetNameOwnerMethodInfo         ,
#endif
    dBusProxyGetNameOwner                   ,


-- ** getObjectPath #method:getObjectPath#

#if defined(ENABLE_OVERLOADING)
    DBusProxyGetObjectPathMethodInfo        ,
#endif
    dBusProxyGetObjectPath                  ,


-- ** new #method:new#

    dBusProxyNew                            ,


-- ** newFinish #method:newFinish#

    dBusProxyNewFinish                      ,


-- ** newForBus #method:newForBus#

    dBusProxyNewForBus                      ,


-- ** newForBusFinish #method:newForBusFinish#

    dBusProxyNewForBusFinish                ,


-- ** newForBusSync #method:newForBusSync#

    dBusProxyNewForBusSync                  ,


-- ** newSync #method:newSync#

    dBusProxyNewSync                        ,


-- ** setCachedProperty #method:setCachedProperty#

#if defined(ENABLE_OVERLOADING)
    DBusProxySetCachedPropertyMethodInfo    ,
#endif
    dBusProxySetCachedProperty              ,


-- ** setDefaultTimeout #method:setDefaultTimeout#

#if defined(ENABLE_OVERLOADING)
    DBusProxySetDefaultTimeoutMethodInfo    ,
#endif
    dBusProxySetDefaultTimeout              ,


-- ** setInterfaceInfo #method:setInterfaceInfo#

#if defined(ENABLE_OVERLOADING)
    DBusProxySetInterfaceInfoMethodInfo     ,
#endif
    dBusProxySetInterfaceInfo               ,




 -- * Properties


-- ** gBusType #attr:gBusType#
-- | If this property is not 'GI.Gio.Enums.BusTypeNone', then
-- [DBusProxy:gConnection]("GI.Gio.Objects.DBusProxy#g:attr:gConnection") must be 'P.Nothing' and will be set to the
-- t'GI.Gio.Objects.DBusConnection.DBusConnection' obtained by calling 'GI.Gio.Functions.busGet' with the value
-- of this property.
-- 
-- /Since: 2.26/

#if defined(ENABLE_OVERLOADING)
    DBusProxyGBusTypePropertyInfo           ,
#endif
    constructDBusProxyGBusType              ,
#if defined(ENABLE_OVERLOADING)
    dBusProxyGBusType                       ,
#endif


-- ** gConnection #attr:gConnection#
-- | The t'GI.Gio.Objects.DBusConnection.DBusConnection' the proxy is for.
-- 
-- /Since: 2.26/

#if defined(ENABLE_OVERLOADING)
    DBusProxyGConnectionPropertyInfo        ,
#endif
    constructDBusProxyGConnection           ,
#if defined(ENABLE_OVERLOADING)
    dBusProxyGConnection                    ,
#endif
    getDBusProxyGConnection                 ,


-- ** gDefaultTimeout #attr:gDefaultTimeout#
-- | The timeout to use if -1 (specifying default timeout) is passed
-- as /@timeoutMsec@/ in the 'GI.Gio.Objects.DBusProxy.dBusProxyCall' and
-- 'GI.Gio.Objects.DBusProxy.dBusProxyCallSync' functions.
-- 
-- This allows applications to set a proxy-wide timeout for all
-- remote method invocations on the proxy. If this property is -1,
-- the default timeout (typically 25 seconds) is used. If set to
-- @/G_MAXINT/@, then no timeout is used.
-- 
-- /Since: 2.26/

#if defined(ENABLE_OVERLOADING)
    DBusProxyGDefaultTimeoutPropertyInfo    ,
#endif
    constructDBusProxyGDefaultTimeout       ,
#if defined(ENABLE_OVERLOADING)
    dBusProxyGDefaultTimeout                ,
#endif
    getDBusProxyGDefaultTimeout             ,
    setDBusProxyGDefaultTimeout             ,


-- ** gFlags #attr:gFlags#
-- | Flags from the t'GI.Gio.Flags.DBusProxyFlags' enumeration.
-- 
-- /Since: 2.26/

#if defined(ENABLE_OVERLOADING)
    DBusProxyGFlagsPropertyInfo             ,
#endif
    constructDBusProxyGFlags                ,
#if defined(ENABLE_OVERLOADING)
    dBusProxyGFlags                         ,
#endif
    getDBusProxyGFlags                      ,


-- ** gInterfaceInfo #attr:gInterfaceInfo#
-- | Ensure that interactions with this proxy conform to the given
-- interface. This is mainly to ensure that malformed data received
-- from the other peer is ignored. The given t'GI.Gio.Structs.DBusInterfaceInfo.DBusInterfaceInfo' is
-- said to be the \"expected interface\".
-- 
-- The checks performed are:
-- * When completing a method call, if the type signature of
--   the reply message isn\'t what\'s expected, the reply is
--   discarded and the t'GError' is set to 'GI.Gio.Enums.IOErrorEnumInvalidArgument'.
-- * Received signals that have a type signature mismatch are dropped and
--   a warning is logged via @/g_warning()/@.
-- * Properties received via the initial @GetAll()@ call or via the
--   @::PropertiesChanged@ signal (on the
--   <http://dbus.freedesktop.org/doc/dbus-specification.html#standard-interfaces-properties org.freedesktop.DBus.Properties>
--   interface) or set using 'GI.Gio.Objects.DBusProxy.dBusProxySetCachedProperty'
--   with a type signature mismatch are ignored and a warning is
--   logged via @/g_warning()/@.
-- 
-- 
-- Note that these checks are never done on methods, signals and
-- properties that are not referenced in the given
-- t'GI.Gio.Structs.DBusInterfaceInfo.DBusInterfaceInfo', since extending a D-Bus interface on the
-- service-side is not considered an ABI break.
-- 
-- /Since: 2.26/

#if defined(ENABLE_OVERLOADING)
    DBusProxyGInterfaceInfoPropertyInfo     ,
#endif
    clearDBusProxyGInterfaceInfo            ,
    constructDBusProxyGInterfaceInfo        ,
#if defined(ENABLE_OVERLOADING)
    dBusProxyGInterfaceInfo                 ,
#endif
    getDBusProxyGInterfaceInfo              ,
    setDBusProxyGInterfaceInfo              ,


-- ** gInterfaceName #attr:gInterfaceName#
-- | The D-Bus interface name the proxy is for.
-- 
-- /Since: 2.26/

#if defined(ENABLE_OVERLOADING)
    DBusProxyGInterfaceNamePropertyInfo     ,
#endif
    constructDBusProxyGInterfaceName        ,
#if defined(ENABLE_OVERLOADING)
    dBusProxyGInterfaceName                 ,
#endif
    getDBusProxyGInterfaceName              ,


-- ** gName #attr:gName#
-- | The well-known or unique name that the proxy is for.
-- 
-- /Since: 2.26/

#if defined(ENABLE_OVERLOADING)
    DBusProxyGNamePropertyInfo              ,
#endif
    constructDBusProxyGName                 ,
#if defined(ENABLE_OVERLOADING)
    dBusProxyGName                          ,
#endif
    getDBusProxyGName                       ,


-- ** gNameOwner #attr:gNameOwner#
-- | The unique name that owns [DBusProxy:gName]("GI.Gio.Objects.DBusProxy#g:attr:gName") or 'P.Nothing' if no-one
-- currently owns that name. You may connect to [Object::notify]("GI.GObject.Objects.Object#g:signal:notify") signal to
-- track changes to this property.
-- 
-- /Since: 2.26/

#if defined(ENABLE_OVERLOADING)
    DBusProxyGNameOwnerPropertyInfo         ,
#endif
#if defined(ENABLE_OVERLOADING)
    dBusProxyGNameOwner                     ,
#endif
    getDBusProxyGNameOwner                  ,


-- ** gObjectPath #attr:gObjectPath#
-- | The object path the proxy is for.
-- 
-- /Since: 2.26/

#if defined(ENABLE_OVERLOADING)
    DBusProxyGObjectPathPropertyInfo        ,
#endif
    constructDBusProxyGObjectPath           ,
#if defined(ENABLE_OVERLOADING)
    dBusProxyGObjectPath                    ,
#endif
    getDBusProxyGObjectPath                 ,




 -- * Signals


-- ** gPropertiesChanged #signal:gPropertiesChanged#

    DBusProxyGPropertiesChangedCallback     ,
#if defined(ENABLE_OVERLOADING)
    DBusProxyGPropertiesChangedSignalInfo   ,
#endif
    afterDBusProxyGPropertiesChanged        ,
    onDBusProxyGPropertiesChanged           ,


-- ** gSignal #signal:gSignal#

    DBusProxyGSignalCallback                ,
#if defined(ENABLE_OVERLOADING)
    DBusProxyGSignalSignalInfo              ,
#endif
    afterDBusProxyGSignal                   ,
    onDBusProxyGSignal                      ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.GLib.Callbacks as GLib.Callbacks
import qualified GI.GLib.Structs.Bytes as GLib.Bytes
import qualified GI.GLib.Structs.PollFD as GLib.PollFD
import qualified GI.GLib.Structs.Source as GLib.Source
import qualified GI.GLib.Structs.String as GLib.String
import qualified GI.GLib.Structs.VariantType as GLib.VariantType
import qualified GI.GObject.Callbacks as GObject.Callbacks
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.GObject.Structs.Parameter as GObject.Parameter
import qualified GI.Gio.Callbacks as Gio.Callbacks
import {-# SOURCE #-} qualified GI.Gio.Enums as Gio.Enums
import {-# SOURCE #-} qualified GI.Gio.Flags as Gio.Flags
import {-# SOURCE #-} qualified GI.Gio.Interfaces.ActionGroup as Gio.ActionGroup
import {-# SOURCE #-} qualified GI.Gio.Interfaces.AsyncInitable as Gio.AsyncInitable
import {-# SOURCE #-} qualified GI.Gio.Interfaces.AsyncResult as Gio.AsyncResult
import {-# SOURCE #-} qualified GI.Gio.Interfaces.DBusInterface as Gio.DBusInterface
import {-# SOURCE #-} qualified GI.Gio.Interfaces.DBusObject as Gio.DBusObject
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Initable as Gio.Initable
import {-# SOURCE #-} qualified GI.Gio.Objects.Cancellable as Gio.Cancellable
import {-# SOURCE #-} qualified GI.Gio.Objects.Credentials as Gio.Credentials
import {-# SOURCE #-} qualified GI.Gio.Objects.DBusAuthObserver as Gio.DBusAuthObserver
import {-# SOURCE #-} qualified GI.Gio.Objects.DBusConnection as Gio.DBusConnection
import {-# SOURCE #-} qualified GI.Gio.Objects.DBusMessage as Gio.DBusMessage
import {-# SOURCE #-} qualified GI.Gio.Objects.IOStream as Gio.IOStream
import {-# SOURCE #-} qualified GI.Gio.Objects.InputStream as Gio.InputStream
import {-# SOURCE #-} qualified GI.Gio.Objects.MenuAttributeIter as Gio.MenuAttributeIter
import {-# SOURCE #-} qualified GI.Gio.Objects.MenuLinkIter as Gio.MenuLinkIter
import {-# SOURCE #-} qualified GI.Gio.Objects.MenuModel as Gio.MenuModel
import {-# SOURCE #-} qualified GI.Gio.Objects.OutputStream as Gio.OutputStream
import {-# SOURCE #-} qualified GI.Gio.Objects.UnixFDList as Gio.UnixFDList
import {-# SOURCE #-} qualified GI.Gio.Structs.DBusAnnotationInfo as Gio.DBusAnnotationInfo
import {-# SOURCE #-} qualified GI.Gio.Structs.DBusArgInfo as Gio.DBusArgInfo
import {-# SOURCE #-} qualified GI.Gio.Structs.DBusInterfaceInfo as Gio.DBusInterfaceInfo
import {-# SOURCE #-} qualified GI.Gio.Structs.DBusMethodInfo as Gio.DBusMethodInfo
import {-# SOURCE #-} qualified GI.Gio.Structs.DBusPropertyInfo as Gio.DBusPropertyInfo
import {-# SOURCE #-} qualified GI.Gio.Structs.DBusSignalInfo as Gio.DBusSignalInfo
import {-# SOURCE #-} qualified GI.Gio.Structs.DBusSubtreeVTable as Gio.DBusSubtreeVTable
import {-# SOURCE #-} qualified GI.Gio.Structs.OutputVector as Gio.OutputVector

#else
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gio.Callbacks as Gio.Callbacks
import {-# SOURCE #-} qualified GI.Gio.Enums as Gio.Enums
import {-# SOURCE #-} qualified GI.Gio.Flags as Gio.Flags
import {-# SOURCE #-} qualified GI.Gio.Interfaces.AsyncInitable as Gio.AsyncInitable
import {-# SOURCE #-} qualified GI.Gio.Interfaces.AsyncResult as Gio.AsyncResult
import {-# SOURCE #-} qualified GI.Gio.Interfaces.DBusInterface as Gio.DBusInterface
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Initable as Gio.Initable
import {-# SOURCE #-} qualified GI.Gio.Objects.Cancellable as Gio.Cancellable
import {-# SOURCE #-} qualified GI.Gio.Objects.DBusConnection as Gio.DBusConnection
import {-# SOURCE #-} qualified GI.Gio.Objects.UnixFDList as Gio.UnixFDList
import {-# SOURCE #-} qualified GI.Gio.Structs.DBusInterfaceInfo as Gio.DBusInterfaceInfo

#endif

-- | Memory-managed wrapper type.
newtype DBusProxy = DBusProxy (SP.ManagedPtr DBusProxy)
    deriving (Eq)

instance SP.ManagedPtrNewtype DBusProxy where
    toManagedPtr (DBusProxy p) = p

foreign import ccall "g_dbus_proxy_get_type"
    c_g_dbus_proxy_get_type :: IO B.Types.GType

instance B.Types.TypedObject DBusProxy where
    glibType = c_g_dbus_proxy_get_type

instance B.Types.GObject DBusProxy

-- | Type class for types which can be safely cast to t'DBusProxy', for instance with `toDBusProxy`.
class (SP.GObject o, O.IsDescendantOf DBusProxy o) => IsDBusProxy o
instance (SP.GObject o, O.IsDescendantOf DBusProxy o) => IsDBusProxy o

instance O.HasParentTypes DBusProxy
type instance O.ParentTypes DBusProxy = '[GObject.Object.Object, Gio.AsyncInitable.AsyncInitable, Gio.DBusInterface.DBusInterface, Gio.Initable.Initable]

-- | Cast to t'DBusProxy', for types for which this is known to be safe. For general casts, use 'Data.GI.Base.ManagedPtr.castTo'.
toDBusProxy :: (MIO.MonadIO m, IsDBusProxy o) => o -> m DBusProxy
toDBusProxy = MIO.liftIO . B.ManagedPtr.unsafeCastTo DBusProxy

-- | Convert t'DBusProxy' to and from t'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe DBusProxy) where
    gvalueGType_ = c_g_dbus_proxy_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr DBusProxy)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr DBusProxy)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject DBusProxy ptr
        else return P.Nothing
        
    

#if defined(ENABLE_OVERLOADING)
type family ResolveDBusProxyMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveDBusProxyMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolveDBusProxyMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolveDBusProxyMethod "call" o = DBusProxyCallMethodInfo
    ResolveDBusProxyMethod "callFinish" o = DBusProxyCallFinishMethodInfo
    ResolveDBusProxyMethod "callSync" o = DBusProxyCallSyncMethodInfo
    ResolveDBusProxyMethod "callWithUnixFdList" o = DBusProxyCallWithUnixFdListMethodInfo
    ResolveDBusProxyMethod "callWithUnixFdListFinish" o = DBusProxyCallWithUnixFdListFinishMethodInfo
    ResolveDBusProxyMethod "callWithUnixFdListSync" o = DBusProxyCallWithUnixFdListSyncMethodInfo
    ResolveDBusProxyMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolveDBusProxyMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolveDBusProxyMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolveDBusProxyMethod "init" o = Gio.Initable.InitableInitMethodInfo
    ResolveDBusProxyMethod "initAsync" o = Gio.AsyncInitable.AsyncInitableInitAsyncMethodInfo
    ResolveDBusProxyMethod "initFinish" o = Gio.AsyncInitable.AsyncInitableInitFinishMethodInfo
    ResolveDBusProxyMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolveDBusProxyMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolveDBusProxyMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolveDBusProxyMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolveDBusProxyMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolveDBusProxyMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolveDBusProxyMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolveDBusProxyMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolveDBusProxyMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolveDBusProxyMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolveDBusProxyMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolveDBusProxyMethod "getCachedProperty" o = DBusProxyGetCachedPropertyMethodInfo
    ResolveDBusProxyMethod "getCachedPropertyNames" o = DBusProxyGetCachedPropertyNamesMethodInfo
    ResolveDBusProxyMethod "getConnection" o = DBusProxyGetConnectionMethodInfo
    ResolveDBusProxyMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolveDBusProxyMethod "getDefaultTimeout" o = DBusProxyGetDefaultTimeoutMethodInfo
    ResolveDBusProxyMethod "getFlags" o = DBusProxyGetFlagsMethodInfo
    ResolveDBusProxyMethod "getInfo" o = Gio.DBusInterface.DBusInterfaceGetInfoMethodInfo
    ResolveDBusProxyMethod "getInterfaceInfo" o = DBusProxyGetInterfaceInfoMethodInfo
    ResolveDBusProxyMethod "getInterfaceName" o = DBusProxyGetInterfaceNameMethodInfo
    ResolveDBusProxyMethod "getName" o = DBusProxyGetNameMethodInfo
    ResolveDBusProxyMethod "getNameOwner" o = DBusProxyGetNameOwnerMethodInfo
    ResolveDBusProxyMethod "getObject" o = Gio.DBusInterface.DBusInterfaceGetObjectMethodInfo
    ResolveDBusProxyMethod "getObjectPath" o = DBusProxyGetObjectPathMethodInfo
    ResolveDBusProxyMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolveDBusProxyMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolveDBusProxyMethod "setCachedProperty" o = DBusProxySetCachedPropertyMethodInfo
    ResolveDBusProxyMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolveDBusProxyMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolveDBusProxyMethod "setDefaultTimeout" o = DBusProxySetDefaultTimeoutMethodInfo
    ResolveDBusProxyMethod "setInterfaceInfo" o = DBusProxySetInterfaceInfoMethodInfo
    ResolveDBusProxyMethod "setObject" o = Gio.DBusInterface.DBusInterfaceSetObjectMethodInfo
    ResolveDBusProxyMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolveDBusProxyMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveDBusProxyMethod t DBusProxy, O.OverloadedMethod info DBusProxy p) => OL.IsLabel t (DBusProxy -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveDBusProxyMethod t DBusProxy, O.OverloadedMethod info DBusProxy p, R.HasField t DBusProxy p) => R.HasField t DBusProxy p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveDBusProxyMethod t DBusProxy, O.OverloadedMethodInfo info DBusProxy) => OL.IsLabel t (O.MethodProxy info DBusProxy) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

-- signal DBusProxy::g-properties-changed
-- | Emitted when one or more D-Bus properties on /@proxy@/ changes. The
-- local cache has already been updated when this signal fires. Note
-- that both /@changedProperties@/ and /@invalidatedProperties@/ are
-- guaranteed to never be 'P.Nothing' (either may be empty though).
-- 
-- If the proxy has the flag
-- 'GI.Gio.Flags.DBusProxyFlagsGetInvalidatedProperties' set, then
-- /@invalidatedProperties@/ will always be empty.
-- 
-- This signal corresponds to the
-- @PropertiesChanged@ D-Bus signal on the
-- @org.freedesktop.DBus.Properties@ interface.
-- 
-- /Since: 2.26/
type DBusProxyGPropertiesChangedCallback =
    GVariant
    -- ^ /@changedProperties@/: A t'GVariant' containing the properties that changed (type: @a{sv}@)
    -> [T.Text]
    -- ^ /@invalidatedProperties@/: A 'P.Nothing' terminated array of properties that was invalidated
    -> IO ()

type C_DBusProxyGPropertiesChangedCallback =
    Ptr DBusProxy ->                        -- object
    Ptr GVariant ->
    Ptr CString ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_DBusProxyGPropertiesChangedCallback`.
foreign import ccall "wrapper"
    mk_DBusProxyGPropertiesChangedCallback :: C_DBusProxyGPropertiesChangedCallback -> IO (FunPtr C_DBusProxyGPropertiesChangedCallback)

wrap_DBusProxyGPropertiesChangedCallback :: 
    GObject a => (a -> DBusProxyGPropertiesChangedCallback) ->
    C_DBusProxyGPropertiesChangedCallback
wrap_DBusProxyGPropertiesChangedCallback gi'cb gi'selfPtr changedProperties invalidatedProperties _ = do
    changedProperties' <- B.GVariant.newGVariantFromPtr changedProperties
    invalidatedProperties' <- unpackZeroTerminatedUTF8CArray invalidatedProperties
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  changedProperties' invalidatedProperties'


-- | Connect a signal handler for the [gPropertiesChanged](#signal:gPropertiesChanged) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' dBusProxy #gPropertiesChanged callback
-- @
-- 
-- 
onDBusProxyGPropertiesChanged :: (IsDBusProxy a, MonadIO m) => a -> ((?self :: a) => DBusProxyGPropertiesChangedCallback) -> m SignalHandlerId
onDBusProxyGPropertiesChanged obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_DBusProxyGPropertiesChangedCallback wrapped
    wrapped'' <- mk_DBusProxyGPropertiesChangedCallback wrapped'
    connectSignalFunPtr obj "g-properties-changed" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [gPropertiesChanged](#signal:gPropertiesChanged) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' dBusProxy #gPropertiesChanged callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterDBusProxyGPropertiesChanged :: (IsDBusProxy a, MonadIO m) => a -> ((?self :: a) => DBusProxyGPropertiesChangedCallback) -> m SignalHandlerId
afterDBusProxyGPropertiesChanged obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_DBusProxyGPropertiesChangedCallback wrapped
    wrapped'' <- mk_DBusProxyGPropertiesChangedCallback wrapped'
    connectSignalFunPtr obj "g-properties-changed" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data DBusProxyGPropertiesChangedSignalInfo
instance SignalInfo DBusProxyGPropertiesChangedSignalInfo where
    type HaskellCallbackType DBusProxyGPropertiesChangedSignalInfo = DBusProxyGPropertiesChangedCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_DBusProxyGPropertiesChangedCallback cb
        cb'' <- mk_DBusProxyGPropertiesChangedCallback cb'
        connectSignalFunPtr obj "g-properties-changed" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusProxy::g-properties-changed"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusProxy.html#g:signal:gPropertiesChanged"})

#endif

-- signal DBusProxy::g-signal
-- | Emitted when a signal from the remote object and interface that /@proxy@/ is for, has been received.
-- 
-- Since 2.72 this signal supports detailed connections. You can connect to
-- the detailed signal @g-signal::x@ in order to receive callbacks only when
-- signal @x@ is received from the remote object.
-- 
-- /Since: 2.26/
type DBusProxyGSignalCallback =
    Maybe T.Text
    -- ^ /@senderName@/: The sender of the signal or 'P.Nothing' if the connection is not a bus connection.
    -> T.Text
    -- ^ /@signalName@/: The name of the signal.
    -> GVariant
    -- ^ /@parameters@/: A t'GVariant' tuple with parameters for the signal.
    -> IO ()

type C_DBusProxyGSignalCallback =
    Ptr DBusProxy ->                        -- object
    CString ->
    CString ->
    Ptr GVariant ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_DBusProxyGSignalCallback`.
foreign import ccall "wrapper"
    mk_DBusProxyGSignalCallback :: C_DBusProxyGSignalCallback -> IO (FunPtr C_DBusProxyGSignalCallback)

wrap_DBusProxyGSignalCallback :: 
    GObject a => (a -> DBusProxyGSignalCallback) ->
    C_DBusProxyGSignalCallback
wrap_DBusProxyGSignalCallback gi'cb gi'selfPtr senderName signalName parameters _ = do
    maybeSenderName <-
        if senderName == FP.nullPtr
        then return Nothing
        else do
            senderName' <- cstringToText senderName
            return $ Just senderName'
    signalName' <- cstringToText signalName
    parameters' <- B.GVariant.newGVariantFromPtr parameters
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  maybeSenderName signalName' parameters'


-- | Connect a signal handler for the [gSignal](#signal:gSignal) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' dBusProxy #gSignal callback
-- @
-- 
-- This signal admits a optional parameter @detail@.
-- If it's not @Nothing@, we will connect to “@g-signal::detail@” instead.
-- 
onDBusProxyGSignal :: (IsDBusProxy a, MonadIO m) => a -> P.Maybe T.Text -> ((?self :: a) => DBusProxyGSignalCallback) -> m SignalHandlerId
onDBusProxyGSignal obj detail cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_DBusProxyGSignalCallback wrapped
    wrapped'' <- mk_DBusProxyGSignalCallback wrapped'
    connectSignalFunPtr obj "g-signal" wrapped'' SignalConnectBefore detail

-- | Connect a signal handler for the [gSignal](#signal:gSignal) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' dBusProxy #gSignal callback
-- @
-- 
-- This signal admits a optional parameter @detail@.
-- If it's not @Nothing@, we will connect to “@g-signal::detail@” instead.
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterDBusProxyGSignal :: (IsDBusProxy a, MonadIO m) => a -> P.Maybe T.Text -> ((?self :: a) => DBusProxyGSignalCallback) -> m SignalHandlerId
afterDBusProxyGSignal obj detail cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_DBusProxyGSignalCallback wrapped
    wrapped'' <- mk_DBusProxyGSignalCallback wrapped'
    connectSignalFunPtr obj "g-signal" wrapped'' SignalConnectAfter detail


#if defined(ENABLE_OVERLOADING)
data DBusProxyGSignalSignalInfo
instance SignalInfo DBusProxyGSignalSignalInfo where
    type HaskellCallbackType DBusProxyGSignalSignalInfo = DBusProxyGSignalCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_DBusProxyGSignalCallback cb
        cb'' <- mk_DBusProxyGSignalCallback cb'
        connectSignalFunPtr obj "g-signal" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusProxy::g-signal"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusProxy.html#g:signal:gSignal"})

#endif

-- VVV Prop "g-bus-type"
   -- Type: TInterface (Name {namespace = "Gio", name = "BusType"})
   -- Flags: [PropertyWritable,PropertyConstructOnly]
   -- Nullable: (Nothing,Nothing)

-- | Construct a t'GValueConstruct' with valid value for the “@g-bus-type@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructDBusProxyGBusType :: (IsDBusProxy o, MIO.MonadIO m) => Gio.Enums.BusType -> m (GValueConstruct o)
constructDBusProxyGBusType val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyEnum "g-bus-type" val

#if defined(ENABLE_OVERLOADING)
data DBusProxyGBusTypePropertyInfo
instance AttrInfo DBusProxyGBusTypePropertyInfo where
    type AttrAllowedOps DBusProxyGBusTypePropertyInfo = '[ 'AttrConstruct]
    type AttrBaseTypeConstraint DBusProxyGBusTypePropertyInfo = IsDBusProxy
    type AttrSetTypeConstraint DBusProxyGBusTypePropertyInfo = (~) Gio.Enums.BusType
    type AttrTransferTypeConstraint DBusProxyGBusTypePropertyInfo = (~) Gio.Enums.BusType
    type AttrTransferType DBusProxyGBusTypePropertyInfo = Gio.Enums.BusType
    type AttrGetType DBusProxyGBusTypePropertyInfo = ()
    type AttrLabel DBusProxyGBusTypePropertyInfo = "g-bus-type"
    type AttrOrigin DBusProxyGBusTypePropertyInfo = DBusProxy
    attrGet = undefined
    attrSet = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructDBusProxyGBusType
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusProxy.gBusType"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusProxy.html#g:attr:gBusType"
        })
#endif

-- VVV Prop "g-connection"
   -- Type: TInterface (Name {namespace = "Gio", name = "DBusConnection"})
   -- Flags: [PropertyReadable,PropertyWritable,PropertyConstructOnly]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@g-connection@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' dBusProxy #gConnection
-- @
getDBusProxyGConnection :: (MonadIO m, IsDBusProxy o) => o -> m (Maybe Gio.DBusConnection.DBusConnection)
getDBusProxyGConnection obj = MIO.liftIO $ B.Properties.getObjectPropertyObject obj "g-connection" Gio.DBusConnection.DBusConnection

-- | Construct a t'GValueConstruct' with valid value for the “@g-connection@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructDBusProxyGConnection :: (IsDBusProxy o, MIO.MonadIO m, Gio.DBusConnection.IsDBusConnection a) => a -> m (GValueConstruct o)
constructDBusProxyGConnection val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyObject "g-connection" (P.Just val)

#if defined(ENABLE_OVERLOADING)
data DBusProxyGConnectionPropertyInfo
instance AttrInfo DBusProxyGConnectionPropertyInfo where
    type AttrAllowedOps DBusProxyGConnectionPropertyInfo = '[ 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint DBusProxyGConnectionPropertyInfo = IsDBusProxy
    type AttrSetTypeConstraint DBusProxyGConnectionPropertyInfo = Gio.DBusConnection.IsDBusConnection
    type AttrTransferTypeConstraint DBusProxyGConnectionPropertyInfo = Gio.DBusConnection.IsDBusConnection
    type AttrTransferType DBusProxyGConnectionPropertyInfo = Gio.DBusConnection.DBusConnection
    type AttrGetType DBusProxyGConnectionPropertyInfo = (Maybe Gio.DBusConnection.DBusConnection)
    type AttrLabel DBusProxyGConnectionPropertyInfo = "g-connection"
    type AttrOrigin DBusProxyGConnectionPropertyInfo = DBusProxy
    attrGet = getDBusProxyGConnection
    attrSet = undefined
    attrTransfer _ v = do
        unsafeCastTo Gio.DBusConnection.DBusConnection v
    attrConstruct = constructDBusProxyGConnection
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusProxy.gConnection"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusProxy.html#g:attr:gConnection"
        })
#endif

-- VVV Prop "g-default-timeout"
   -- Type: TBasicType TInt
   -- Flags: [PropertyReadable,PropertyWritable,PropertyConstruct]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@g-default-timeout@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' dBusProxy #gDefaultTimeout
-- @
getDBusProxyGDefaultTimeout :: (MonadIO m, IsDBusProxy o) => o -> m Int32
getDBusProxyGDefaultTimeout obj = MIO.liftIO $ B.Properties.getObjectPropertyInt32 obj "g-default-timeout"

-- | Set the value of the “@g-default-timeout@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' dBusProxy [ #gDefaultTimeout 'Data.GI.Base.Attributes.:=' value ]
-- @
setDBusProxyGDefaultTimeout :: (MonadIO m, IsDBusProxy o) => o -> Int32 -> m ()
setDBusProxyGDefaultTimeout obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyInt32 obj "g-default-timeout" val

-- | Construct a t'GValueConstruct' with valid value for the “@g-default-timeout@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructDBusProxyGDefaultTimeout :: (IsDBusProxy o, MIO.MonadIO m) => Int32 -> m (GValueConstruct o)
constructDBusProxyGDefaultTimeout val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyInt32 "g-default-timeout" val

#if defined(ENABLE_OVERLOADING)
data DBusProxyGDefaultTimeoutPropertyInfo
instance AttrInfo DBusProxyGDefaultTimeoutPropertyInfo where
    type AttrAllowedOps DBusProxyGDefaultTimeoutPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint DBusProxyGDefaultTimeoutPropertyInfo = IsDBusProxy
    type AttrSetTypeConstraint DBusProxyGDefaultTimeoutPropertyInfo = (~) Int32
    type AttrTransferTypeConstraint DBusProxyGDefaultTimeoutPropertyInfo = (~) Int32
    type AttrTransferType DBusProxyGDefaultTimeoutPropertyInfo = Int32
    type AttrGetType DBusProxyGDefaultTimeoutPropertyInfo = Int32
    type AttrLabel DBusProxyGDefaultTimeoutPropertyInfo = "g-default-timeout"
    type AttrOrigin DBusProxyGDefaultTimeoutPropertyInfo = DBusProxy
    attrGet = getDBusProxyGDefaultTimeout
    attrSet = setDBusProxyGDefaultTimeout
    attrTransfer _ v = do
        return v
    attrConstruct = constructDBusProxyGDefaultTimeout
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusProxy.gDefaultTimeout"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusProxy.html#g:attr:gDefaultTimeout"
        })
#endif

-- VVV Prop "g-flags"
   -- Type: TInterface (Name {namespace = "Gio", name = "DBusProxyFlags"})
   -- Flags: [PropertyReadable,PropertyWritable,PropertyConstructOnly]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@g-flags@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' dBusProxy #gFlags
-- @
getDBusProxyGFlags :: (MonadIO m, IsDBusProxy o) => o -> m [Gio.Flags.DBusProxyFlags]
getDBusProxyGFlags obj = MIO.liftIO $ B.Properties.getObjectPropertyFlags obj "g-flags"

-- | Construct a t'GValueConstruct' with valid value for the “@g-flags@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructDBusProxyGFlags :: (IsDBusProxy o, MIO.MonadIO m) => [Gio.Flags.DBusProxyFlags] -> m (GValueConstruct o)
constructDBusProxyGFlags val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyFlags "g-flags" val

#if defined(ENABLE_OVERLOADING)
data DBusProxyGFlagsPropertyInfo
instance AttrInfo DBusProxyGFlagsPropertyInfo where
    type AttrAllowedOps DBusProxyGFlagsPropertyInfo = '[ 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint DBusProxyGFlagsPropertyInfo = IsDBusProxy
    type AttrSetTypeConstraint DBusProxyGFlagsPropertyInfo = (~) [Gio.Flags.DBusProxyFlags]
    type AttrTransferTypeConstraint DBusProxyGFlagsPropertyInfo = (~) [Gio.Flags.DBusProxyFlags]
    type AttrTransferType DBusProxyGFlagsPropertyInfo = [Gio.Flags.DBusProxyFlags]
    type AttrGetType DBusProxyGFlagsPropertyInfo = [Gio.Flags.DBusProxyFlags]
    type AttrLabel DBusProxyGFlagsPropertyInfo = "g-flags"
    type AttrOrigin DBusProxyGFlagsPropertyInfo = DBusProxy
    attrGet = getDBusProxyGFlags
    attrSet = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructDBusProxyGFlags
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusProxy.gFlags"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusProxy.html#g:attr:gFlags"
        })
#endif

-- VVV Prop "g-interface-info"
   -- Type: TInterface (Name {namespace = "Gio", name = "DBusInterfaceInfo"})
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@g-interface-info@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' dBusProxy #gInterfaceInfo
-- @
getDBusProxyGInterfaceInfo :: (MonadIO m, IsDBusProxy o) => o -> m (Maybe Gio.DBusInterfaceInfo.DBusInterfaceInfo)
getDBusProxyGInterfaceInfo obj = MIO.liftIO $ B.Properties.getObjectPropertyBoxed obj "g-interface-info" Gio.DBusInterfaceInfo.DBusInterfaceInfo

-- | Set the value of the “@g-interface-info@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' dBusProxy [ #gInterfaceInfo 'Data.GI.Base.Attributes.:=' value ]
-- @
setDBusProxyGInterfaceInfo :: (MonadIO m, IsDBusProxy o) => o -> Gio.DBusInterfaceInfo.DBusInterfaceInfo -> m ()
setDBusProxyGInterfaceInfo obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBoxed obj "g-interface-info" (Just val)

-- | Construct a t'GValueConstruct' with valid value for the “@g-interface-info@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructDBusProxyGInterfaceInfo :: (IsDBusProxy o, MIO.MonadIO m) => Gio.DBusInterfaceInfo.DBusInterfaceInfo -> m (GValueConstruct o)
constructDBusProxyGInterfaceInfo val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBoxed "g-interface-info" (P.Just val)

-- | Set the value of the “@g-interface-info@” property to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #gInterfaceInfo
-- @
clearDBusProxyGInterfaceInfo :: (MonadIO m, IsDBusProxy o) => o -> m ()
clearDBusProxyGInterfaceInfo obj = liftIO $ B.Properties.setObjectPropertyBoxed obj "g-interface-info" (Nothing :: Maybe Gio.DBusInterfaceInfo.DBusInterfaceInfo)

#if defined(ENABLE_OVERLOADING)
data DBusProxyGInterfaceInfoPropertyInfo
instance AttrInfo DBusProxyGInterfaceInfoPropertyInfo where
    type AttrAllowedOps DBusProxyGInterfaceInfoPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint DBusProxyGInterfaceInfoPropertyInfo = IsDBusProxy
    type AttrSetTypeConstraint DBusProxyGInterfaceInfoPropertyInfo = (~) Gio.DBusInterfaceInfo.DBusInterfaceInfo
    type AttrTransferTypeConstraint DBusProxyGInterfaceInfoPropertyInfo = (~) Gio.DBusInterfaceInfo.DBusInterfaceInfo
    type AttrTransferType DBusProxyGInterfaceInfoPropertyInfo = Gio.DBusInterfaceInfo.DBusInterfaceInfo
    type AttrGetType DBusProxyGInterfaceInfoPropertyInfo = (Maybe Gio.DBusInterfaceInfo.DBusInterfaceInfo)
    type AttrLabel DBusProxyGInterfaceInfoPropertyInfo = "g-interface-info"
    type AttrOrigin DBusProxyGInterfaceInfoPropertyInfo = DBusProxy
    attrGet = getDBusProxyGInterfaceInfo
    attrSet = setDBusProxyGInterfaceInfo
    attrTransfer _ v = do
        return v
    attrConstruct = constructDBusProxyGInterfaceInfo
    attrClear = clearDBusProxyGInterfaceInfo
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusProxy.gInterfaceInfo"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusProxy.html#g:attr:gInterfaceInfo"
        })
#endif

-- VVV Prop "g-interface-name"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable,PropertyWritable,PropertyConstructOnly]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@g-interface-name@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' dBusProxy #gInterfaceName
-- @
getDBusProxyGInterfaceName :: (MonadIO m, IsDBusProxy o) => o -> m (Maybe T.Text)
getDBusProxyGInterfaceName obj = MIO.liftIO $ B.Properties.getObjectPropertyString obj "g-interface-name"

-- | Construct a t'GValueConstruct' with valid value for the “@g-interface-name@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructDBusProxyGInterfaceName :: (IsDBusProxy o, MIO.MonadIO m) => T.Text -> m (GValueConstruct o)
constructDBusProxyGInterfaceName val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyString "g-interface-name" (P.Just val)

#if defined(ENABLE_OVERLOADING)
data DBusProxyGInterfaceNamePropertyInfo
instance AttrInfo DBusProxyGInterfaceNamePropertyInfo where
    type AttrAllowedOps DBusProxyGInterfaceNamePropertyInfo = '[ 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint DBusProxyGInterfaceNamePropertyInfo = IsDBusProxy
    type AttrSetTypeConstraint DBusProxyGInterfaceNamePropertyInfo = (~) T.Text
    type AttrTransferTypeConstraint DBusProxyGInterfaceNamePropertyInfo = (~) T.Text
    type AttrTransferType DBusProxyGInterfaceNamePropertyInfo = T.Text
    type AttrGetType DBusProxyGInterfaceNamePropertyInfo = (Maybe T.Text)
    type AttrLabel DBusProxyGInterfaceNamePropertyInfo = "g-interface-name"
    type AttrOrigin DBusProxyGInterfaceNamePropertyInfo = DBusProxy
    attrGet = getDBusProxyGInterfaceName
    attrSet = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructDBusProxyGInterfaceName
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusProxy.gInterfaceName"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusProxy.html#g:attr:gInterfaceName"
        })
#endif

-- VVV Prop "g-name"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable,PropertyWritable,PropertyConstructOnly]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@g-name@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' dBusProxy #gName
-- @
getDBusProxyGName :: (MonadIO m, IsDBusProxy o) => o -> m (Maybe T.Text)
getDBusProxyGName obj = MIO.liftIO $ B.Properties.getObjectPropertyString obj "g-name"

-- | Construct a t'GValueConstruct' with valid value for the “@g-name@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructDBusProxyGName :: (IsDBusProxy o, MIO.MonadIO m) => T.Text -> m (GValueConstruct o)
constructDBusProxyGName val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyString "g-name" (P.Just val)

#if defined(ENABLE_OVERLOADING)
data DBusProxyGNamePropertyInfo
instance AttrInfo DBusProxyGNamePropertyInfo where
    type AttrAllowedOps DBusProxyGNamePropertyInfo = '[ 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint DBusProxyGNamePropertyInfo = IsDBusProxy
    type AttrSetTypeConstraint DBusProxyGNamePropertyInfo = (~) T.Text
    type AttrTransferTypeConstraint DBusProxyGNamePropertyInfo = (~) T.Text
    type AttrTransferType DBusProxyGNamePropertyInfo = T.Text
    type AttrGetType DBusProxyGNamePropertyInfo = (Maybe T.Text)
    type AttrLabel DBusProxyGNamePropertyInfo = "g-name"
    type AttrOrigin DBusProxyGNamePropertyInfo = DBusProxy
    attrGet = getDBusProxyGName
    attrSet = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructDBusProxyGName
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusProxy.gName"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusProxy.html#g:attr:gName"
        })
#endif

-- VVV Prop "g-name-owner"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@g-name-owner@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' dBusProxy #gNameOwner
-- @
getDBusProxyGNameOwner :: (MonadIO m, IsDBusProxy o) => o -> m (Maybe T.Text)
getDBusProxyGNameOwner obj = MIO.liftIO $ B.Properties.getObjectPropertyString obj "g-name-owner"

#if defined(ENABLE_OVERLOADING)
data DBusProxyGNameOwnerPropertyInfo
instance AttrInfo DBusProxyGNameOwnerPropertyInfo where
    type AttrAllowedOps DBusProxyGNameOwnerPropertyInfo = '[ 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint DBusProxyGNameOwnerPropertyInfo = IsDBusProxy
    type AttrSetTypeConstraint DBusProxyGNameOwnerPropertyInfo = (~) ()
    type AttrTransferTypeConstraint DBusProxyGNameOwnerPropertyInfo = (~) ()
    type AttrTransferType DBusProxyGNameOwnerPropertyInfo = ()
    type AttrGetType DBusProxyGNameOwnerPropertyInfo = (Maybe T.Text)
    type AttrLabel DBusProxyGNameOwnerPropertyInfo = "g-name-owner"
    type AttrOrigin DBusProxyGNameOwnerPropertyInfo = DBusProxy
    attrGet = getDBusProxyGNameOwner
    attrSet = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusProxy.gNameOwner"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusProxy.html#g:attr:gNameOwner"
        })
#endif

-- VVV Prop "g-object-path"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable,PropertyWritable,PropertyConstructOnly]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@g-object-path@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' dBusProxy #gObjectPath
-- @
getDBusProxyGObjectPath :: (MonadIO m, IsDBusProxy o) => o -> m (Maybe T.Text)
getDBusProxyGObjectPath obj = MIO.liftIO $ B.Properties.getObjectPropertyString obj "g-object-path"

-- | Construct a t'GValueConstruct' with valid value for the “@g-object-path@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructDBusProxyGObjectPath :: (IsDBusProxy o, MIO.MonadIO m) => T.Text -> m (GValueConstruct o)
constructDBusProxyGObjectPath val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyString "g-object-path" (P.Just val)

#if defined(ENABLE_OVERLOADING)
data DBusProxyGObjectPathPropertyInfo
instance AttrInfo DBusProxyGObjectPathPropertyInfo where
    type AttrAllowedOps DBusProxyGObjectPathPropertyInfo = '[ 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint DBusProxyGObjectPathPropertyInfo = IsDBusProxy
    type AttrSetTypeConstraint DBusProxyGObjectPathPropertyInfo = (~) T.Text
    type AttrTransferTypeConstraint DBusProxyGObjectPathPropertyInfo = (~) T.Text
    type AttrTransferType DBusProxyGObjectPathPropertyInfo = T.Text
    type AttrGetType DBusProxyGObjectPathPropertyInfo = (Maybe T.Text)
    type AttrLabel DBusProxyGObjectPathPropertyInfo = "g-object-path"
    type AttrOrigin DBusProxyGObjectPathPropertyInfo = DBusProxy
    attrGet = getDBusProxyGObjectPath
    attrSet = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructDBusProxyGObjectPath
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusProxy.gObjectPath"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusProxy.html#g:attr:gObjectPath"
        })
#endif

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList DBusProxy
type instance O.AttributeList DBusProxy = DBusProxyAttributeList
type DBusProxyAttributeList = ('[ '("gBusType", DBusProxyGBusTypePropertyInfo), '("gConnection", DBusProxyGConnectionPropertyInfo), '("gDefaultTimeout", DBusProxyGDefaultTimeoutPropertyInfo), '("gFlags", DBusProxyGFlagsPropertyInfo), '("gInterfaceInfo", DBusProxyGInterfaceInfoPropertyInfo), '("gInterfaceName", DBusProxyGInterfaceNamePropertyInfo), '("gName", DBusProxyGNamePropertyInfo), '("gNameOwner", DBusProxyGNameOwnerPropertyInfo), '("gObjectPath", DBusProxyGObjectPathPropertyInfo)] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
dBusProxyGBusType :: AttrLabelProxy "gBusType"
dBusProxyGBusType = AttrLabelProxy

dBusProxyGConnection :: AttrLabelProxy "gConnection"
dBusProxyGConnection = AttrLabelProxy

dBusProxyGDefaultTimeout :: AttrLabelProxy "gDefaultTimeout"
dBusProxyGDefaultTimeout = AttrLabelProxy

dBusProxyGFlags :: AttrLabelProxy "gFlags"
dBusProxyGFlags = AttrLabelProxy

dBusProxyGInterfaceInfo :: AttrLabelProxy "gInterfaceInfo"
dBusProxyGInterfaceInfo = AttrLabelProxy

dBusProxyGInterfaceName :: AttrLabelProxy "gInterfaceName"
dBusProxyGInterfaceName = AttrLabelProxy

dBusProxyGName :: AttrLabelProxy "gName"
dBusProxyGName = AttrLabelProxy

dBusProxyGNameOwner :: AttrLabelProxy "gNameOwner"
dBusProxyGNameOwner = AttrLabelProxy

dBusProxyGObjectPath :: AttrLabelProxy "gObjectPath"
dBusProxyGObjectPath = AttrLabelProxy

#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList DBusProxy = DBusProxySignalList
type DBusProxySignalList = ('[ '("gPropertiesChanged", DBusProxyGPropertiesChangedSignalInfo), '("gSignal", DBusProxyGSignalSignalInfo), '("notify", GObject.Object.ObjectNotifySignalInfo)] :: [(Symbol, DK.Type)])

#endif

-- method DBusProxy::new_finish
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "res"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "A #GAsyncResult obtained from the #GAsyncReadyCallback function passed to g_dbus_proxy_new()."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "DBusProxy" })
-- throws : True
-- Skip return : False

foreign import ccall "g_dbus_proxy_new_finish" g_dbus_proxy_new_finish :: 
    Ptr Gio.AsyncResult.AsyncResult ->      -- res : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr DBusProxy)

-- | Finishes creating a t'GI.Gio.Objects.DBusProxy.DBusProxy'.
-- 
-- /Since: 2.26/
dBusProxyNewFinish ::
    (B.CallStack.HasCallStack, MonadIO m, Gio.AsyncResult.IsAsyncResult a) =>
    a
    -- ^ /@res@/: A t'GI.Gio.Interfaces.AsyncResult.AsyncResult' obtained from the t'GI.Gio.Callbacks.AsyncReadyCallback' function passed to 'GI.Gio.Objects.DBusProxy.dBusProxyNew'.
    -> m DBusProxy
    -- ^ __Returns:__ A t'GI.Gio.Objects.DBusProxy.DBusProxy' or 'P.Nothing' if /@error@/ is set.
    --    Free with 'GI.GObject.Objects.Object.objectUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
dBusProxyNewFinish res = liftIO $ do
    res' <- unsafeManagedPtrCastPtr res
    onException (do
        result <- propagateGError $ g_dbus_proxy_new_finish res'
        checkUnexpectedReturnNULL "dBusProxyNewFinish" result
        result' <- (wrapObject DBusProxy) result
        touchManagedPtr res
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
#endif

-- method DBusProxy::new_for_bus_finish
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "res"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "A #GAsyncResult obtained from the #GAsyncReadyCallback function passed to g_dbus_proxy_new_for_bus()."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "DBusProxy" })
-- throws : True
-- Skip return : False

foreign import ccall "g_dbus_proxy_new_for_bus_finish" g_dbus_proxy_new_for_bus_finish :: 
    Ptr Gio.AsyncResult.AsyncResult ->      -- res : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr DBusProxy)

-- | Finishes creating a t'GI.Gio.Objects.DBusProxy.DBusProxy'.
-- 
-- /Since: 2.26/
dBusProxyNewForBusFinish ::
    (B.CallStack.HasCallStack, MonadIO m, Gio.AsyncResult.IsAsyncResult a) =>
    a
    -- ^ /@res@/: A t'GI.Gio.Interfaces.AsyncResult.AsyncResult' obtained from the t'GI.Gio.Callbacks.AsyncReadyCallback' function passed to 'GI.Gio.Objects.DBusProxy.dBusProxyNewForBus'.
    -> m DBusProxy
    -- ^ __Returns:__ A t'GI.Gio.Objects.DBusProxy.DBusProxy' or 'P.Nothing' if /@error@/ is set.
    --    Free with 'GI.GObject.Objects.Object.objectUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
dBusProxyNewForBusFinish res = liftIO $ do
    res' <- unsafeManagedPtrCastPtr res
    onException (do
        result <- propagateGError $ g_dbus_proxy_new_for_bus_finish res'
        checkUnexpectedReturnNULL "dBusProxyNewForBusFinish" result
        result' <- (wrapObject DBusProxy) result
        touchManagedPtr res
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
#endif

-- method DBusProxy::new_for_bus_sync
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "bus_type"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "BusType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GBusType." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusProxyFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Flags used when constructing the proxy."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "info"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusInterfaceInfo" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "A #GDBusInterfaceInfo specifying the minimal interface\n       that @proxy conforms to or %NULL."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A bus name (well-known or unique)."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "object_path"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "An object path." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "interface_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A D-Bus interface name."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GCancellable or %NULL."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "DBusProxy" })
-- throws : True
-- Skip return : False

foreign import ccall "g_dbus_proxy_new_for_bus_sync" g_dbus_proxy_new_for_bus_sync :: 
    CInt ->                                 -- bus_type : TInterface (Name {namespace = "Gio", name = "BusType"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "DBusProxyFlags"})
    Ptr Gio.DBusInterfaceInfo.DBusInterfaceInfo -> -- info : TInterface (Name {namespace = "Gio", name = "DBusInterfaceInfo"})
    CString ->                              -- name : TBasicType TUTF8
    CString ->                              -- object_path : TBasicType TUTF8
    CString ->                              -- interface_name : TBasicType TUTF8
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr DBusProxy)

-- | Like 'GI.Gio.Objects.DBusProxy.dBusProxyNewSync' but takes a t'GI.Gio.Enums.BusType' instead of a t'GI.Gio.Objects.DBusConnection.DBusConnection'.
-- 
-- t'GI.Gio.Objects.DBusProxy.DBusProxy' is used in this [example][class/@gio@/.DBusProxy@/a/@-watch-proxy-example].
-- 
-- /Since: 2.26/
dBusProxyNewForBusSync ::
    (B.CallStack.HasCallStack, MonadIO m, Gio.Cancellable.IsCancellable a) =>
    Gio.Enums.BusType
    -- ^ /@busType@/: A t'GI.Gio.Enums.BusType'.
    -> [Gio.Flags.DBusProxyFlags]
    -- ^ /@flags@/: Flags used when constructing the proxy.
    -> Maybe (Gio.DBusInterfaceInfo.DBusInterfaceInfo)
    -- ^ /@info@/: A t'GI.Gio.Structs.DBusInterfaceInfo.DBusInterfaceInfo' specifying the minimal interface
    --        that /@proxy@/ conforms to or 'P.Nothing'.
    -> T.Text
    -- ^ /@name@/: A bus name (well-known or unique).
    -> T.Text
    -- ^ /@objectPath@/: An object path.
    -> T.Text
    -- ^ /@interfaceName@/: A D-Bus interface name.
    -> Maybe (a)
    -- ^ /@cancellable@/: A t'GI.Gio.Objects.Cancellable.Cancellable' or 'P.Nothing'.
    -> m DBusProxy
    -- ^ __Returns:__ A t'GI.Gio.Objects.DBusProxy.DBusProxy' or 'P.Nothing' if error is set.
    --    Free with 'GI.GObject.Objects.Object.objectUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
dBusProxyNewForBusSync busType flags info name objectPath interfaceName cancellable = liftIO $ do
    let busType' = (fromIntegral . fromEnum) busType
    let flags' = gflagsToWord flags
    maybeInfo <- case info of
        Nothing -> return FP.nullPtr
        Just jInfo -> do
            jInfo' <- unsafeManagedPtrGetPtr jInfo
            return jInfo'
    name' <- textToCString name
    objectPath' <- textToCString objectPath
    interfaceName' <- textToCString interfaceName
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        result <- propagateGError $ g_dbus_proxy_new_for_bus_sync busType' flags' maybeInfo name' objectPath' interfaceName' maybeCancellable
        checkUnexpectedReturnNULL "dBusProxyNewForBusSync" result
        result' <- (wrapObject DBusProxy) result
        whenJust info touchManagedPtr
        whenJust cancellable touchManagedPtr
        freeMem name'
        freeMem objectPath'
        freeMem interfaceName'
        return result'
     ) (do
        freeMem name'
        freeMem objectPath'
        freeMem interfaceName'
     )

#if defined(ENABLE_OVERLOADING)
#endif

-- method DBusProxy::new_sync
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "connection"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusConnection" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusConnection."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusProxyFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Flags used when constructing the proxy."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "info"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusInterfaceInfo" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "A #GDBusInterfaceInfo specifying the minimal interface that @proxy conforms to or %NULL."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "A bus name (well-known or unique) or %NULL if @connection is not a message bus connection."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "object_path"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "An object path." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "interface_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A D-Bus interface name."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GCancellable or %NULL."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "DBusProxy" })
-- throws : True
-- Skip return : False

foreign import ccall "g_dbus_proxy_new_sync" g_dbus_proxy_new_sync :: 
    Ptr Gio.DBusConnection.DBusConnection -> -- connection : TInterface (Name {namespace = "Gio", name = "DBusConnection"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "DBusProxyFlags"})
    Ptr Gio.DBusInterfaceInfo.DBusInterfaceInfo -> -- info : TInterface (Name {namespace = "Gio", name = "DBusInterfaceInfo"})
    CString ->                              -- name : TBasicType TUTF8
    CString ->                              -- object_path : TBasicType TUTF8
    CString ->                              -- interface_name : TBasicType TUTF8
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr DBusProxy)

-- | Creates a proxy for accessing /@interfaceName@/ on the remote object
-- at /@objectPath@/ owned by /@name@/ at /@connection@/ and synchronously
-- loads D-Bus properties unless the
-- 'GI.Gio.Flags.DBusProxyFlagsDoNotLoadProperties' flag is used.
-- 
-- If the 'GI.Gio.Flags.DBusProxyFlagsDoNotConnectSignals' flag is not set, also sets up
-- match rules for signals. Connect to the [DBusProxy::gSignal]("GI.Gio.Objects.DBusProxy#g:signal:gSignal") signal
-- to handle signals from the remote object.
-- 
-- If both 'GI.Gio.Flags.DBusProxyFlagsDoNotLoadProperties' and
-- 'GI.Gio.Flags.DBusProxyFlagsDoNotConnectSignals' are set, this constructor is
-- guaranteed to return immediately without blocking.
-- 
-- If /@name@/ is a well-known name and the
-- 'GI.Gio.Flags.DBusProxyFlagsDoNotAutoStart' and 'GI.Gio.Flags.DBusProxyFlagsDoNotAutoStartAtConstruction'
-- flags aren\'t set and no name owner currently exists, the message bus
-- will be requested to launch a name owner for the name.
-- 
-- This is a synchronous failable constructor. See 'GI.Gio.Objects.DBusProxy.dBusProxyNew'
-- and 'GI.Gio.Objects.DBusProxy.dBusProxyNewFinish' for the asynchronous version.
-- 
-- t'GI.Gio.Objects.DBusProxy.DBusProxy' is used in this [example][class/@gio@/.DBusProxy@/a/@-watch-proxy-example].
-- 
-- /Since: 2.26/
dBusProxyNewSync ::
    (B.CallStack.HasCallStack, MonadIO m, Gio.DBusConnection.IsDBusConnection a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@connection@/: A t'GI.Gio.Objects.DBusConnection.DBusConnection'.
    -> [Gio.Flags.DBusProxyFlags]
    -- ^ /@flags@/: Flags used when constructing the proxy.
    -> Maybe (Gio.DBusInterfaceInfo.DBusInterfaceInfo)
    -- ^ /@info@/: A t'GI.Gio.Structs.DBusInterfaceInfo.DBusInterfaceInfo' specifying the minimal interface that /@proxy@/ conforms to or 'P.Nothing'.
    -> Maybe (T.Text)
    -- ^ /@name@/: A bus name (well-known or unique) or 'P.Nothing' if /@connection@/ is not a message bus connection.
    -> T.Text
    -- ^ /@objectPath@/: An object path.
    -> T.Text
    -- ^ /@interfaceName@/: A D-Bus interface name.
    -> Maybe (b)
    -- ^ /@cancellable@/: A t'GI.Gio.Objects.Cancellable.Cancellable' or 'P.Nothing'.
    -> m DBusProxy
    -- ^ __Returns:__ A t'GI.Gio.Objects.DBusProxy.DBusProxy' or 'P.Nothing' if error is set.
    --    Free with 'GI.GObject.Objects.Object.objectUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
dBusProxyNewSync connection flags info name objectPath interfaceName cancellable = liftIO $ do
    connection' <- unsafeManagedPtrCastPtr connection
    let flags' = gflagsToWord flags
    maybeInfo <- case info of
        Nothing -> return FP.nullPtr
        Just jInfo -> do
            jInfo' <- unsafeManagedPtrGetPtr jInfo
            return jInfo'
    maybeName <- case name of
        Nothing -> return FP.nullPtr
        Just jName -> do
            jName' <- textToCString jName
            return jName'
    objectPath' <- textToCString objectPath
    interfaceName' <- textToCString interfaceName
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        result <- propagateGError $ g_dbus_proxy_new_sync connection' flags' maybeInfo maybeName objectPath' interfaceName' maybeCancellable
        checkUnexpectedReturnNULL "dBusProxyNewSync" result
        result' <- (wrapObject DBusProxy) result
        touchManagedPtr connection
        whenJust info touchManagedPtr
        whenJust cancellable touchManagedPtr
        freeMem maybeName
        freeMem objectPath'
        freeMem interfaceName'
        return result'
     ) (do
        freeMem maybeName
        freeMem objectPath'
        freeMem interfaceName'
     )

#if defined(ENABLE_OVERLOADING)
#endif

-- method DBusProxy::call
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "proxy"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusProxy" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusProxy." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "method_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Name of method to invoke."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "parameters"
--           , argType = TVariant
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "A #GVariant tuple with parameters for the signal or %NULL if not passing parameters."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusCallFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Flags from the #GDBusCallFlags enumeration."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "timeout_msec"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "The timeout in milliseconds (with %G_MAXINT meaning\n               \"infinite\") or -1 to use the proxy default timeout."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GCancellable or %NULL."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "A #GAsyncReadyCallback to call when the request is satisfied or %NULL if you don't\ncare about the result of the method invocation."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 7
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "The data to pass to @callback."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_dbus_proxy_call" g_dbus_proxy_call :: 
    Ptr DBusProxy ->                        -- proxy : TInterface (Name {namespace = "Gio", name = "DBusProxy"})
    CString ->                              -- method_name : TBasicType TUTF8
    Ptr GVariant ->                         -- parameters : TVariant
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "DBusCallFlags"})
    Int32 ->                                -- timeout_msec : TBasicType TInt
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Asynchronously invokes the /@methodName@/ method on /@proxy@/.
-- 
-- If /@methodName@/ contains any dots, then /@name@/ is split into interface and
-- method name parts. This allows using /@proxy@/ for invoking methods on
-- other interfaces.
-- 
-- If the t'GI.Gio.Objects.DBusConnection.DBusConnection' associated with /@proxy@/ is closed then
-- the operation will fail with 'GI.Gio.Enums.IOErrorEnumClosed'. If
-- /@cancellable@/ is canceled, the operation will fail with
-- 'GI.Gio.Enums.IOErrorEnumCancelled'. If /@parameters@/ contains a value not
-- compatible with the D-Bus protocol, the operation fails with
-- 'GI.Gio.Enums.IOErrorEnumInvalidArgument'.
-- 
-- If the /@parameters@/ t'GVariant' is floating, it is consumed. This allows
-- convenient \'inline\' use of @/g_variant_new()/@, e.g.:
-- 
-- === /C code/
-- >
-- > g_dbus_proxy_call (proxy,
-- >                    "TwoStrings",
-- >                    g_variant_new ("(ss)",
-- >                                   "Thing One",
-- >                                   "Thing Two"),
-- >                    G_DBUS_CALL_FLAGS_NONE,
-- >                    -1,
-- >                    NULL,
-- >                    (GAsyncReadyCallback) two_strings_done,
-- >                    &data);
-- 
-- 
-- If /@proxy@/ has an expected interface (see
-- [DBusProxy:gInterfaceInfo]("GI.Gio.Objects.DBusProxy#g:attr:gInterfaceInfo")) and /@methodName@/ is referenced by it,
-- then the return value is checked against the return type.
-- 
-- This is an asynchronous method. When the operation is finished,
-- /@callback@/ will be invoked in the thread-default main context
-- (see 'GI.GLib.Structs.MainContext.mainContextPushThreadDefault')
-- of the thread you are calling this method from.
-- You can then call 'GI.Gio.Objects.DBusProxy.dBusProxyCallFinish' to get the result of
-- the operation. See 'GI.Gio.Objects.DBusProxy.dBusProxyCallSync' for the synchronous
-- version of this method.
-- 
-- If /@callback@/ is 'P.Nothing' then the D-Bus method call message will be sent with
-- the 'GI.Gio.Flags.DBusMessageFlagsNoReplyExpected' flag set.
-- 
-- /Since: 2.26/
dBusProxyCall ::
    (B.CallStack.HasCallStack, MonadIO m, IsDBusProxy a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@proxy@/: A t'GI.Gio.Objects.DBusProxy.DBusProxy'.
    -> T.Text
    -- ^ /@methodName@/: Name of method to invoke.
    -> Maybe (GVariant)
    -- ^ /@parameters@/: A t'GVariant' tuple with parameters for the signal or 'P.Nothing' if not passing parameters.
    -> [Gio.Flags.DBusCallFlags]
    -- ^ /@flags@/: Flags from the t'GI.Gio.Flags.DBusCallFlags' enumeration.
    -> Int32
    -- ^ /@timeoutMsec@/: The timeout in milliseconds (with @/G_MAXINT/@ meaning
    --                \"infinite\") or -1 to use the proxy default timeout.
    -> Maybe (b)
    -- ^ /@cancellable@/: A t'GI.Gio.Objects.Cancellable.Cancellable' or 'P.Nothing'.
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: A t'GI.Gio.Callbacks.AsyncReadyCallback' to call when the request is satisfied or 'P.Nothing' if you don\'t
    -- care about the result of the method invocation.
    -> m ()
dBusProxyCall proxy methodName parameters flags timeoutMsec cancellable callback = liftIO $ do
    proxy' <- unsafeManagedPtrCastPtr proxy
    methodName' <- textToCString methodName
    maybeParameters <- case parameters of
        Nothing -> return FP.nullPtr
        Just jParameters -> do
            jParameters' <- unsafeManagedPtrGetPtr jParameters
            return jParameters'
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_dbus_proxy_call proxy' methodName' maybeParameters flags' timeoutMsec maybeCancellable maybeCallback userData
    touchManagedPtr proxy
    whenJust parameters touchManagedPtr
    whenJust cancellable touchManagedPtr
    freeMem methodName'
    return ()

#if defined(ENABLE_OVERLOADING)
data DBusProxyCallMethodInfo
instance (signature ~ (T.Text -> Maybe (GVariant) -> [Gio.Flags.DBusCallFlags] -> Int32 -> Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsDBusProxy a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod DBusProxyCallMethodInfo a signature where
    overloadedMethod = dBusProxyCall

instance O.OverloadedMethodInfo DBusProxyCallMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusProxy.dBusProxyCall",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusProxy.html#v:dBusProxyCall"
        })


#endif

-- method DBusProxy::call_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "proxy"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusProxy" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusProxy." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "res"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "A #GAsyncResult obtained from the #GAsyncReadyCallback passed to g_dbus_proxy_call()."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just TVariant
-- throws : True
-- Skip return : False

foreign import ccall "g_dbus_proxy_call_finish" g_dbus_proxy_call_finish :: 
    Ptr DBusProxy ->                        -- proxy : TInterface (Name {namespace = "Gio", name = "DBusProxy"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- res : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr GVariant)

-- | Finishes an operation started with 'GI.Gio.Objects.DBusProxy.dBusProxyCall'.
-- 
-- /Since: 2.26/
dBusProxyCallFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsDBusProxy a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@proxy@/: A t'GI.Gio.Objects.DBusProxy.DBusProxy'.
    -> b
    -- ^ /@res@/: A t'GI.Gio.Interfaces.AsyncResult.AsyncResult' obtained from the t'GI.Gio.Callbacks.AsyncReadyCallback' passed to 'GI.Gio.Objects.DBusProxy.dBusProxyCall'.
    -> m GVariant
    -- ^ __Returns:__ 'P.Nothing' if /@error@/ is set. Otherwise a t'GVariant' tuple with
    -- return values. Free with 'GI.GLib.Structs.Variant.variantUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
dBusProxyCallFinish proxy res = liftIO $ do
    proxy' <- unsafeManagedPtrCastPtr proxy
    res' <- unsafeManagedPtrCastPtr res
    onException (do
        result <- propagateGError $ g_dbus_proxy_call_finish proxy' res'
        checkUnexpectedReturnNULL "dBusProxyCallFinish" result
        result' <- B.GVariant.wrapGVariantPtr result
        touchManagedPtr proxy
        touchManagedPtr res
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data DBusProxyCallFinishMethodInfo
instance (signature ~ (b -> m GVariant), MonadIO m, IsDBusProxy a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod DBusProxyCallFinishMethodInfo a signature where
    overloadedMethod = dBusProxyCallFinish

instance O.OverloadedMethodInfo DBusProxyCallFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusProxy.dBusProxyCallFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusProxy.html#v:dBusProxyCallFinish"
        })


#endif

-- method DBusProxy::call_sync
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "proxy"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusProxy" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusProxy." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "method_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Name of method to invoke."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "parameters"
--           , argType = TVariant
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "A #GVariant tuple with parameters for the signal\n             or %NULL if not passing parameters."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusCallFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Flags from the #GDBusCallFlags enumeration."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "timeout_msec"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "The timeout in milliseconds (with %G_MAXINT meaning\n               \"infinite\") or -1 to use the proxy default timeout."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GCancellable or %NULL."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just TVariant
-- throws : True
-- Skip return : False

foreign import ccall "g_dbus_proxy_call_sync" g_dbus_proxy_call_sync :: 
    Ptr DBusProxy ->                        -- proxy : TInterface (Name {namespace = "Gio", name = "DBusProxy"})
    CString ->                              -- method_name : TBasicType TUTF8
    Ptr GVariant ->                         -- parameters : TVariant
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "DBusCallFlags"})
    Int32 ->                                -- timeout_msec : TBasicType TInt
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr GVariant)

-- | Synchronously invokes the /@methodName@/ method on /@proxy@/.
-- 
-- If /@methodName@/ contains any dots, then /@name@/ is split into interface and
-- method name parts. This allows using /@proxy@/ for invoking methods on
-- other interfaces.
-- 
-- If the t'GI.Gio.Objects.DBusConnection.DBusConnection' associated with /@proxy@/ is disconnected then
-- the operation will fail with 'GI.Gio.Enums.IOErrorEnumClosed'. If
-- /@cancellable@/ is canceled, the operation will fail with
-- 'GI.Gio.Enums.IOErrorEnumCancelled'. If /@parameters@/ contains a value not
-- compatible with the D-Bus protocol, the operation fails with
-- 'GI.Gio.Enums.IOErrorEnumInvalidArgument'.
-- 
-- If the /@parameters@/ t'GVariant' is floating, it is consumed. This allows
-- convenient \'inline\' use of @/g_variant_new()/@, e.g.:
-- 
-- === /C code/
-- >
-- > g_dbus_proxy_call_sync (proxy,
-- >                         "TwoStrings",
-- >                         g_variant_new ("(ss)",
-- >                                        "Thing One",
-- >                                        "Thing Two"),
-- >                         G_DBUS_CALL_FLAGS_NONE,
-- >                         -1,
-- >                         NULL,
-- >                         &error);
-- 
-- 
-- The calling thread is blocked until a reply is received. See
-- 'GI.Gio.Objects.DBusProxy.dBusProxyCall' for the asynchronous version of this
-- method.
-- 
-- If /@proxy@/ has an expected interface (see
-- [DBusProxy:gInterfaceInfo]("GI.Gio.Objects.DBusProxy#g:attr:gInterfaceInfo")) and /@methodName@/ is referenced by it,
-- then the return value is checked against the return type.
-- 
-- /Since: 2.26/
dBusProxyCallSync ::
    (B.CallStack.HasCallStack, MonadIO m, IsDBusProxy a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@proxy@/: A t'GI.Gio.Objects.DBusProxy.DBusProxy'.
    -> T.Text
    -- ^ /@methodName@/: Name of method to invoke.
    -> Maybe (GVariant)
    -- ^ /@parameters@/: A t'GVariant' tuple with parameters for the signal
    --              or 'P.Nothing' if not passing parameters.
    -> [Gio.Flags.DBusCallFlags]
    -- ^ /@flags@/: Flags from the t'GI.Gio.Flags.DBusCallFlags' enumeration.
    -> Int32
    -- ^ /@timeoutMsec@/: The timeout in milliseconds (with @/G_MAXINT/@ meaning
    --                \"infinite\") or -1 to use the proxy default timeout.
    -> Maybe (b)
    -- ^ /@cancellable@/: A t'GI.Gio.Objects.Cancellable.Cancellable' or 'P.Nothing'.
    -> m GVariant
    -- ^ __Returns:__ 'P.Nothing' if /@error@/ is set. Otherwise a t'GVariant' tuple with
    -- return values. Free with 'GI.GLib.Structs.Variant.variantUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
dBusProxyCallSync proxy methodName parameters flags timeoutMsec cancellable = liftIO $ do
    proxy' <- unsafeManagedPtrCastPtr proxy
    methodName' <- textToCString methodName
    maybeParameters <- case parameters of
        Nothing -> return FP.nullPtr
        Just jParameters -> do
            jParameters' <- unsafeManagedPtrGetPtr jParameters
            return jParameters'
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        result <- propagateGError $ g_dbus_proxy_call_sync proxy' methodName' maybeParameters flags' timeoutMsec maybeCancellable
        checkUnexpectedReturnNULL "dBusProxyCallSync" result
        result' <- B.GVariant.wrapGVariantPtr result
        touchManagedPtr proxy
        whenJust parameters touchManagedPtr
        whenJust cancellable touchManagedPtr
        freeMem methodName'
        return result'
     ) (do
        freeMem methodName'
     )

#if defined(ENABLE_OVERLOADING)
data DBusProxyCallSyncMethodInfo
instance (signature ~ (T.Text -> Maybe (GVariant) -> [Gio.Flags.DBusCallFlags] -> Int32 -> Maybe (b) -> m GVariant), MonadIO m, IsDBusProxy a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod DBusProxyCallSyncMethodInfo a signature where
    overloadedMethod = dBusProxyCallSync

instance O.OverloadedMethodInfo DBusProxyCallSyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusProxy.dBusProxyCallSync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusProxy.html#v:dBusProxyCallSync"
        })


#endif

-- method DBusProxy::call_with_unix_fd_list
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "proxy"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusProxy" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusProxy." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "method_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Name of method to invoke."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "parameters"
--           , argType = TVariant
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "A #GVariant tuple with parameters for the signal or %NULL if not passing parameters."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusCallFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Flags from the #GDBusCallFlags enumeration."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "timeout_msec"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "The timeout in milliseconds (with %G_MAXINT meaning\n               \"infinite\") or -1 to use the proxy default timeout."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "fd_list"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "UnixFDList" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GUnixFDList or %NULL."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GCancellable or %NULL."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "A #GAsyncReadyCallback to call when the request is satisfied or %NULL if you don't\ncare about the result of the method invocation."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 8
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "The data to pass to @callback."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_dbus_proxy_call_with_unix_fd_list" g_dbus_proxy_call_with_unix_fd_list :: 
    Ptr DBusProxy ->                        -- proxy : TInterface (Name {namespace = "Gio", name = "DBusProxy"})
    CString ->                              -- method_name : TBasicType TUTF8
    Ptr GVariant ->                         -- parameters : TVariant
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "DBusCallFlags"})
    Int32 ->                                -- timeout_msec : TBasicType TInt
    Ptr Gio.UnixFDList.UnixFDList ->        -- fd_list : TInterface (Name {namespace = "Gio", name = "UnixFDList"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Like 'GI.Gio.Objects.DBusProxy.dBusProxyCall' but also takes a t'GI.Gio.Objects.UnixFDList.UnixFDList' object.
-- 
-- This method is only available on UNIX.
-- 
-- /Since: 2.30/
dBusProxyCallWithUnixFdList ::
    (B.CallStack.HasCallStack, MonadIO m, IsDBusProxy a, Gio.UnixFDList.IsUnixFDList b, Gio.Cancellable.IsCancellable c) =>
    a
    -- ^ /@proxy@/: A t'GI.Gio.Objects.DBusProxy.DBusProxy'.
    -> T.Text
    -- ^ /@methodName@/: Name of method to invoke.
    -> Maybe (GVariant)
    -- ^ /@parameters@/: A t'GVariant' tuple with parameters for the signal or 'P.Nothing' if not passing parameters.
    -> [Gio.Flags.DBusCallFlags]
    -- ^ /@flags@/: Flags from the t'GI.Gio.Flags.DBusCallFlags' enumeration.
    -> Int32
    -- ^ /@timeoutMsec@/: The timeout in milliseconds (with @/G_MAXINT/@ meaning
    --                \"infinite\") or -1 to use the proxy default timeout.
    -> Maybe (b)
    -- ^ /@fdList@/: A t'GI.Gio.Objects.UnixFDList.UnixFDList' or 'P.Nothing'.
    -> Maybe (c)
    -- ^ /@cancellable@/: A t'GI.Gio.Objects.Cancellable.Cancellable' or 'P.Nothing'.
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: A t'GI.Gio.Callbacks.AsyncReadyCallback' to call when the request is satisfied or 'P.Nothing' if you don\'t
    -- care about the result of the method invocation.
    -> m ()
dBusProxyCallWithUnixFdList proxy methodName parameters flags timeoutMsec fdList cancellable callback = liftIO $ do
    proxy' <- unsafeManagedPtrCastPtr proxy
    methodName' <- textToCString methodName
    maybeParameters <- case parameters of
        Nothing -> return FP.nullPtr
        Just jParameters -> do
            jParameters' <- unsafeManagedPtrGetPtr jParameters
            return jParameters'
    let flags' = gflagsToWord flags
    maybeFdList <- case fdList of
        Nothing -> return FP.nullPtr
        Just jFdList -> do
            jFdList' <- unsafeManagedPtrCastPtr jFdList
            return jFdList'
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_dbus_proxy_call_with_unix_fd_list proxy' methodName' maybeParameters flags' timeoutMsec maybeFdList maybeCancellable maybeCallback userData
    touchManagedPtr proxy
    whenJust parameters touchManagedPtr
    whenJust fdList touchManagedPtr
    whenJust cancellable touchManagedPtr
    freeMem methodName'
    return ()

#if defined(ENABLE_OVERLOADING)
data DBusProxyCallWithUnixFdListMethodInfo
instance (signature ~ (T.Text -> Maybe (GVariant) -> [Gio.Flags.DBusCallFlags] -> Int32 -> Maybe (b) -> Maybe (c) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsDBusProxy a, Gio.UnixFDList.IsUnixFDList b, Gio.Cancellable.IsCancellable c) => O.OverloadedMethod DBusProxyCallWithUnixFdListMethodInfo a signature where
    overloadedMethod = dBusProxyCallWithUnixFdList

instance O.OverloadedMethodInfo DBusProxyCallWithUnixFdListMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusProxy.dBusProxyCallWithUnixFdList",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusProxy.html#v:dBusProxyCallWithUnixFdList"
        })


#endif

-- method DBusProxy::call_with_unix_fd_list_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "proxy"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusProxy" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusProxy." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "out_fd_list"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "UnixFDList" }
--           , direction = DirectionOut
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Return location for a #GUnixFDList or %NULL."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "res"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "A #GAsyncResult obtained from the #GAsyncReadyCallback passed to g_dbus_proxy_call_with_unix_fd_list()."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just TVariant
-- throws : True
-- Skip return : False

foreign import ccall "g_dbus_proxy_call_with_unix_fd_list_finish" g_dbus_proxy_call_with_unix_fd_list_finish :: 
    Ptr DBusProxy ->                        -- proxy : TInterface (Name {namespace = "Gio", name = "DBusProxy"})
    Ptr (Ptr Gio.UnixFDList.UnixFDList) ->  -- out_fd_list : TInterface (Name {namespace = "Gio", name = "UnixFDList"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- res : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr GVariant)

-- | Finishes an operation started with 'GI.Gio.Objects.DBusProxy.dBusProxyCallWithUnixFdList'.
-- 
-- /Since: 2.30/
dBusProxyCallWithUnixFdListFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsDBusProxy a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@proxy@/: A t'GI.Gio.Objects.DBusProxy.DBusProxy'.
    -> b
    -- ^ /@res@/: A t'GI.Gio.Interfaces.AsyncResult.AsyncResult' obtained from the t'GI.Gio.Callbacks.AsyncReadyCallback' passed to 'GI.Gio.Objects.DBusProxy.dBusProxyCallWithUnixFdList'.
    -> m ((GVariant, Maybe Gio.UnixFDList.UnixFDList))
    -- ^ __Returns:__ 'P.Nothing' if /@error@/ is set. Otherwise a t'GVariant' tuple with
    -- return values. Free with 'GI.GLib.Structs.Variant.variantUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
dBusProxyCallWithUnixFdListFinish proxy res = liftIO $ do
    proxy' <- unsafeManagedPtrCastPtr proxy
    outFdList <- callocMem :: IO (Ptr (Ptr Gio.UnixFDList.UnixFDList))
    res' <- unsafeManagedPtrCastPtr res
    onException (do
        result <- propagateGError $ g_dbus_proxy_call_with_unix_fd_list_finish proxy' outFdList res'
        checkUnexpectedReturnNULL "dBusProxyCallWithUnixFdListFinish" result
        result' <- B.GVariant.wrapGVariantPtr result
        outFdList' <- peek outFdList
        maybeOutFdList' <- convertIfNonNull outFdList' $ \outFdList'' -> do
            outFdList''' <- (wrapObject Gio.UnixFDList.UnixFDList) outFdList''
            return outFdList'''
        touchManagedPtr proxy
        touchManagedPtr res
        freeMem outFdList
        return (result', maybeOutFdList')
     ) (do
        freeMem outFdList
     )

#if defined(ENABLE_OVERLOADING)
data DBusProxyCallWithUnixFdListFinishMethodInfo
instance (signature ~ (b -> m ((GVariant, Maybe Gio.UnixFDList.UnixFDList))), MonadIO m, IsDBusProxy a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod DBusProxyCallWithUnixFdListFinishMethodInfo a signature where
    overloadedMethod = dBusProxyCallWithUnixFdListFinish

instance O.OverloadedMethodInfo DBusProxyCallWithUnixFdListFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusProxy.dBusProxyCallWithUnixFdListFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusProxy.html#v:dBusProxyCallWithUnixFdListFinish"
        })


#endif

-- method DBusProxy::call_with_unix_fd_list_sync
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "proxy"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusProxy" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusProxy." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "method_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Name of method to invoke."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "parameters"
--           , argType = TVariant
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "A #GVariant tuple with parameters for the signal\n             or %NULL if not passing parameters."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusCallFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Flags from the #GDBusCallFlags enumeration."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "timeout_msec"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "The timeout in milliseconds (with %G_MAXINT meaning\n               \"infinite\") or -1 to use the proxy default timeout."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "fd_list"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "UnixFDList" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GUnixFDList or %NULL."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "out_fd_list"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "UnixFDList" }
--           , direction = DirectionOut
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Return location for a #GUnixFDList or %NULL."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GCancellable or %NULL."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just TVariant
-- throws : True
-- Skip return : False

foreign import ccall "g_dbus_proxy_call_with_unix_fd_list_sync" g_dbus_proxy_call_with_unix_fd_list_sync :: 
    Ptr DBusProxy ->                        -- proxy : TInterface (Name {namespace = "Gio", name = "DBusProxy"})
    CString ->                              -- method_name : TBasicType TUTF8
    Ptr GVariant ->                         -- parameters : TVariant
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "DBusCallFlags"})
    Int32 ->                                -- timeout_msec : TBasicType TInt
    Ptr Gio.UnixFDList.UnixFDList ->        -- fd_list : TInterface (Name {namespace = "Gio", name = "UnixFDList"})
    Ptr (Ptr Gio.UnixFDList.UnixFDList) ->  -- out_fd_list : TInterface (Name {namespace = "Gio", name = "UnixFDList"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr GVariant)

-- | Like 'GI.Gio.Objects.DBusProxy.dBusProxyCallSync' but also takes and returns t'GI.Gio.Objects.UnixFDList.UnixFDList' objects.
-- 
-- This method is only available on UNIX.
-- 
-- /Since: 2.30/
dBusProxyCallWithUnixFdListSync ::
    (B.CallStack.HasCallStack, MonadIO m, IsDBusProxy a, Gio.UnixFDList.IsUnixFDList b, Gio.Cancellable.IsCancellable c) =>
    a
    -- ^ /@proxy@/: A t'GI.Gio.Objects.DBusProxy.DBusProxy'.
    -> T.Text
    -- ^ /@methodName@/: Name of method to invoke.
    -> Maybe (GVariant)
    -- ^ /@parameters@/: A t'GVariant' tuple with parameters for the signal
    --              or 'P.Nothing' if not passing parameters.
    -> [Gio.Flags.DBusCallFlags]
    -- ^ /@flags@/: Flags from the t'GI.Gio.Flags.DBusCallFlags' enumeration.
    -> Int32
    -- ^ /@timeoutMsec@/: The timeout in milliseconds (with @/G_MAXINT/@ meaning
    --                \"infinite\") or -1 to use the proxy default timeout.
    -> Maybe (b)
    -- ^ /@fdList@/: A t'GI.Gio.Objects.UnixFDList.UnixFDList' or 'P.Nothing'.
    -> Maybe (c)
    -- ^ /@cancellable@/: A t'GI.Gio.Objects.Cancellable.Cancellable' or 'P.Nothing'.
    -> m ((GVariant, Maybe Gio.UnixFDList.UnixFDList))
    -- ^ __Returns:__ 'P.Nothing' if /@error@/ is set. Otherwise a t'GVariant' tuple with
    -- return values. Free with 'GI.GLib.Structs.Variant.variantUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
dBusProxyCallWithUnixFdListSync proxy methodName parameters flags timeoutMsec fdList cancellable = liftIO $ do
    proxy' <- unsafeManagedPtrCastPtr proxy
    methodName' <- textToCString methodName
    maybeParameters <- case parameters of
        Nothing -> return FP.nullPtr
        Just jParameters -> do
            jParameters' <- unsafeManagedPtrGetPtr jParameters
            return jParameters'
    let flags' = gflagsToWord flags
    maybeFdList <- case fdList of
        Nothing -> return FP.nullPtr
        Just jFdList -> do
            jFdList' <- unsafeManagedPtrCastPtr jFdList
            return jFdList'
    outFdList <- callocMem :: IO (Ptr (Ptr Gio.UnixFDList.UnixFDList))
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        result <- propagateGError $ g_dbus_proxy_call_with_unix_fd_list_sync proxy' methodName' maybeParameters flags' timeoutMsec maybeFdList outFdList maybeCancellable
        checkUnexpectedReturnNULL "dBusProxyCallWithUnixFdListSync" result
        result' <- B.GVariant.wrapGVariantPtr result
        outFdList' <- peek outFdList
        maybeOutFdList' <- convertIfNonNull outFdList' $ \outFdList'' -> do
            outFdList''' <- (wrapObject Gio.UnixFDList.UnixFDList) outFdList''
            return outFdList'''
        touchManagedPtr proxy
        whenJust parameters touchManagedPtr
        whenJust fdList touchManagedPtr
        whenJust cancellable touchManagedPtr
        freeMem methodName'
        freeMem outFdList
        return (result', maybeOutFdList')
     ) (do
        freeMem methodName'
        freeMem outFdList
     )

#if defined(ENABLE_OVERLOADING)
data DBusProxyCallWithUnixFdListSyncMethodInfo
instance (signature ~ (T.Text -> Maybe (GVariant) -> [Gio.Flags.DBusCallFlags] -> Int32 -> Maybe (b) -> Maybe (c) -> m ((GVariant, Maybe Gio.UnixFDList.UnixFDList))), MonadIO m, IsDBusProxy a, Gio.UnixFDList.IsUnixFDList b, Gio.Cancellable.IsCancellable c) => O.OverloadedMethod DBusProxyCallWithUnixFdListSyncMethodInfo a signature where
    overloadedMethod = dBusProxyCallWithUnixFdListSync

instance O.OverloadedMethodInfo DBusProxyCallWithUnixFdListSyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusProxy.dBusProxyCallWithUnixFdListSync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusProxy.html#v:dBusProxyCallWithUnixFdListSync"
        })


#endif

-- method DBusProxy::get_cached_property
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "proxy"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusProxy" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusProxy." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "property_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Property name." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just TVariant
-- throws : False
-- Skip return : False

foreign import ccall "g_dbus_proxy_get_cached_property" g_dbus_proxy_get_cached_property :: 
    Ptr DBusProxy ->                        -- proxy : TInterface (Name {namespace = "Gio", name = "DBusProxy"})
    CString ->                              -- property_name : TBasicType TUTF8
    IO (Ptr GVariant)

-- | Looks up the value for a property from the cache. This call does no
-- blocking IO.
-- 
-- If /@proxy@/ has an expected interface (see
-- [DBusProxy:gInterfaceInfo]("GI.Gio.Objects.DBusProxy#g:attr:gInterfaceInfo")) and /@propertyName@/ is referenced by
-- it, then /@value@/ is checked against the type of the property.
-- 
-- /Since: 2.26/
dBusProxyGetCachedProperty ::
    (B.CallStack.HasCallStack, MonadIO m, IsDBusProxy a) =>
    a
    -- ^ /@proxy@/: A t'GI.Gio.Objects.DBusProxy.DBusProxy'.
    -> T.Text
    -- ^ /@propertyName@/: Property name.
    -> m (Maybe GVariant)
    -- ^ __Returns:__ A reference to the t'GVariant' instance
    --    that holds the value for /@propertyName@/ or 'P.Nothing' if the value is not in
    --    the cache. The returned reference must be freed with 'GI.GLib.Structs.Variant.variantUnref'.
dBusProxyGetCachedProperty proxy propertyName = liftIO $ do
    proxy' <- unsafeManagedPtrCastPtr proxy
    propertyName' <- textToCString propertyName
    result <- g_dbus_proxy_get_cached_property proxy' propertyName'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- B.GVariant.wrapGVariantPtr result'
        return result''
    touchManagedPtr proxy
    freeMem propertyName'
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data DBusProxyGetCachedPropertyMethodInfo
instance (signature ~ (T.Text -> m (Maybe GVariant)), MonadIO m, IsDBusProxy a) => O.OverloadedMethod DBusProxyGetCachedPropertyMethodInfo a signature where
    overloadedMethod = dBusProxyGetCachedProperty

instance O.OverloadedMethodInfo DBusProxyGetCachedPropertyMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusProxy.dBusProxyGetCachedProperty",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusProxy.html#v:dBusProxyGetCachedProperty"
        })


#endif

-- method DBusProxy::get_cached_property_names
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "proxy"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusProxy" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusProxy." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TCArray True (-1) (-1) (TBasicType TUTF8))
-- throws : False
-- Skip return : False

foreign import ccall "g_dbus_proxy_get_cached_property_names" g_dbus_proxy_get_cached_property_names :: 
    Ptr DBusProxy ->                        -- proxy : TInterface (Name {namespace = "Gio", name = "DBusProxy"})
    IO (Ptr CString)

-- | Gets the names of all cached properties on /@proxy@/.
-- 
-- /Since: 2.26/
dBusProxyGetCachedPropertyNames ::
    (B.CallStack.HasCallStack, MonadIO m, IsDBusProxy a) =>
    a
    -- ^ /@proxy@/: A t'GI.Gio.Objects.DBusProxy.DBusProxy'.
    -> m (Maybe [T.Text])
    -- ^ __Returns:__ A
    --          'P.Nothing'-terminated array of strings or 'P.Nothing' if
    --          /@proxy@/ has no cached properties. Free the returned array with
    --          'GI.GLib.Functions.strfreev'.
dBusProxyGetCachedPropertyNames proxy = liftIO $ do
    proxy' <- unsafeManagedPtrCastPtr proxy
    result <- g_dbus_proxy_get_cached_property_names proxy'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- unpackZeroTerminatedUTF8CArray result'
        mapZeroTerminatedCArray freeMem result'
        freeMem result'
        return result''
    touchManagedPtr proxy
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data DBusProxyGetCachedPropertyNamesMethodInfo
instance (signature ~ (m (Maybe [T.Text])), MonadIO m, IsDBusProxy a) => O.OverloadedMethod DBusProxyGetCachedPropertyNamesMethodInfo a signature where
    overloadedMethod = dBusProxyGetCachedPropertyNames

instance O.OverloadedMethodInfo DBusProxyGetCachedPropertyNamesMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusProxy.dBusProxyGetCachedPropertyNames",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusProxy.html#v:dBusProxyGetCachedPropertyNames"
        })


#endif

-- method DBusProxy::get_connection
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "proxy"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusProxy" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusProxy." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gio" , name = "DBusConnection" })
-- throws : False
-- Skip return : False

foreign import ccall "g_dbus_proxy_get_connection" g_dbus_proxy_get_connection :: 
    Ptr DBusProxy ->                        -- proxy : TInterface (Name {namespace = "Gio", name = "DBusProxy"})
    IO (Ptr Gio.DBusConnection.DBusConnection)

-- | Gets the connection /@proxy@/ is for.
-- 
-- /Since: 2.26/
dBusProxyGetConnection ::
    (B.CallStack.HasCallStack, MonadIO m, IsDBusProxy a) =>
    a
    -- ^ /@proxy@/: A t'GI.Gio.Objects.DBusProxy.DBusProxy'.
    -> m Gio.DBusConnection.DBusConnection
    -- ^ __Returns:__ A t'GI.Gio.Objects.DBusConnection.DBusConnection' owned by /@proxy@/. Do not free.
dBusProxyGetConnection proxy = liftIO $ do
    proxy' <- unsafeManagedPtrCastPtr proxy
    result <- g_dbus_proxy_get_connection proxy'
    checkUnexpectedReturnNULL "dBusProxyGetConnection" result
    result' <- (newObject Gio.DBusConnection.DBusConnection) result
    touchManagedPtr proxy
    return result'

#if defined(ENABLE_OVERLOADING)
data DBusProxyGetConnectionMethodInfo
instance (signature ~ (m Gio.DBusConnection.DBusConnection), MonadIO m, IsDBusProxy a) => O.OverloadedMethod DBusProxyGetConnectionMethodInfo a signature where
    overloadedMethod = dBusProxyGetConnection

instance O.OverloadedMethodInfo DBusProxyGetConnectionMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusProxy.dBusProxyGetConnection",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusProxy.html#v:dBusProxyGetConnection"
        })


#endif

-- method DBusProxy::get_default_timeout
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "proxy"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusProxy" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusProxy." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_dbus_proxy_get_default_timeout" g_dbus_proxy_get_default_timeout :: 
    Ptr DBusProxy ->                        -- proxy : TInterface (Name {namespace = "Gio", name = "DBusProxy"})
    IO Int32

-- | Gets the timeout to use if -1 (specifying default timeout) is
-- passed as /@timeoutMsec@/ in the 'GI.Gio.Objects.DBusProxy.dBusProxyCall' and
-- 'GI.Gio.Objects.DBusProxy.dBusProxyCallSync' functions.
-- 
-- See the [DBusProxy:gDefaultTimeout]("GI.Gio.Objects.DBusProxy#g:attr:gDefaultTimeout") property for more details.
-- 
-- /Since: 2.26/
dBusProxyGetDefaultTimeout ::
    (B.CallStack.HasCallStack, MonadIO m, IsDBusProxy a) =>
    a
    -- ^ /@proxy@/: A t'GI.Gio.Objects.DBusProxy.DBusProxy'.
    -> m Int32
    -- ^ __Returns:__ Timeout to use for /@proxy@/.
dBusProxyGetDefaultTimeout proxy = liftIO $ do
    proxy' <- unsafeManagedPtrCastPtr proxy
    result <- g_dbus_proxy_get_default_timeout proxy'
    touchManagedPtr proxy
    return result

#if defined(ENABLE_OVERLOADING)
data DBusProxyGetDefaultTimeoutMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsDBusProxy a) => O.OverloadedMethod DBusProxyGetDefaultTimeoutMethodInfo a signature where
    overloadedMethod = dBusProxyGetDefaultTimeout

instance O.OverloadedMethodInfo DBusProxyGetDefaultTimeoutMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusProxy.dBusProxyGetDefaultTimeout",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusProxy.html#v:dBusProxyGetDefaultTimeout"
        })


#endif

-- method DBusProxy::get_flags
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "proxy"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusProxy" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusProxy." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gio" , name = "DBusProxyFlags" })
-- throws : False
-- Skip return : False

foreign import ccall "g_dbus_proxy_get_flags" g_dbus_proxy_get_flags :: 
    Ptr DBusProxy ->                        -- proxy : TInterface (Name {namespace = "Gio", name = "DBusProxy"})
    IO CUInt

-- | Gets the flags that /@proxy@/ was constructed with.
-- 
-- /Since: 2.26/
dBusProxyGetFlags ::
    (B.CallStack.HasCallStack, MonadIO m, IsDBusProxy a) =>
    a
    -- ^ /@proxy@/: A t'GI.Gio.Objects.DBusProxy.DBusProxy'.
    -> m [Gio.Flags.DBusProxyFlags]
    -- ^ __Returns:__ Flags from the t'GI.Gio.Flags.DBusProxyFlags' enumeration.
dBusProxyGetFlags proxy = liftIO $ do
    proxy' <- unsafeManagedPtrCastPtr proxy
    result <- g_dbus_proxy_get_flags proxy'
    let result' = wordToGFlags result
    touchManagedPtr proxy
    return result'

#if defined(ENABLE_OVERLOADING)
data DBusProxyGetFlagsMethodInfo
instance (signature ~ (m [Gio.Flags.DBusProxyFlags]), MonadIO m, IsDBusProxy a) => O.OverloadedMethod DBusProxyGetFlagsMethodInfo a signature where
    overloadedMethod = dBusProxyGetFlags

instance O.OverloadedMethodInfo DBusProxyGetFlagsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusProxy.dBusProxyGetFlags",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusProxy.html#v:dBusProxyGetFlags"
        })


#endif

-- method DBusProxy::get_interface_info
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "proxy"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusProxy" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusProxy" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface
--                  Name { namespace = "Gio" , name = "DBusInterfaceInfo" })
-- throws : False
-- Skip return : False

foreign import ccall "g_dbus_proxy_get_interface_info" g_dbus_proxy_get_interface_info :: 
    Ptr DBusProxy ->                        -- proxy : TInterface (Name {namespace = "Gio", name = "DBusProxy"})
    IO (Ptr Gio.DBusInterfaceInfo.DBusInterfaceInfo)

-- | Returns the t'GI.Gio.Structs.DBusInterfaceInfo.DBusInterfaceInfo', if any, specifying the interface
-- that /@proxy@/ conforms to. See the [DBusProxy:gInterfaceInfo]("GI.Gio.Objects.DBusProxy#g:attr:gInterfaceInfo")
-- property for more details.
-- 
-- /Since: 2.26/
dBusProxyGetInterfaceInfo ::
    (B.CallStack.HasCallStack, MonadIO m, IsDBusProxy a) =>
    a
    -- ^ /@proxy@/: A t'GI.Gio.Objects.DBusProxy.DBusProxy'
    -> m (Maybe Gio.DBusInterfaceInfo.DBusInterfaceInfo)
    -- ^ __Returns:__ A t'GI.Gio.Structs.DBusInterfaceInfo.DBusInterfaceInfo' or 'P.Nothing'.
    --    Do not unref the returned object, it is owned by /@proxy@/.
dBusProxyGetInterfaceInfo proxy = liftIO $ do
    proxy' <- unsafeManagedPtrCastPtr proxy
    result <- g_dbus_proxy_get_interface_info proxy'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newBoxed Gio.DBusInterfaceInfo.DBusInterfaceInfo) result'
        return result''
    touchManagedPtr proxy
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data DBusProxyGetInterfaceInfoMethodInfo
instance (signature ~ (m (Maybe Gio.DBusInterfaceInfo.DBusInterfaceInfo)), MonadIO m, IsDBusProxy a) => O.OverloadedMethod DBusProxyGetInterfaceInfoMethodInfo a signature where
    overloadedMethod = dBusProxyGetInterfaceInfo

instance O.OverloadedMethodInfo DBusProxyGetInterfaceInfoMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusProxy.dBusProxyGetInterfaceInfo",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusProxy.html#v:dBusProxyGetInterfaceInfo"
        })


#endif

-- method DBusProxy::get_interface_name
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "proxy"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusProxy" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusProxy." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_dbus_proxy_get_interface_name" g_dbus_proxy_get_interface_name :: 
    Ptr DBusProxy ->                        -- proxy : TInterface (Name {namespace = "Gio", name = "DBusProxy"})
    IO CString

-- | Gets the D-Bus interface name /@proxy@/ is for.
-- 
-- /Since: 2.26/
dBusProxyGetInterfaceName ::
    (B.CallStack.HasCallStack, MonadIO m, IsDBusProxy a) =>
    a
    -- ^ /@proxy@/: A t'GI.Gio.Objects.DBusProxy.DBusProxy'.
    -> m T.Text
    -- ^ __Returns:__ A string owned by /@proxy@/. Do not free.
dBusProxyGetInterfaceName proxy = liftIO $ do
    proxy' <- unsafeManagedPtrCastPtr proxy
    result <- g_dbus_proxy_get_interface_name proxy'
    checkUnexpectedReturnNULL "dBusProxyGetInterfaceName" result
    result' <- cstringToText result
    touchManagedPtr proxy
    return result'

#if defined(ENABLE_OVERLOADING)
data DBusProxyGetInterfaceNameMethodInfo
instance (signature ~ (m T.Text), MonadIO m, IsDBusProxy a) => O.OverloadedMethod DBusProxyGetInterfaceNameMethodInfo a signature where
    overloadedMethod = dBusProxyGetInterfaceName

instance O.OverloadedMethodInfo DBusProxyGetInterfaceNameMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusProxy.dBusProxyGetInterfaceName",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusProxy.html#v:dBusProxyGetInterfaceName"
        })


#endif

-- method DBusProxy::get_name
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "proxy"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusProxy" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusProxy." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_dbus_proxy_get_name" g_dbus_proxy_get_name :: 
    Ptr DBusProxy ->                        -- proxy : TInterface (Name {namespace = "Gio", name = "DBusProxy"})
    IO CString

-- | Gets the name that /@proxy@/ was constructed for.
-- 
-- When connected to a message bus, this will usually be non-'P.Nothing'.
-- However, it may be 'P.Nothing' for a proxy that communicates using a peer-to-peer
-- pattern.
-- 
-- /Since: 2.26/
dBusProxyGetName ::
    (B.CallStack.HasCallStack, MonadIO m, IsDBusProxy a) =>
    a
    -- ^ /@proxy@/: A t'GI.Gio.Objects.DBusProxy.DBusProxy'.
    -> m (Maybe T.Text)
    -- ^ __Returns:__ A string owned by /@proxy@/. Do not free.
dBusProxyGetName proxy = liftIO $ do
    proxy' <- unsafeManagedPtrCastPtr proxy
    result <- g_dbus_proxy_get_name proxy'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        return result''
    touchManagedPtr proxy
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data DBusProxyGetNameMethodInfo
instance (signature ~ (m (Maybe T.Text)), MonadIO m, IsDBusProxy a) => O.OverloadedMethod DBusProxyGetNameMethodInfo a signature where
    overloadedMethod = dBusProxyGetName

instance O.OverloadedMethodInfo DBusProxyGetNameMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusProxy.dBusProxyGetName",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusProxy.html#v:dBusProxyGetName"
        })


#endif

-- method DBusProxy::get_name_owner
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "proxy"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusProxy" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusProxy." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_dbus_proxy_get_name_owner" g_dbus_proxy_get_name_owner :: 
    Ptr DBusProxy ->                        -- proxy : TInterface (Name {namespace = "Gio", name = "DBusProxy"})
    IO CString

-- | The unique name that owns the name that /@proxy@/ is for or 'P.Nothing' if
-- no-one currently owns that name. You may connect to the
-- [Object::notify]("GI.GObject.Objects.Object#g:signal:notify") signal to track changes to the
-- [DBusProxy:gNameOwner]("GI.Gio.Objects.DBusProxy#g:attr:gNameOwner") property.
-- 
-- /Since: 2.26/
dBusProxyGetNameOwner ::
    (B.CallStack.HasCallStack, MonadIO m, IsDBusProxy a) =>
    a
    -- ^ /@proxy@/: A t'GI.Gio.Objects.DBusProxy.DBusProxy'.
    -> m (Maybe T.Text)
    -- ^ __Returns:__ The name owner or 'P.Nothing' if no name
    --    owner exists. Free with 'GI.GLib.Functions.free'.
dBusProxyGetNameOwner proxy = liftIO $ do
    proxy' <- unsafeManagedPtrCastPtr proxy
    result <- g_dbus_proxy_get_name_owner proxy'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        freeMem result'
        return result''
    touchManagedPtr proxy
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data DBusProxyGetNameOwnerMethodInfo
instance (signature ~ (m (Maybe T.Text)), MonadIO m, IsDBusProxy a) => O.OverloadedMethod DBusProxyGetNameOwnerMethodInfo a signature where
    overloadedMethod = dBusProxyGetNameOwner

instance O.OverloadedMethodInfo DBusProxyGetNameOwnerMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusProxy.dBusProxyGetNameOwner",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusProxy.html#v:dBusProxyGetNameOwner"
        })


#endif

-- method DBusProxy::get_object_path
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "proxy"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusProxy" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusProxy." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_dbus_proxy_get_object_path" g_dbus_proxy_get_object_path :: 
    Ptr DBusProxy ->                        -- proxy : TInterface (Name {namespace = "Gio", name = "DBusProxy"})
    IO CString

-- | Gets the object path /@proxy@/ is for.
-- 
-- /Since: 2.26/
dBusProxyGetObjectPath ::
    (B.CallStack.HasCallStack, MonadIO m, IsDBusProxy a) =>
    a
    -- ^ /@proxy@/: A t'GI.Gio.Objects.DBusProxy.DBusProxy'.
    -> m T.Text
    -- ^ __Returns:__ A string owned by /@proxy@/. Do not free.
dBusProxyGetObjectPath proxy = liftIO $ do
    proxy' <- unsafeManagedPtrCastPtr proxy
    result <- g_dbus_proxy_get_object_path proxy'
    checkUnexpectedReturnNULL "dBusProxyGetObjectPath" result
    result' <- cstringToText result
    touchManagedPtr proxy
    return result'

#if defined(ENABLE_OVERLOADING)
data DBusProxyGetObjectPathMethodInfo
instance (signature ~ (m T.Text), MonadIO m, IsDBusProxy a) => O.OverloadedMethod DBusProxyGetObjectPathMethodInfo a signature where
    overloadedMethod = dBusProxyGetObjectPath

instance O.OverloadedMethodInfo DBusProxyGetObjectPathMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusProxy.dBusProxyGetObjectPath",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusProxy.html#v:dBusProxyGetObjectPath"
        })


#endif

-- method DBusProxy::set_cached_property
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "proxy"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusProxy" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusProxy" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "property_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Property name." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "value"
--           , argType = TVariant
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "Value for the property or %NULL to remove it from the cache."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_dbus_proxy_set_cached_property" g_dbus_proxy_set_cached_property :: 
    Ptr DBusProxy ->                        -- proxy : TInterface (Name {namespace = "Gio", name = "DBusProxy"})
    CString ->                              -- property_name : TBasicType TUTF8
    Ptr GVariant ->                         -- value : TVariant
    IO ()

-- | If /@value@/ is not 'P.Nothing', sets the cached value for the property with
-- name /@propertyName@/ to the value in /@value@/.
-- 
-- If /@value@/ is 'P.Nothing', then the cached value is removed from the
-- property cache.
-- 
-- If /@proxy@/ has an expected interface (see
-- [DBusProxy:gInterfaceInfo]("GI.Gio.Objects.DBusProxy#g:attr:gInterfaceInfo")) and /@propertyName@/ is referenced by
-- it, then /@value@/ is checked against the type of the property.
-- 
-- If the /@value@/ t'GVariant' is floating, it is consumed. This allows
-- convenient \'inline\' use of @/g_variant_new()/@, e.g.
-- 
-- === /C code/
-- >
-- > g_dbus_proxy_set_cached_property (proxy,
-- >                                   "SomeProperty",
-- >                                   g_variant_new ("(si)",
-- >                                                 "A String",
-- >                                                 42));
-- 
-- 
-- Normally you will not need to use this method since /@proxy@/
-- is tracking changes using the
-- @org.freedesktop.DBus.Properties.PropertiesChanged@
-- D-Bus signal. However, for performance reasons an object may
-- decide to not use this signal for some properties and instead
-- use a proprietary out-of-band mechanism to transmit changes.
-- 
-- As a concrete example, consider an object with a property
-- @ChatroomParticipants@ which is an array of strings. Instead of
-- transmitting the same (long) array every time the property changes,
-- it is more efficient to only transmit the delta using e.g. signals
-- @ChatroomParticipantJoined(String name)@ and
-- @ChatroomParticipantParted(String name)@.
-- 
-- /Since: 2.26/
dBusProxySetCachedProperty ::
    (B.CallStack.HasCallStack, MonadIO m, IsDBusProxy a) =>
    a
    -- ^ /@proxy@/: A t'GI.Gio.Objects.DBusProxy.DBusProxy'
    -> T.Text
    -- ^ /@propertyName@/: Property name.
    -> Maybe (GVariant)
    -- ^ /@value@/: Value for the property or 'P.Nothing' to remove it from the cache.
    -> m ()
dBusProxySetCachedProperty proxy propertyName value = liftIO $ do
    proxy' <- unsafeManagedPtrCastPtr proxy
    propertyName' <- textToCString propertyName
    maybeValue <- case value of
        Nothing -> return FP.nullPtr
        Just jValue -> do
            jValue' <- unsafeManagedPtrGetPtr jValue
            return jValue'
    g_dbus_proxy_set_cached_property proxy' propertyName' maybeValue
    touchManagedPtr proxy
    whenJust value touchManagedPtr
    freeMem propertyName'
    return ()

#if defined(ENABLE_OVERLOADING)
data DBusProxySetCachedPropertyMethodInfo
instance (signature ~ (T.Text -> Maybe (GVariant) -> m ()), MonadIO m, IsDBusProxy a) => O.OverloadedMethod DBusProxySetCachedPropertyMethodInfo a signature where
    overloadedMethod = dBusProxySetCachedProperty

instance O.OverloadedMethodInfo DBusProxySetCachedPropertyMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusProxy.dBusProxySetCachedProperty",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusProxy.html#v:dBusProxySetCachedProperty"
        })


#endif

-- method DBusProxy::set_default_timeout
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "proxy"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusProxy" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusProxy." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "timeout_msec"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Timeout in milliseconds."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_dbus_proxy_set_default_timeout" g_dbus_proxy_set_default_timeout :: 
    Ptr DBusProxy ->                        -- proxy : TInterface (Name {namespace = "Gio", name = "DBusProxy"})
    Int32 ->                                -- timeout_msec : TBasicType TInt
    IO ()

-- | Sets the timeout to use if -1 (specifying default timeout) is
-- passed as /@timeoutMsec@/ in the 'GI.Gio.Objects.DBusProxy.dBusProxyCall' and
-- 'GI.Gio.Objects.DBusProxy.dBusProxyCallSync' functions.
-- 
-- See the [DBusProxy:gDefaultTimeout]("GI.Gio.Objects.DBusProxy#g:attr:gDefaultTimeout") property for more details.
-- 
-- /Since: 2.26/
dBusProxySetDefaultTimeout ::
    (B.CallStack.HasCallStack, MonadIO m, IsDBusProxy a) =>
    a
    -- ^ /@proxy@/: A t'GI.Gio.Objects.DBusProxy.DBusProxy'.
    -> Int32
    -- ^ /@timeoutMsec@/: Timeout in milliseconds.
    -> m ()
dBusProxySetDefaultTimeout proxy timeoutMsec = liftIO $ do
    proxy' <- unsafeManagedPtrCastPtr proxy
    g_dbus_proxy_set_default_timeout proxy' timeoutMsec
    touchManagedPtr proxy
    return ()

#if defined(ENABLE_OVERLOADING)
data DBusProxySetDefaultTimeoutMethodInfo
instance (signature ~ (Int32 -> m ()), MonadIO m, IsDBusProxy a) => O.OverloadedMethod DBusProxySetDefaultTimeoutMethodInfo a signature where
    overloadedMethod = dBusProxySetDefaultTimeout

instance O.OverloadedMethodInfo DBusProxySetDefaultTimeoutMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusProxy.dBusProxySetDefaultTimeout",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusProxy.html#v:dBusProxySetDefaultTimeout"
        })


#endif

-- method DBusProxy::set_interface_info
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "proxy"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusProxy" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusProxy" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "info"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusInterfaceInfo" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "Minimum interface this proxy conforms to\n   or %NULL to unset."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_dbus_proxy_set_interface_info" g_dbus_proxy_set_interface_info :: 
    Ptr DBusProxy ->                        -- proxy : TInterface (Name {namespace = "Gio", name = "DBusProxy"})
    Ptr Gio.DBusInterfaceInfo.DBusInterfaceInfo -> -- info : TInterface (Name {namespace = "Gio", name = "DBusInterfaceInfo"})
    IO ()

-- | Ensure that interactions with /@proxy@/ conform to the given
-- interface. See the [DBusProxy:gInterfaceInfo]("GI.Gio.Objects.DBusProxy#g:attr:gInterfaceInfo") property for more
-- details.
-- 
-- /Since: 2.26/
dBusProxySetInterfaceInfo ::
    (B.CallStack.HasCallStack, MonadIO m, IsDBusProxy a) =>
    a
    -- ^ /@proxy@/: A t'GI.Gio.Objects.DBusProxy.DBusProxy'
    -> Maybe (Gio.DBusInterfaceInfo.DBusInterfaceInfo)
    -- ^ /@info@/: Minimum interface this proxy conforms to
    --    or 'P.Nothing' to unset.
    -> m ()
dBusProxySetInterfaceInfo proxy info = liftIO $ do
    proxy' <- unsafeManagedPtrCastPtr proxy
    maybeInfo <- case info of
        Nothing -> return FP.nullPtr
        Just jInfo -> do
            jInfo' <- unsafeManagedPtrGetPtr jInfo
            return jInfo'
    g_dbus_proxy_set_interface_info proxy' maybeInfo
    touchManagedPtr proxy
    whenJust info touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data DBusProxySetInterfaceInfoMethodInfo
instance (signature ~ (Maybe (Gio.DBusInterfaceInfo.DBusInterfaceInfo) -> m ()), MonadIO m, IsDBusProxy a) => O.OverloadedMethod DBusProxySetInterfaceInfoMethodInfo a signature where
    overloadedMethod = dBusProxySetInterfaceInfo

instance O.OverloadedMethodInfo DBusProxySetInterfaceInfoMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusProxy.dBusProxySetInterfaceInfo",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusProxy.html#v:dBusProxySetInterfaceInfo"
        })


#endif

-- method DBusProxy::new
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "connection"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusConnection" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusConnection."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusProxyFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Flags used when constructing the proxy."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "info"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusInterfaceInfo" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "A #GDBusInterfaceInfo specifying the minimal interface that @proxy conforms to or %NULL."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "A bus name (well-known or unique) or %NULL if @connection is not a message bus connection."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "object_path"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "An object path." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "interface_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A D-Bus interface name."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GCancellable or %NULL."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "Callback function to invoke when the proxy is ready."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 8
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "User data to pass to @callback."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_dbus_proxy_new" g_dbus_proxy_new :: 
    Ptr Gio.DBusConnection.DBusConnection -> -- connection : TInterface (Name {namespace = "Gio", name = "DBusConnection"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "DBusProxyFlags"})
    Ptr Gio.DBusInterfaceInfo.DBusInterfaceInfo -> -- info : TInterface (Name {namespace = "Gio", name = "DBusInterfaceInfo"})
    CString ->                              -- name : TBasicType TUTF8
    CString ->                              -- object_path : TBasicType TUTF8
    CString ->                              -- interface_name : TBasicType TUTF8
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Creates a proxy for accessing /@interfaceName@/ on the remote object
-- at /@objectPath@/ owned by /@name@/ at /@connection@/ and asynchronously
-- loads D-Bus properties unless the
-- 'GI.Gio.Flags.DBusProxyFlagsDoNotLoadProperties' flag is used. Connect to
-- the [DBusProxy::gPropertiesChanged]("GI.Gio.Objects.DBusProxy#g:signal:gPropertiesChanged") signal to get notified about
-- property changes.
-- 
-- If the 'GI.Gio.Flags.DBusProxyFlagsDoNotConnectSignals' flag is not set, also sets up
-- match rules for signals. Connect to the [DBusProxy::gSignal]("GI.Gio.Objects.DBusProxy#g:signal:gSignal") signal
-- to handle signals from the remote object.
-- 
-- If both 'GI.Gio.Flags.DBusProxyFlagsDoNotLoadProperties' and
-- 'GI.Gio.Flags.DBusProxyFlagsDoNotConnectSignals' are set, this constructor is
-- guaranteed to complete immediately without blocking.
-- 
-- If /@name@/ is a well-known name and the
-- 'GI.Gio.Flags.DBusProxyFlagsDoNotAutoStart' and 'GI.Gio.Flags.DBusProxyFlagsDoNotAutoStartAtConstruction'
-- flags aren\'t set and no name owner currently exists, the message bus
-- will be requested to launch a name owner for the name.
-- 
-- This is a failable asynchronous constructor - when the proxy is
-- ready, /@callback@/ will be invoked and you can use
-- 'GI.Gio.Objects.DBusProxy.dBusProxyNewFinish' to get the result.
-- 
-- See 'GI.Gio.Objects.DBusProxy.dBusProxyNewSync' and for a synchronous version of this constructor.
-- 
-- t'GI.Gio.Objects.DBusProxy.DBusProxy' is used in this [example][class/@gio@/.DBusProxy@/a/@-watch-proxy-example].
-- 
-- /Since: 2.26/
dBusProxyNew ::
    (B.CallStack.HasCallStack, MonadIO m, Gio.DBusConnection.IsDBusConnection a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@connection@/: A t'GI.Gio.Objects.DBusConnection.DBusConnection'.
    -> [Gio.Flags.DBusProxyFlags]
    -- ^ /@flags@/: Flags used when constructing the proxy.
    -> Maybe (Gio.DBusInterfaceInfo.DBusInterfaceInfo)
    -- ^ /@info@/: A t'GI.Gio.Structs.DBusInterfaceInfo.DBusInterfaceInfo' specifying the minimal interface that /@proxy@/ conforms to or 'P.Nothing'.
    -> Maybe (T.Text)
    -- ^ /@name@/: A bus name (well-known or unique) or 'P.Nothing' if /@connection@/ is not a message bus connection.
    -> T.Text
    -- ^ /@objectPath@/: An object path.
    -> T.Text
    -- ^ /@interfaceName@/: A D-Bus interface name.
    -> Maybe (b)
    -- ^ /@cancellable@/: A t'GI.Gio.Objects.Cancellable.Cancellable' or 'P.Nothing'.
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: Callback function to invoke when the proxy is ready.
    -> m ()
dBusProxyNew connection flags info name objectPath interfaceName cancellable callback = liftIO $ do
    connection' <- unsafeManagedPtrCastPtr connection
    let flags' = gflagsToWord flags
    maybeInfo <- case info of
        Nothing -> return FP.nullPtr
        Just jInfo -> do
            jInfo' <- unsafeManagedPtrGetPtr jInfo
            return jInfo'
    maybeName <- case name of
        Nothing -> return FP.nullPtr
        Just jName -> do
            jName' <- textToCString jName
            return jName'
    objectPath' <- textToCString objectPath
    interfaceName' <- textToCString interfaceName
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_dbus_proxy_new connection' flags' maybeInfo maybeName objectPath' interfaceName' maybeCancellable maybeCallback userData
    touchManagedPtr connection
    whenJust info touchManagedPtr
    whenJust cancellable touchManagedPtr
    freeMem maybeName
    freeMem objectPath'
    freeMem interfaceName'
    return ()

#if defined(ENABLE_OVERLOADING)
#endif

-- method DBusProxy::new_for_bus
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "bus_type"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "BusType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GBusType." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusProxyFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Flags used when constructing the proxy."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "info"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "DBusInterfaceInfo" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "A #GDBusInterfaceInfo specifying the minimal interface that @proxy conforms to or %NULL."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A bus name (well-known or unique)."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "object_path"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "An object path." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "interface_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A D-Bus interface name."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GCancellable or %NULL."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "Callback function to invoke when the proxy is ready."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 8
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "User data to pass to @callback."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_dbus_proxy_new_for_bus" g_dbus_proxy_new_for_bus :: 
    CInt ->                                 -- bus_type : TInterface (Name {namespace = "Gio", name = "BusType"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "DBusProxyFlags"})
    Ptr Gio.DBusInterfaceInfo.DBusInterfaceInfo -> -- info : TInterface (Name {namespace = "Gio", name = "DBusInterfaceInfo"})
    CString ->                              -- name : TBasicType TUTF8
    CString ->                              -- object_path : TBasicType TUTF8
    CString ->                              -- interface_name : TBasicType TUTF8
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Like 'GI.Gio.Objects.DBusProxy.dBusProxyNew' but takes a t'GI.Gio.Enums.BusType' instead of a t'GI.Gio.Objects.DBusConnection.DBusConnection'.
-- 
-- t'GI.Gio.Objects.DBusProxy.DBusProxy' is used in this [example][class/@gio@/.DBusProxy@/a/@-watch-proxy-example].
-- 
-- /Since: 2.26/
dBusProxyNewForBus ::
    (B.CallStack.HasCallStack, MonadIO m, Gio.Cancellable.IsCancellable a) =>
    Gio.Enums.BusType
    -- ^ /@busType@/: A t'GI.Gio.Enums.BusType'.
    -> [Gio.Flags.DBusProxyFlags]
    -- ^ /@flags@/: Flags used when constructing the proxy.
    -> Maybe (Gio.DBusInterfaceInfo.DBusInterfaceInfo)
    -- ^ /@info@/: A t'GI.Gio.Structs.DBusInterfaceInfo.DBusInterfaceInfo' specifying the minimal interface that /@proxy@/ conforms to or 'P.Nothing'.
    -> T.Text
    -- ^ /@name@/: A bus name (well-known or unique).
    -> T.Text
    -- ^ /@objectPath@/: An object path.
    -> T.Text
    -- ^ /@interfaceName@/: A D-Bus interface name.
    -> Maybe (a)
    -- ^ /@cancellable@/: A t'GI.Gio.Objects.Cancellable.Cancellable' or 'P.Nothing'.
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: Callback function to invoke when the proxy is ready.
    -> m ()
dBusProxyNewForBus busType flags info name objectPath interfaceName cancellable callback = liftIO $ do
    let busType' = (fromIntegral . fromEnum) busType
    let flags' = gflagsToWord flags
    maybeInfo <- case info of
        Nothing -> return FP.nullPtr
        Just jInfo -> do
            jInfo' <- unsafeManagedPtrGetPtr jInfo
            return jInfo'
    name' <- textToCString name
    objectPath' <- textToCString objectPath
    interfaceName' <- textToCString interfaceName
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_dbus_proxy_new_for_bus busType' flags' maybeInfo name' objectPath' interfaceName' maybeCancellable maybeCallback userData
    whenJust info touchManagedPtr
    whenJust cancellable touchManagedPtr
    freeMem name'
    freeMem objectPath'
    freeMem interfaceName'
    return ()

#if defined(ENABLE_OVERLOADING)
#endif


