//
// Syd: rock-solid application kernel
// src/kernel/getdents.rs: getdents64(2) handler
//
// Copyright (c) 2023, 2024, 2025 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0

use std::os::fd::RawFd;

use libseccomp::ScmpNotifResp;
use nix::{errno::Errno, NixPath};

use crate::{
    compat::getdents64,
    config::{DIRENT_BUF_SIZE, MMAP_MIN_ADDR},
    fs::CanonicalPath,
    hook::UNotifyEventRequest,
    kernel::sandbox_path,
    sandbox::Capability,
};

#[allow(clippy::cognitive_complexity)]
pub(crate) fn sys_getdents64(request: UNotifyEventRequest) -> ScmpNotifResp {
    syscall_handler!(request, |request: UNotifyEventRequest| {
        let req = request.scmpreq;

        // Validate result buffer.
        if req.data.args[2] == 0 {
            return Err(Errno::EINVAL);
        }

        // Validate file descriptor.
        //
        // AT_FDCWD is an invalid file descriptor.
        let fd = RawFd::try_from(req.data.args[0]).or(Err(Errno::EBADF))?;
        if fd < 0 {
            return Err(Errno::EBADF);
        }

        // Validate dirp pointer.
        if req.data.args[1] < *MMAP_MIN_ADDR {
            // SAFETY: If the second argument which must hold a pointer to a
            // linux_dirent structure is not valid, we must return EFAULT
            // without further processing here.
            return Err(Errno::EFAULT);
        }

        // Get remote fd, and
        // Readlink /proc/thread-self/fd/$fd.
        //
        // Note, the Readdir access check here has been
        // moved to the _open_(2) handler for simplicity and
        // efficiency. The Stat check still takes place.
        let fd = request.get_fd(fd)?;
        let mut path = CanonicalPath::new_fd(fd.into(), req.pid())?;
        if !path.is_dir() {
            return Err(Errno::ENOTDIR);
        }
        #[allow(clippy::disallowed_methods)]
        let fd = path.dir.take().unwrap();
        let mut dir = path.take();

        // SAFETY: The count argument to the getdents call
        // must not be fully trusted, it can be overly large,
        // and allocating a Vector of that capacity may overflow.
        // This bug was discovered by trinity in this build:
        // https://builds.sr.ht/~alip/job/1077263
        let count = usize::try_from(req.data.args[2])
            .or(Err(Errno::EINVAL))?
            .min(DIRENT_BUF_SIZE);
        let pid = req.pid();
        let len = dir.len();
        let mut dot: u8 = 0;
        let mut ret: u64 = 0;
        while ret == 0 {
            let mut entries = match getdents64(&fd, count) {
                Ok(entries) => entries,
                Err(Errno::ECANCELED) => break, // EOF or empty directory
                Err(errno) => return Err(errno),
            };

            // Lock sandbox for read to perform Stat access check.
            let sandbox = request.get_sandbox();
            let safe_name = !sandbox.flags.allow_unsafe_filename();
            let restrict_mkbdev = !sandbox.flags.allow_unsafe_mkbdev();

            #[allow(clippy::arithmetic_side_effects)]
            for entry in &mut entries {
                if dot < 2 && entry.is_dot() {
                    // SAFETY: Allow the special dot entries `.` and `..`.
                    // Note, `..` may point to a denylisted directory,
                    // however at this point there's not much we can do:
                    // even the root directory, ie `/`, has a `..`. In
                    // this exceptional case `..` points to `.`.
                    dot += 1;
                } else {
                    // Append entry name to the directory.
                    dir.push(entry.name_bytes());

                    // SAFETY: Run XPath::check() with file type for global restrictions.
                    if dir
                        .check(
                            pid,
                            Some(&entry.file_type()),
                            Some(entry.as_xpath()),
                            safe_name,
                            restrict_mkbdev,
                        )
                        .is_err()
                    {
                        // skip entry.
                        dir.truncate(len);
                        continue;
                    }

                    // SAFETY: Run sandbox access check with stat capability.
                    let err = sandbox_path(
                        Some(&request),
                        &sandbox,
                        request.scmpreq.pid(), // Unused when request.is_some()
                        &dir,
                        Capability::CAP_STAT,
                        true,
                        "getdents64",
                    )
                    .is_err();
                    dir.truncate(len);
                    if err {
                        // skip entry.
                        continue;
                    }
                }

                // Access granted, write entry to sandbox process memory.
                match request.write_mem(entry.as_bytes(), req.data.args[1] + ret) {
                    Ok(n) => {
                        ret += n as u64;
                        if n != entry.size() {
                            break;
                        }
                    }
                    Err(_) if ret > 0 => break,
                    Err(errno) => return Err(errno),
                };
            }
        }

        #[allow(clippy::cast_possible_wrap)]
        Ok(request.return_syscall(ret as i64))
    })
}
