// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at https://mozilla.org/MPL/2.0/.

use inherent::inherent;

use super::{ChildMetricMeta, CommonMetricData};

use glean::traits::Numerator;
use glean::Rate;

use crate::ipc::{need_ipc, with_ipc_payload};
use crate::private::BaseMetricId;

/// Developer-facing API for recording rate metrics with external denominators.
///
/// Instances of this class type are automatically generated by the parsers
/// at build time, allowing developers to record values that were previously
/// registered in the metrics.yaml file.
#[derive(Clone)]
pub enum NumeratorMetric {
    Parent {
        /// The metric's ID. Used for testing and profiler markers. Numerator
        /// metrics canot be labeled, so we only store a BaseMetricId. If this
        /// changes, this should be changed to a MetricId to distinguish
        /// between metrics and sub-metrics.
        id: BaseMetricId,
        inner: glean::private::NumeratorMetric,
    },
    Child(ChildMetricMeta),
}

define_metric_metadata_getter!(NumeratorMetric, NUMERATOR_MAP);
define_metric_namer!(NumeratorMetric);

impl NumeratorMetric {
    /// The public constructor used by automatically generated metrics.
    pub fn new(id: BaseMetricId, meta: CommonMetricData) -> Self {
        if need_ipc() {
            NumeratorMetric::Child(ChildMetricMeta::from_common_metric_data(id, meta))
        } else {
            let inner = glean::private::NumeratorMetric::new(meta);
            NumeratorMetric::Parent { id, inner }
        }
    }

    #[cfg(test)]
    pub(crate) fn metric_id(&self) -> BaseMetricId {
        match self {
            NumeratorMetric::Parent { id, .. } => *id,
            NumeratorMetric::Child(meta) => meta.id,
        }
    }

    #[cfg(test)]
    pub(crate) fn child_metric(&self) -> Self {
        match self {
            NumeratorMetric::Parent { id, inner } => {
                NumeratorMetric::Child(ChildMetricMeta::from_metric_identifier(*id, inner))
            }
            NumeratorMetric::Child(_) => panic!("Can't get a child metric from a child metric"),
        }
    }
}

#[inherent]
impl Numerator for NumeratorMetric {
    pub fn add_to_numerator(&self, amount: i32) {
        #[allow(unused)]
        let id = match self {
            NumeratorMetric::Parent { id, inner } => {
                inner.add_to_numerator(amount);
                *id
            }
            NumeratorMetric::Child(meta) => {
                with_ipc_payload(move |payload| {
                    if let Some(v) = payload.numerators.get_mut(&meta.id) {
                        *v += amount;
                    } else {
                        payload.numerators.insert(meta.id, amount);
                    }
                });
                meta.id
            }
        };

        #[cfg(feature = "with_gecko")]
        if gecko_profiler::can_accept_markers() {
            gecko_profiler::add_marker(
                "Rate::addToNumerator",
                super::profiler_utils::TelemetryProfilerCategory,
                Default::default(),
                super::profiler_utils::IntLikeMetricMarker::<NumeratorMetric, i32>::new(
                    id.into(),
                    None,
                    amount,
                ),
            );
        }
    }

    pub fn test_get_num_recorded_errors(&self, error: glean::ErrorType) -> i32 {
        match self {
            NumeratorMetric::Parent { inner, .. } => inner.test_get_num_recorded_errors(error),
            NumeratorMetric::Child(meta) => {
                panic!(
                    "Cannot get the number of recorded errors for {:?} in non-parent process!",
                    meta.id
                );
            }
        }
    }
}

#[inherent]
impl glean::TestGetValue<Rate> for NumeratorMetric {
    pub fn test_get_value(&self, ping_name: Option<String>) -> Option<Rate> {
        match self {
            NumeratorMetric::Parent { inner, .. } => inner.test_get_value(ping_name),
            NumeratorMetric::Child(meta) => {
                panic!(
                    "Cannot get test value for {:?} in non-parent process!",
                    meta.id
                );
            }
        }
    }
}

#[cfg(test)]
mod test {
    use crate::{common_test::*, ipc, metrics};

    #[test]
    fn sets_numerator_value_parent() {
        let _lock = lock_test();

        let metric = &metrics::test_only_ipc::rate_with_external_denominator;
        metric.add_to_numerator(1);

        assert_eq!(1, metric.test_get_value(Some("test-ping".to_string())).unwrap().numerator);
    }

    #[test]
    fn sets_numerator_value_child() {
        let _lock = lock_test();

        let parent_metric = &metrics::test_only_ipc::rate_with_external_denominator;
        parent_metric.add_to_numerator(3);

        {
            // scope for need_ipc RAII
            let child_metric = parent_metric.child_metric();
            let _raii = ipc::test_set_need_ipc(true);
            let metric_id = child_metric.metric_id();

            child_metric.add_to_numerator(42);

            ipc::with_ipc_payload(move |payload| {
                assert!(
                    42 == *payload.numerators.get(&metric_id).unwrap(),
                    "Stored the correct value in the ipc payload"
                );
            });
        }

        assert!(
            false == ipc::need_ipc(),
            "RAII dropped, should not need ipc any more"
        );
        assert!(ipc::replay_from_buf(&ipc::take_buf().unwrap()).is_ok());

        assert!(
            45 == parent_metric.test_get_value(Some("test-ping".to_string())).unwrap().numerator,
            "Values from the 'processes' should be summed"
        );
    }
}
