﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/rum/CloudWatchRUMRequest.h>
#include <aws/rum/CloudWatchRUM_EXPORTS.h>
#include <aws/rum/model/MetricDefinitionRequest.h>
#include <aws/rum/model/MetricDestination.h>

#include <utility>

namespace Aws {
namespace CloudWatchRUM {
namespace Model {

/**
 */
class BatchCreateRumMetricDefinitionsRequest : public CloudWatchRUMRequest {
 public:
  AWS_CLOUDWATCHRUM_API BatchCreateRumMetricDefinitionsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "BatchCreateRumMetricDefinitions"; }

  AWS_CLOUDWATCHRUM_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The name of the CloudWatch RUM app monitor that is to send the metrics.</p>
   */
  inline const Aws::String& GetAppMonitorName() const { return m_appMonitorName; }
  inline bool AppMonitorNameHasBeenSet() const { return m_appMonitorNameHasBeenSet; }
  template <typename AppMonitorNameT = Aws::String>
  void SetAppMonitorName(AppMonitorNameT&& value) {
    m_appMonitorNameHasBeenSet = true;
    m_appMonitorName = std::forward<AppMonitorNameT>(value);
  }
  template <typename AppMonitorNameT = Aws::String>
  BatchCreateRumMetricDefinitionsRequest& WithAppMonitorName(AppMonitorNameT&& value) {
    SetAppMonitorName(std::forward<AppMonitorNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The destination to send the metrics to. Valid values are
   * <code>CloudWatch</code> and <code>Evidently</code>. If you specify
   * <code>Evidently</code>, you must also specify the Amazon Resource Name (ARN) of
   * the CloudWatchEvidently experiment that will receive the metrics and an IAM role
   * that has permission to write to the experiment.</p>
   */
  inline MetricDestination GetDestination() const { return m_destination; }
  inline bool DestinationHasBeenSet() const { return m_destinationHasBeenSet; }
  inline void SetDestination(MetricDestination value) {
    m_destinationHasBeenSet = true;
    m_destination = value;
  }
  inline BatchCreateRumMetricDefinitionsRequest& WithDestination(MetricDestination value) {
    SetDestination(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>This parameter is required if <code>Destination</code> is
   * <code>Evidently</code>. If <code>Destination</code> is <code>CloudWatch</code>,
   * do not use this parameter.</p> <p>This parameter specifies the ARN of the
   * Evidently experiment that is to receive the metrics. You must have already
   * defined this experiment as a valid destination. For more information, see <a
   * href="https://docs.aws.amazon.com/cloudwatchrum/latest/APIReference/API_PutRumMetricsDestination.html">PutRumMetricsDestination</a>.</p>
   */
  inline const Aws::String& GetDestinationArn() const { return m_destinationArn; }
  inline bool DestinationArnHasBeenSet() const { return m_destinationArnHasBeenSet; }
  template <typename DestinationArnT = Aws::String>
  void SetDestinationArn(DestinationArnT&& value) {
    m_destinationArnHasBeenSet = true;
    m_destinationArn = std::forward<DestinationArnT>(value);
  }
  template <typename DestinationArnT = Aws::String>
  BatchCreateRumMetricDefinitionsRequest& WithDestinationArn(DestinationArnT&& value) {
    SetDestinationArn(std::forward<DestinationArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An array of structures which define the metrics that you want to send.</p>
   */
  inline const Aws::Vector<MetricDefinitionRequest>& GetMetricDefinitions() const { return m_metricDefinitions; }
  inline bool MetricDefinitionsHasBeenSet() const { return m_metricDefinitionsHasBeenSet; }
  template <typename MetricDefinitionsT = Aws::Vector<MetricDefinitionRequest>>
  void SetMetricDefinitions(MetricDefinitionsT&& value) {
    m_metricDefinitionsHasBeenSet = true;
    m_metricDefinitions = std::forward<MetricDefinitionsT>(value);
  }
  template <typename MetricDefinitionsT = Aws::Vector<MetricDefinitionRequest>>
  BatchCreateRumMetricDefinitionsRequest& WithMetricDefinitions(MetricDefinitionsT&& value) {
    SetMetricDefinitions(std::forward<MetricDefinitionsT>(value));
    return *this;
  }
  template <typename MetricDefinitionsT = MetricDefinitionRequest>
  BatchCreateRumMetricDefinitionsRequest& AddMetricDefinitions(MetricDefinitionsT&& value) {
    m_metricDefinitionsHasBeenSet = true;
    m_metricDefinitions.emplace_back(std::forward<MetricDefinitionsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_appMonitorName;
  bool m_appMonitorNameHasBeenSet = false;

  MetricDestination m_destination{MetricDestination::NOT_SET};
  bool m_destinationHasBeenSet = false;

  Aws::String m_destinationArn;
  bool m_destinationArnHasBeenSet = false;

  Aws::Vector<MetricDefinitionRequest> m_metricDefinitions;
  bool m_metricDefinitionsHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudWatchRUM
}  // namespace Aws
