﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/codepipeline/CodePipelineRequest.h>
#include <aws/codepipeline/CodePipeline_EXPORTS.h>
#include <aws/codepipeline/model/ConditionType.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace CodePipeline {
namespace Model {

/**
 */
class OverrideStageConditionRequest : public CodePipelineRequest {
 public:
  AWS_CODEPIPELINE_API OverrideStageConditionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "OverrideStageCondition"; }

  AWS_CODEPIPELINE_API Aws::String SerializePayload() const override;

  AWS_CODEPIPELINE_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the pipeline with the stage that will override the condition.</p>
   */
  inline const Aws::String& GetPipelineName() const { return m_pipelineName; }
  inline bool PipelineNameHasBeenSet() const { return m_pipelineNameHasBeenSet; }
  template <typename PipelineNameT = Aws::String>
  void SetPipelineName(PipelineNameT&& value) {
    m_pipelineNameHasBeenSet = true;
    m_pipelineName = std::forward<PipelineNameT>(value);
  }
  template <typename PipelineNameT = Aws::String>
  OverrideStageConditionRequest& WithPipelineName(PipelineNameT&& value) {
    SetPipelineName(std::forward<PipelineNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the stage for the override.</p>
   */
  inline const Aws::String& GetStageName() const { return m_stageName; }
  inline bool StageNameHasBeenSet() const { return m_stageNameHasBeenSet; }
  template <typename StageNameT = Aws::String>
  void SetStageName(StageNameT&& value) {
    m_stageNameHasBeenSet = true;
    m_stageName = std::forward<StageNameT>(value);
  }
  template <typename StageNameT = Aws::String>
  OverrideStageConditionRequest& WithStageName(StageNameT&& value) {
    SetStageName(std::forward<StageNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the pipeline execution for the override.</p>
   */
  inline const Aws::String& GetPipelineExecutionId() const { return m_pipelineExecutionId; }
  inline bool PipelineExecutionIdHasBeenSet() const { return m_pipelineExecutionIdHasBeenSet; }
  template <typename PipelineExecutionIdT = Aws::String>
  void SetPipelineExecutionId(PipelineExecutionIdT&& value) {
    m_pipelineExecutionIdHasBeenSet = true;
    m_pipelineExecutionId = std::forward<PipelineExecutionIdT>(value);
  }
  template <typename PipelineExecutionIdT = Aws::String>
  OverrideStageConditionRequest& WithPipelineExecutionId(PipelineExecutionIdT&& value) {
    SetPipelineExecutionId(std::forward<PipelineExecutionIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of condition to override for the stage, such as entry conditions,
   * failure conditions, or success conditions.</p>
   */
  inline ConditionType GetConditionType() const { return m_conditionType; }
  inline bool ConditionTypeHasBeenSet() const { return m_conditionTypeHasBeenSet; }
  inline void SetConditionType(ConditionType value) {
    m_conditionTypeHasBeenSet = true;
    m_conditionType = value;
  }
  inline OverrideStageConditionRequest& WithConditionType(ConditionType value) {
    SetConditionType(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_pipelineName;
  bool m_pipelineNameHasBeenSet = false;

  Aws::String m_stageName;
  bool m_stageNameHasBeenSet = false;

  Aws::String m_pipelineExecutionId;
  bool m_pipelineExecutionIdHasBeenSet = false;

  ConditionType m_conditionType{ConditionType::NOT_SET};
  bool m_conditionTypeHasBeenSet = false;
};

}  // namespace Model
}  // namespace CodePipeline
}  // namespace Aws
