﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/sagemaker/SageMakerRequest.h>
#include <aws/sagemaker/SageMaker_EXPORTS.h>
#include <aws/sagemaker/model/EdgeOutputConfig.h>

#include <utility>

namespace Aws {
namespace SageMaker {
namespace Model {

/**
 */
class UpdateDeviceFleetRequest : public SageMakerRequest {
 public:
  AWS_SAGEMAKER_API UpdateDeviceFleetRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateDeviceFleet"; }

  AWS_SAGEMAKER_API Aws::String SerializePayload() const override;

  AWS_SAGEMAKER_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the fleet.</p>
   */
  inline const Aws::String& GetDeviceFleetName() const { return m_deviceFleetName; }
  inline bool DeviceFleetNameHasBeenSet() const { return m_deviceFleetNameHasBeenSet; }
  template <typename DeviceFleetNameT = Aws::String>
  void SetDeviceFleetName(DeviceFleetNameT&& value) {
    m_deviceFleetNameHasBeenSet = true;
    m_deviceFleetName = std::forward<DeviceFleetNameT>(value);
  }
  template <typename DeviceFleetNameT = Aws::String>
  UpdateDeviceFleetRequest& WithDeviceFleetName(DeviceFleetNameT&& value) {
    SetDeviceFleetName(std::forward<DeviceFleetNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the device.</p>
   */
  inline const Aws::String& GetRoleArn() const { return m_roleArn; }
  inline bool RoleArnHasBeenSet() const { return m_roleArnHasBeenSet; }
  template <typename RoleArnT = Aws::String>
  void SetRoleArn(RoleArnT&& value) {
    m_roleArnHasBeenSet = true;
    m_roleArn = std::forward<RoleArnT>(value);
  }
  template <typename RoleArnT = Aws::String>
  UpdateDeviceFleetRequest& WithRoleArn(RoleArnT&& value) {
    SetRoleArn(std::forward<RoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Description of the fleet.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  UpdateDeviceFleetRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Output configuration for storing sample data collected by the fleet.</p>
   */
  inline const EdgeOutputConfig& GetOutputConfig() const { return m_outputConfig; }
  inline bool OutputConfigHasBeenSet() const { return m_outputConfigHasBeenSet; }
  template <typename OutputConfigT = EdgeOutputConfig>
  void SetOutputConfig(OutputConfigT&& value) {
    m_outputConfigHasBeenSet = true;
    m_outputConfig = std::forward<OutputConfigT>(value);
  }
  template <typename OutputConfigT = EdgeOutputConfig>
  UpdateDeviceFleetRequest& WithOutputConfig(OutputConfigT&& value) {
    SetOutputConfig(std::forward<OutputConfigT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Whether to create an Amazon Web Services IoT Role Alias during device fleet
   * creation. The name of the role alias generated will match this pattern:
   * "SageMakerEdge-{DeviceFleetName}".</p> <p>For example, if your device fleet is
   * called "demo-fleet", the name of the role alias will be
   * "SageMakerEdge-demo-fleet".</p>
   */
  inline bool GetEnableIotRoleAlias() const { return m_enableIotRoleAlias; }
  inline bool EnableIotRoleAliasHasBeenSet() const { return m_enableIotRoleAliasHasBeenSet; }
  inline void SetEnableIotRoleAlias(bool value) {
    m_enableIotRoleAliasHasBeenSet = true;
    m_enableIotRoleAlias = value;
  }
  inline UpdateDeviceFleetRequest& WithEnableIotRoleAlias(bool value) {
    SetEnableIotRoleAlias(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_deviceFleetName;

  Aws::String m_roleArn;

  Aws::String m_description;

  EdgeOutputConfig m_outputConfig;

  bool m_enableIotRoleAlias{false};
  bool m_deviceFleetNameHasBeenSet = false;
  bool m_roleArnHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_outputConfigHasBeenSet = false;
  bool m_enableIotRoleAliasHasBeenSet = false;
};

}  // namespace Model
}  // namespace SageMaker
}  // namespace Aws
