/**
 * Pattern that matches a `# yaml-language-server-disable` comment.
 *
 * Usage in YAML files:
 *
 *   - `# yaml-language-server-disable` - suppress ALL diagnostics on the next line
 *   - `# yaml-language-server-disable Incorrect type` - suppress diagnostics whose message contains "Incorrect type"
 *   - `# yaml-language-server-disable Incorrect type, not accepted` - suppress diagnostics matching any of the substrings
 *
 * Capture group 1 (optional) contains the comma-separated list of message
 * substrings to match against. If absent, all diagnostics are suppressed.
 */
export declare const YAML_DISABLE_PATTERN: RegExp;
/**
 * A callback that returns the text content of a given zero-based line number,
 * or `undefined` if the line does not exist.
 */
export type GetLineText = (line: number) => string | undefined;
/**
 * Parse the text after `yaml-language-server-disable` into an array of trimmed,
 * lower-cased message substrings.  Returns an empty array when no
 * specifiers are provided (meaning "suppress all").
 */
export declare function parseDisableSpecifiers(raw: string): string[];
/**
 * Determine whether a diagnostic should be suppressed based on the
 * specifiers from a `# yaml-language-server-disable` comment.
 *
 * @param specifiers - Parsed specifiers (empty means suppress all).
 * @param diagnosticMessage - The diagnostic's message text.
 * @returns `true` if the diagnostic should be suppressed.
 */
export declare function shouldSuppressDiagnostic(specifiers: string[], diagnosticMessage: string): boolean;
/**
 * Filters an array of diagnostics, removing any whose starting line is
 * immediately preceded by a `# yaml-language-server-disable` comment.
 *
 * When the comment includes one or more comma-separated message substrings,
 * only diagnostics whose message contains at least one of those substrings
 * (case-insensitive) are suppressed.  Without specifiers, all diagnostics
 * on the next line are suppressed.
 *
 * @param diagnostics - The diagnostics to filter.
 * @param getStartLine - Extracts the zero-based starting line number from a diagnostic.
 * @param getMessage - Extracts the message string from a diagnostic.
 * @param getLineText - Returns the text of a document line by its zero-based index,
 *   or `undefined` if the line is out of range.
 * @returns A new array containing only the diagnostics that are not suppressed.
 */
export declare function filterSuppressedDiagnostics<T>(diagnostics: T[], getStartLine: (diag: T) => number, getMessage: (diag: T) => string, getLineText: GetLineText): T[];
